package main

import (
	"encoding/json"
	"errors"
	"flag"
	"fmt"
	"io/ioutil"
	"log/syslog"
	"net/http"
	"strings"
	"sync"
	"time"

	"github.com/go-zookeeper/zk"

	s "a.yandex-team.ru/direct/infra/dt-db-manager/pkg/support"
	"a.yandex-team.ru/direct/infra/dt-db-manager/pkg/zookeeper"
)

var logger *syslog.Writer
var err error
var f = fmt.Sprintf
var wrap = s.Wrap
var gmlock = sync.Mutex{}

func main() {
	//читаем базовый конфиг
	configPath := flag.String("config", "", "read config file")
	flag.Parse()

	configValues := s.ParseConfig(*configPath)
	logger = configValues.Logger

	fmt.Printf("%+v", configValues)

	hosts := s.NewHosts()
	groupsManager := s.NewGroupsManager()

	var zkconn zookeeper.ZkConnect
	go func() {
		var err error
		for zkconn, err = zookeeper.Connect(configValues.ZookeeperServers, logger); err != nil; {
			time.Sleep(5 * time.Second)
		}
	}()

	nodes := zookeeper.Nodes{}

	go func() {
		ticker := time.NewTicker(5 * time.Second)
		defer ticker.Stop()
		for range ticker.C {
		E:
			select {
			case event := <-zkconn.Event:
				switch event.State {
				case zk.StateExpired, zk.StateConnecting:
					wrap(logger.Info(f("zookeeper session expired/connectig: %s. Wait", event.State)))
					break E
				case zk.StateDisconnected:
					wrap(logger.Warning(f("zookeeper disconnected: %s. Reconnect", event.State)))
				default:
					wrap(logger.Warning(f("zookeeper event state %s", event.State)))
				}
			default:
			}

			for _, node := range nodes {
				if node.Deleted == 1 {
					wrap(logger.Info(f("start delete node %+v\n", node)))
					if err := zkconn.DeleteNode(*node); err != nil {
						wrap(logger.Crit(f("error delete node %s: %s", node, err)))
					}
					continue
				}
				if err := zkconn.SaveNode(*node); err != nil {
					wrap(logger.Crit(f("error save node %v: %s\n", node, err)))
				}
			}

			zkNodes := zookeeper.Nodes{}
			//groupName это db_dev7
			for groupName, nodeName := range configValues.ZkPathList(&zkconn, "db_") {
				for _, name := range nodeName {
					valnode, err := zkconn.LoadNode(groupName, name)
					if err != nil {
						wrap(logger.Crit(f("error load node %s: %s\n", name, err)))
						continue
					}
					zkNodes = append(zkNodes, &valnode)
				}
			}

			/*обновляем только при условии не пустого nodes от zookeeper
			if len(loadedNodes) > 0 {
				nodes = loadedNodes
			}*/

			//обновляем nodes при условии, что они более старые чем в zookeeper или отсутствуют вовсе
			var loadedNodes zookeeper.Nodes
			for _, newnode := range zkNodes {
				zkpath := newnode.Path
				if !nodes.ExistNode(*newnode) {
					nodes.AddNode(*newnode)
					loadedNodes = append(loadedNodes, newnode)
				} else if nodes.GetNodeVersion(zkpath) < (*newnode).GetNodeVersion() {
					nodes.DeleteNode(*newnode)
					nodes.AddNode(*newnode)
					loadedNodes = append(loadedNodes, newnode)
				}
			}

			//проходимся по nodes и обновляем при необходимости hosts(т.ж. учасвует в первой загрузке hosts)
			for _, node := range loadedNodes {
				if strings.Contains(node.Path, "servers") {
					var servers []string
					if len(node.Data) == 0 {
						wrap(logger.Crit(f("no servers into node %s", node.Path)))
						continue
					}
					if err := json.Unmarshal(node.Data, &servers); err != nil {
						wrap(logger.Crit(f("error unmarshal node %s: %s", node.Data, err)))
						continue
					}
					if len(servers) == 0 {
						node.Deleted = 1
						continue
					}
					var newHosts s.Hosts
					for _, server := range servers {
						newHost := s.NewHost(server, node.GetGroup())
						newHosts = append(newHosts, newHost)
					}
					wrap(logger.Info(f("add new hosts %+v", newHosts)))
					hosts.UpdateHosts(newHosts)
					wrap(logger.Debug(f("hosts: %+v zknode %s", hosts, node.Path)))
				} else if strings.Contains(node.Path, "status") {
					zkGroupsManager := s.NewGroupManager("default", "finish",
						"finish", hosts, configValues)
					if len(node.Data) == 0 {
						wrap(logger.Crit(f("no stages into node %s", node.Path)))
						continue
					}
					if err := json.Unmarshal(node.Data, &zkGroupsManager); err != nil {
						wrap(logger.Crit(f("error unmarshal node %s: %s", node.Data, err)))
						continue
					}
					//*zkGroupsManager.YtReplicator = configValues.LoadReplicatorConf(zkGroupsManager.GroupName)
					groupsManager.DeleteMangerGroups(zkGroupsManager)
					groupsManager.AddMangerGroups(zkGroupsManager)
				}
			}
			wrap(logger.Debug(f("[Save/LoadZkNodes] all hosts: %+v", hosts)))
		}
	}()

	go func() {
		ticker := time.NewTicker(5 * time.Second)
		defer ticker.Stop()
		for range ticker.C {
			for _, group := range hosts.AllGroups() {
				if !groupsManager.HasGroup(string(group)) {
					//replicator := configValues.LoadReplicatorConf(group.GetGroupName())
					//teleportHosts := configValues.TeleportHosts
					newGroup := s.NewGroupManager(group.ToString(), "default", "finish",
						hosts, configValues)
					if gm := nodes.FindGroup(string(group), "status"); gm != nil {
						_ = json.Unmarshal(gm.Data, &newGroup)
					}
					groupsManager.AddManagerGroup(&newGroup)
					continue
				}
				for _, host := range *hosts {
					for _, savedGroup := range *groupsManager.GetGroupManager(group) {
						if savedGroup.HasHost(host) {
							continue
						}
						savedGroup.Hosts = append(savedGroup.Hosts, host)
					}
				}
			}
			gmlock.Lock()
			for _, groupManager := range *groupsManager {
				zkpath := configValues.ZkPathStatusGenerate(groupManager.MyGroup.ToString())
				node := zookeeper.NewZkNode(string(groupManager.MyGroup), zkpath)
				if data, err := json.Marshal(groupManager); err != nil {
					wrap(logger.Crit(f("error marshal status %s: %s", groupManager, err)))
					continue
				} else {
					node.Data = data
				}
				nodes.UpdateNode(*node)
			}
			(*groupsManager).RemoveEmptyGroups()
			gmlock.Unlock()
		}
	}()

	go hosts.StartMonitoringResources()

	complects := s.NewComplects()            //нужно только для http ручки
	go complects.StartMonitoringMdsBackups() //нужно только для http ручки

	ticker := time.NewTicker(5 * time.Second)
	defer ticker.Stop()

	go groupsManager.StartMonitorStages()
	go groupsManager.StartMonitoringGroupTasks()

	go func() {
		for range ticker.C {
			for _, host := range *hosts {
				wrap(logger.Debug(f("%s\n%+v\t%s\terr:%s\n",
					host.GetHost(), host.GetResources(), host.GetGroups(), host.GetError())))
				host.GetResources()
				for _, task := range host.TaskList {
					wrap(logger.Info(f("HOST %s TASK %+v", host.GetHost(), task)))
				}
			}

			for _, group := range *groupsManager {
				wrap(logger.Info(f("GROUP MANAGER %v hosts %+v status %s tests: %+v\n",
					group.MyGroup, group.Hosts, *group.GroupStatus, group.GetResources())))
			}
			/*for _, node := range nodes {
				wrap(logger.Debug(f("[Zookeeper] NODES %+v\n", node)))
			}*/
		}
	}()

	http.HandleFunc("/add-host", func(w http.ResponseWriter, r *http.Request) {
		var line []byte
		var newHosts s.Hosts
		if line, err = ioutil.ReadAll(r.Body); err != nil {
			wrap(logger.Warning(f("[http/add-host] error request body %s", err)))
			return
		}
		if len(line) == 0 {
			wrap(logger.Warning(f("[http/add-host] empty request body")))
			return
		}
		if err = json.Unmarshal(line, &newHosts); err != nil {
			wrap(logger.Warning(f("[http/add-host] error parse request json %s: %s", line, err)))
			return
		}
		hosts.UpdateHosts(newHosts) //??
		UpdateZkNodes(&nodes, hosts, configValues)

		_, _ = fmt.Fprintf(w, "%s", s.GetStatusDone())
	})

	http.HandleFunc("/remove-host", func(w http.ResponseWriter, r *http.Request) {
		var line []byte
		var removeHosts s.Hosts
		if line, err = ioutil.ReadAll(r.Body); err != nil {
			wrap(logger.Warning(f("error request body %s", err)))
			return
		}
		if len(line) == 0 {
			wrap(logger.Warning(f("[http/addHost] empty request body")))
			return
		}
		if err = json.Unmarshal(line, &removeHosts); err != nil {
			wrap(logger.Warning(f("[http/addHost] error parse request json %s: %s", line, err)))
			return
		}

		hosts.RemoveHost(removeHosts)
		UpdateZkNodes(&nodes, hosts, configValues)
		_, _ = fmt.Fprintf(w, "%s", s.GetStatusDone())
	})

	http.HandleFunc("/list-complects", func(w http.ResponseWriter, r *http.Request) {
		var output []byte
		complects := complects.FullComplects()
		if output, err = json.Marshal(complects); err != nil {
			wrap(logger.Crit(f("[http/list-complects] error marshal %+v: %s", complects, err)))
			return
		}
		_, _ = fmt.Fprintf(w, "%s", output)
	})

	http.HandleFunc("/plan-restore", func(w http.ResponseWriter, r *http.Request) {
		var plansJSON []byte
		var verdict *s.Verdict
		var planRestore s.PlanRestore
		planRequest, err := readRequest(r)
		if err != nil {
			return
		}
		_ = planRequest.Normalize()
		replicators := configValues.LoadReplicatorConf(planRequest.MyGroup.GetGroupName())
		planning := s.NewPlan(hosts, complects, &replicators) //нужно для http ручки
		wrap(logger.Info(f("restore plan %s", planRequest)))
		if len(planRequest.MyGroup) == 0 {
			err = errors.New("no group for planning")
			planRestore.Error = err
			wrap(logger.Warning(err.Error()))
		} else {
			verdict = planning.NewPlanRestore(planRequest.MyGroup, planRequest.GetNameComplect(),
				planRequest.GetInstanceNames(), planRequest.EnableReplicator)
		}

		if plansJSON, err = json.Marshal(verdict); err != nil {
			wrap(logger.Warning(f("[http/plan-restore] error marshal %+v: %s", verdict, err)))
			return
		}
		wrap(logger.Info(f("составлен план %s", plansJSON)))
		_, _ = fmt.Fprintf(w, "%s", plansJSON)
	})

	http.HandleFunc("/gtid-status", func(w http.ResponseWriter, r *http.Request) {
		planRequest, err := readRequest(r)
		if err != nil {
			return
		}
		group := planRequest.MyGroup
		hs := hosts.FindGroupHosts(group)
		gtids := hs.GtidPerInstance()
		gtidJSON, err := json.Marshal(gtids)
		if err != nil {
			wrap(logger.Warning(f("[http/gtid-status] error marshal %+v: %s", gtids, err)))
		}
		_, _ = fmt.Fprintf(w, "%s", gtidJSON)
	})

	http.HandleFunc("/list-servers", func(w http.ResponseWriter, r *http.Request) {
		var hostsJSON []byte
		if hostsJSON, err = json.Marshal(hosts); err != nil {
			wrap(logger.Crit(f("[http/list-servers] error marshal %+v: %s", hostsJSON, err)))
			return
		}
		_, _ = fmt.Fprintf(w, "%s", hostsJSON)
	})

	http.HandleFunc("/running-servers", func(w http.ResponseWriter, r *http.Request) {
		var hostsJSON []byte
		planRequest, err := readRequest(r)
		if err != nil {
			return
		}
		group := planRequest.MyGroup
		myhosts := *hosts.HostsPerGroup(group)

		if hostsJSON, err = json.Marshal(myhosts); err != nil {
			wrap(logger.Crit(f("[http/list-servers] error marshal %+v: %s", hostsJSON, err)))
			return
		}
		_, _ = fmt.Fprintf(w, "%s", hostsJSON)
	})

	http.HandleFunc("/restore-backup", func(w http.ResponseWriter, r *http.Request) {
		var values []byte
		var planRequest s.PlanRequest
		var groups s.Groups
		reader := r.Body
		if values, err = ioutil.ReadAll(reader); err != nil {
			wrap(logger.Warning(f("[http/restore-backup] error read request %s: %s", values, err)))
			return
		}
		if err = json.Unmarshal(values, &planRequest); err != nil {
			wrap(logger.Warning(f("[http/restore-backup] error marshal %+v: %s", values, err)))
		}
		gmlock.Lock()
		defer gmlock.Unlock()
		if err := planRequest.Normalize(); err != nil {
			wrap(logger.Warning(f("[http/restore-backup] error normalize %s", err)))
			_, _ = fmt.Fprintf(w, "%s", s.GetStatusFailed())
			return
		}
		group := planRequest.MyGroup
		//if len(group.GetReplica()) > 0 {
		//	groups = s.Groups{planRequest.GroupName}
		//} else {
		groups = hosts.ReplicsPerGroupName(group)
		//}
		//check name stages planRequest.FinishStage
		for _, mygroup := range groups {
			for _, gm := range *groupsManager.GetGroupManager(mygroup) {
				gm.SetFinishStage(planRequest.FinishStage)
				gm.SetComplectName(planRequest.NameComplect)
				gm.SetAllowInstances(planRequest.GetInstanceNames())
				gm.SetGroupStatus("start")
				gm.SetReplicator(planRequest.EnableReplicator)
				gm.SetBinlogWriter(planRequest.EnableBinlogWriter)
				wrap(logger.Info(f("start restore last groups: %+v to hosts %+v",
					groupsManager, gm.Servers.FindGroupHosts(gm.MyGroup))))
			}
		}
		_, _ = fmt.Fprintf(w, "%s", s.GetStatusDone())
	})

	http.HandleFunc("/task-status", func(w http.ResponseWriter, r *http.Request) {
		var gmJSON []byte
		if gmJSON, err = json.Marshal(groupsManager); err != nil {
			wrap(logger.Info(f("[http/group-status] error marshal %+v: %s", gmJSON, err)))
		}
		_, _ = fmt.Fprintf(w, "%s", gmJSON)
	})

	http.HandleFunc("/set-stage", func(w http.ResponseWriter, r *http.Request) {
		var values []byte
		var stage s.Stage
		var groups s.Groups
		reader := r.Body
		if values, err = ioutil.ReadAll(reader); err != nil {
			wrap(logger.Warning(f("[set-stage] error read request %s: %s", values, err)))
			return
		}
		if err = json.Unmarshal(values, &stage); err != nil {
			wrap(logger.Warning(f("[http/set-stage] error unmarshal %+v: %s", values, err)))
		}
		gmlock.Lock()
		defer gmlock.Unlock()

		//if len(stage.GroupName.GetReplica()) > 0 {
		//	groups = s.Groups{stage.GroupName}
		//} else {
		groups = hosts.ReplicsPerGroupName(stage.GroupName)
		//}
		for _, mygroup := range groups {
			for _, gm := range *groupsManager.GetGroupManager(mygroup) {
				//Удаляем сначала старые таски по stage, иначе мониторящий поток выставит снова failed
				for _, h := range *hosts.FindGroupHosts(gm.MyGroup) {
					gm.RemoveTasks(h.Hostname, stage)
				}
				gm.SetGroupStatus(stage.StageName)
				//gm.SetFinishStage(stage.GetLastStage())
			}
		}
		wrap(logger.Info(f("[server/set-stage] set group %s stage to %s", stage.GroupName, stage.StageName)))
		_, _ = fmt.Fprintf(w, "%s", s.GetStatusDone())
	})

	http.HandleFunc("/restart-failed-tasks", func(w http.ResponseWriter, r *http.Request) {
		groupsManager.RestartFailedTasks()
		wrap(logger.Info(f("restart failed tasks")))
		_, _ = fmt.Fprintf(w, "%s", s.GetStatusDone())
	})

	http.HandleFunc("/replicas-plan", func(w http.ResponseWriter, r *http.Request) {
		var values []byte
		var group s.Group
		reader := r.Body
		if values, err = ioutil.ReadAll(reader); err != nil {
			wrap(logger.Warning(f("[replicas] error read request %s: %s", values, err)))
			return
		}
		if err = json.Unmarshal(values, &group); err != nil {
			wrap(logger.Warning(f("[http/replicas] error unmarshal %+v: %s", values, err)))
		}
		myplan := groupsManager.PlansForReplicas(group)
		out, err := json.Marshal(myplan)
		if err != nil {
			wrap(logger.Info(f("[http/replicas] error marshal %+v: %s", myplan, err)))
		}
		_, _ = fmt.Fprintf(w, "%s", out)
	})

	http.HandleFunc("/unpaused", func(w http.ResponseWriter, r *http.Request) {
		var values []byte
		var msg string
		var planRequest s.PlanRequest
		var groups s.Groups
		reader := r.Body
		if values, err = ioutil.ReadAll(reader); err != nil {
			wrap(logger.Warning(f("[unpaused-restore] error read request %s: %s", values, err)))
			return
		}
		if err = json.Unmarshal(values, &planRequest); err != nil {
			wrap(logger.Warning(f("[http/unpaused] error marshal %+v: %s", values, err)))
		}
		if len(planRequest.MyGroup) == 0 {
			//msg = f("не указаны группа для unpause: -group <group>")
			_, _ = fmt.Fprintf(w, "%s", s.GetStatusFailed())
			return
		}

		gmlock.Lock()
		defer gmlock.Unlock()

		if err := planRequest.Normalize(); err != nil {
			wrap(logger.Warning(f("[http/unpaused] error normalize %s", err)))
			_, _ = fmt.Fprintf(w, "%s", s.GetStatusFailed())
			return
		}
		group := planRequest.MyGroup
		//if len(group.GetReplica()) == 0 {
		//	groups = s.Groups{group}
		//} else {
		groups = hosts.ReplicsPerGroupName(group)
		//}
		fmt.Println("REPLICS", groups)
		for _, mygroup := range groups {
			for _, gm := range *groupsManager.GetGroupManager(mygroup) {
				gm.SetFinishStage(planRequest.FinishStage)
				gm.SetGroupStatus(gm.GetLastStatus())
			}
		}
		msg = f("unpaused для %s выполнен", planRequest.MyGroup)
		_, _ = fmt.Fprintf(w, "%s", s.GetStatusDone())
		wrap(logger.Info(msg))
	})

	wrap(http.ListenAndServe(f("%s:%d", configValues.ListenHost, configValues.ListenPort), nil))
}

func UpdateZkNodes(nodes *zookeeper.Nodes, hosts *s.Hosts, config s.Config) {
	//fmt.Println("ALLGROUPS", hosts.AllGroups())
	for _, group := range hosts.AllGroups() {
		groupName := string(group)
		if ok := nodes.FindGroup(groupName, "servers"); ok == nil {
			newNode := zookeeper.NewZkNode(string(group), config.ZkPathGroupGenerate(groupName))
			*nodes = append(*nodes, newNode)
		}
	}

	for _, node := range *nodes {
		var result []string
		var deleted int
		findedHosts := hosts.FindGroupHosts(node.GetGroup())
		for _, host := range *findedHosts {
			result = append(result, host.Hostname)
		}
		if len(result) == 0 {
			deleted = 1
		}
		if data, err := json.Marshal(result); err != nil {
			wrap(logger.Crit(f("[server/UpdateZkNode] error marshal %v: %s", result, err)))
		} else {
			node.Data = data
			node.Stat.Version += 1
			node.Deleted = deleted
		}
		wrap(logger.Info(f("[server/UpdateZkNode] node: %+v\n", node)))
	}
}

func readRequest(request *http.Request) (planRequest s.PlanRequest, err error) {
	reader := request.Body
	values, err := ioutil.ReadAll(reader)
	if err != nil {
		wrap(logger.Warning(f("[plan-restore] error read request %s: %s", values, err)))
		return
	}
	err = json.Unmarshal(values, &planRequest)
	if err != nil {
		wrap(logger.Warning(f("[plan-restore] error unmarshal request %s: %s", values, err)))
	}
	return
}
