package main

import (
	"encoding/json"
	"flag"
	"fmt"
	"io/ioutil"
	"log"
	"os"
	"strings"

	s "a.yandex-team.ru/direct/infra/dt-db-manager/pkg/support"
	tt "a.yandex-team.ru/direct/infra/dt-db-manager/pkg/yttransfer"
)

var (
	commands []string
)

/*
Утилита для разворачивания бекапа YT mysql репликатора на ТС. Состоит из нескольких этапов:
1. Приезжают статические данные из бекапов посредством transfer manager
2. Обновляются позиции gtidset в mysql-stat-sync
3. Таблицы конвертируются в динамические, происходит монтирование и удаление статических
4. Создается симлинка current указывающая на новый бекап
*/

func main() {
	sourceDir := flag.String("directory-backup", "//home/direct/mysql-sync-backup", "путь до бекапов")
	destinationDir := flag.String("directory-restore", "//home/direct/test/mysql-sync/dev7", "путь куда привезти бекап")
	backupCluster := flag.String("cluster-backup", "hahn", "кластер YT с бекапами")
	restoreCluster := flag.String("cluster-restore", "zeno", "кластер YT для восстановления бекапов")
	account := flag.String("account", "direct", "аккаунт в YT")
	gtidfile := flag.String("gtid-file", "", "файл с json для восстановления gtid")
	yttoken := flag.String("yt-token", "/tmp/pass1", "токен для подключения к YT")
	command := flag.String("command", "list", "команда для выполнения")
	flag.Parse()

	ytreplicator := s.YtReplicator{
		SourceCluster:      *backupCluster,
		SourceDir:          *sourceDir,
		DestinationCluster: *restoreCluster,
		DestinationDir:     *destinationDir,
		Account:            *account,
		YTTokenFile:        *yttoken,
	}

	gtids := make(s.GtidMysqlBinlog)
	if _, err := os.Stat(*gtidfile); len(*gtidfile) > 0 && err == nil {
		data, err := ioutil.ReadFile(*gtidfile)
		if err != nil {
			log.Fatal(err)
		}
		if err := json.Unmarshal(data, &gtids); err != nil {
			log.Fatal(err)
		}
	}

	//create connection
	ytconnector, err := s.NewGroupYtReplicator(ytreplicator)
	if err != nil {
		log.Fatal(err)
	}

	//create transfer tables
	var transferTables tt.YtTransferTables

	if *command == "restore-full" {
		commands = append(commands, "download-tables", "update-gtid", "convert-tables", "update-link", "remove-old-tables")
	} else if *command == "restore-tables" {
		commands = append(commands, "download-tables", "update-gtid", "convert-tables")
	} else {
		commands = append(commands, *command)
	}

	if !strings.Contains("remove-old-databases", *command) {
		transferTables, err = ytconnector.NewTransferYtBackup()
		if err != nil {
			log.Fatal(err)
		}
	}

	for _, cmd := range commands {
		switch cmd {
		case "list":
			fmt.Println(transferTables)
		case "download-tables":
			if _, err := ytconnector.RunTransferYtBackups(transferTables); err != nil {
				log.Fatalln(err)
			}
		case "update-gtid":
			if err := ytconnector.UpdateYtMysqlSyncStates(gtids, transferTables); err != nil {
				log.Fatal(err)
			}
		case "convert-tables":
			if err := ytconnector.RunConvertTables(transferTables); err != nil {
				log.Fatal(err)
			}
		case "update-link":
			if err := ytconnector.ChangeYtCurrentLink(transferTables); err != nil {
				log.Fatal(err)
			}
		case "remove-old-databases":
			if err := ytconnector.RemoveOldTables(); err != nil {
				log.Fatal(err)
			}
		default:
			return
		}
	}
}
