package support

import (
	"bytes"
	"encoding/json"
	"errors"
	"fmt"
	"math/rand"
	"sort"
	"strings"
	"sync"
	"time"

	y "a.yandex-team.ru/direct/infra/dt-db-manager/pkg/yttransfer"
)

var (
	stageMutex = &sync.Mutex{}
	//явно указываем последовательность выполнения для вывода task-status
	STAGES = []string{"clean", "copy", "prepare_data", "teleport_saved", "prepare_grants",
		"prepare_packages", "move_databases", "set_gtid", "teleport_restore", "prepare_haproxy", "start_replication",
		"yt_transfer_tables", "yt_update_gtid", "yt_convert_tables", "yt_update_current_link",
		"yt_replicator_restart", "yt_clean_directory", "yt_check_replication", "binlogwriter_update",
		"binlogwriter_stop", "binlogwriter_start"}

	StagesStat = []string{"unpaused", "start", "finish"}
)

const (
	ReserveSpace = 1 << 39 //512Gb
	MINTASKAGE   = 2 * time.Minute
)

type AllResources struct {
	*Hosts
	*Complects
	*YtReplicator
}

func NewPlan(hosts *Hosts, complects *Complects, replicator *YtReplicator) *AllResources {
	return &AllResources{hosts, complects, replicator}
}

type Sheduled map[string]Backups

//Распределяет бекапы по доступным машинам. При выборе сначала учитывается количество
//бекапов на одной машине, а потом свободное место. Например: если на машину не влезает бекап, то
//он будет отправлен на машину с наименьшим количеством баз и достаточному количеству места.
//На вход принимает базу бекапа и список машин, меняет данные в планировании и возвращает true/false.
func (s *Sheduled) sheduleBackup(backup Backup, hosts *Hosts) bool {
	counts := make(map[int]*Hosts) //для группировки объектов Host по количеству бекапов
	for hostname, backups := range *s {
		host, ok := hosts.FindHost(hostname)
		if !ok {
			break
		}
		saved, ok := counts[len(backups)]
		if !ok {
			saved = NewHosts()
		}
		*saved = append(*saved, *host)
		counts[len(backups)] = saved
	}

	var countBackups []int //сортируем по возрастанию для более быстрого выбора
	for cnt := range counts {
		countBackups = append(countBackups, cnt)
	}
	sort.Ints(countBackups)

	for _, cnt := range countBackups {
		availibleHost := (*counts[cnt]).MaxFreeSpaceHost()
		prognosedSizeHost := (*s)[availibleHost.GetHost()].TotalPrognosedSize() //место, которое зарезервировано другими бекапами
		prognosedSizeBackup := backup.GetPrognosedSize()                        //приблизительный размер бекапа
		availibleSizeFree := availibleHost.GetFreeSpace()                       //свободное место на машине
		Wrap(logger.Info(f("[sheduleBackup] availibleHost: %s free size %d countBackups %d\n", availibleHost, availibleSizeFree, cnt)))
		Wrap(logger.Info(f("[sheduleBackup] prognosedSizeBackup: %d prognosedSizeBackup %d\n", prognosedSizeHost, prognosedSizeBackup)))
		if availibleSizeFree >
			(prognosedSizeHost + prognosedSizeBackup + ReserveSpace) {
			(*s)[availibleHost.GetHost()] = append((*s)[availibleHost.GetHost()], backup)
			return true
		}
	}
	return false
}

func (ar *AllResources) NewPlanRestore(group Group, complectName string, instances []string, enableReplicator bool) *Verdict {
	Wrap(logger.Info(f("new plan for %s comlect name %s instances %s", group, complectName, instances)))
	planHostsWithReplica := ar.FindGroupHosts(group)
	//var planBackups Backups
	var plans Plans
	var verdict *Verdict
	errbuf := bytes.NewBufferString("")
	lostBackups := make(map[Group]Backups)

	for _, gr := range planHostsWithReplica.ReplicsPerGroupName(group) {
		var planBackups Backups

		planHosts := planHostsWithReplica.FindGroupHosts(gr)

		if len(*planHosts) == 0 {
			msg := fmt.Sprintf("для указанной группы %s не найдены сервера: %v", group, planHosts)
			Wrap(logger.Warning(msg))
			errbuf.WriteString(msg)
			continue
		}

		if strings.Contains(complectName, "last") {
			planBackups = ar.LastFullComplects()
		} else {
			if plan, ok := ar.HasFullComplects(complectName); ok {
				planBackups = plan
			} else {
				Wrap(logger.Crit(f("not found full complect with name %s", complectName)))
			}
		}

		//отфильтровываем нужные нам инстансы
		if len(instances) > 0 {
			planBackups.FilterBackupsPerInstance(instances)
		}

		if len(planBackups) == 0 {
			msg := "нет бекапов для восстановления"
			Wrap(logger.Warning(msg))
			errbuf.WriteString(msg)
			//verdict = newVerdict(plans, complectName, group, planBackups, *planHosts, errors.New(msg))
			//verdict.PrintPlan()
			//return verdict
			continue
		}

		sheduled := make(Sheduled)
		for _, host := range *planHosts {
			sheduled[host.GetHost()] = NewBackups()
		}

		for _, backup := range planBackups {
			if ok := sheduled.sheduleBackup(backup, planHosts); !ok {
				lostBackups[gr] = append(lostBackups[gr], backup)
			}
		}

		if len(lostBackups[gr]) > 0 {
			msg := fmt.Sprintf("для бекапов %s не найдено ресурсов", strings.Join(lostBackups[gr].AllInstances(), ","))
			errbuf.WriteString(msg)
		}

		Wrap(logger.Info(f("verdict backup\n: %+v\n", sheduled)))
		for hostname, backups := range sheduled {
			host, _ := (planHosts).FindHost(hostname)
			plans = append(plans, Plan{*host, backups})
		}
	}

	replicator := ar.YtReplicator
	replicatorVerdict := NewReplicatorVerdict(*replicator, enableReplicator, group)
	if enableReplicator {
		if *replicator.Error != nil {
			msg := f("not found group %s for restore yt replicator", group.GetGroupName())
			errbuf.WriteString(msg)
			replicatorVerdict.LastError = msg
		} else {
			ytconn, err := NewGroupYtReplicator(*replicator)
			if err != nil {
				errbuf.WriteString(err.Error())
				replicatorVerdict.LastError = err.Error()
			} else {
				//TABLES !!
				if tt, err := ytconn.NewTransferYtBackup(); err != nil {
					errbuf.WriteString(err.Error())
					replicatorVerdict.LastError = err.Error()
					Wrap(logger.Warning(f("error get new transfer tables: %s", err)))
				} else {
					replicatorVerdict.TransferTables = tt
				}
			}
		}
	}

	var err error
	if errbuf.Len() > 0 {
		err = errors.New(errbuf.String())
	}
	verdict = newVerdict(plans, complectName, group, lostBackups, *planHostsWithReplica, err, replicatorVerdict)
	verdict.PrintPlan()
	return verdict
}

func (v *Verdict) ChangeYTBackup() bool {
	return false
}

func (v *Verdict) PrintPlan() {
	if v.Error != nil {
		fmt.Printf("[Verdict/PrintPlan] Error planning: %s\n", v.Error)
	}

	if len(v.Plans) > 0 {
		for _, group := range v.AllGroups() {
			plans := v.Plans.PlanPerGroup(group)
			if len(plans) == 0 {
				continue
			}
			fmt.Printf("\tТекущий план рассчитан на %d серверов для %s:\n", len(plans), group)
			for _, plan := range plans {
				if len(plan.Backups) == 0 {
					continue
				}
				fmt.Printf("\t\tна сервер %s со свободным местом %s планируется установить:\n", plan.GetHost(), plan.GetFreeSpace().SizeHuman())
				for _, backup := range plan.Backups {
					fmt.Printf("\t\t\t%s с приблизительным размером %s\n", backup.NameBackup, backup.GetPrognosedSize().SizeHuman())
				}
			}
			if len(v.LostBackups[group]) > 0 {
				fmt.Printf("\t\tДля следующих бекапов не найдено ресурсов:\n")
				for _, backup := range v.LostBackups[group] {
					fmt.Printf("\t\t\t%s c приблизительным размером %s\n", backup.NameBackup, backup.GetPrognosedSize().SizeHuman())
				}
			}
		}
	}
	r := v.Replicator
	if r.Enable {
		fmt.Printf("\tПлан восстановления для YT\n")
		if len(r.LastError) > 0 {
			fmt.Printf("\t\tневозможно восстановить %s%s/%s --> %s%s/%s: %s\n", r.SourceCluster, r.SourceDir, r.TransferTables.Name(),
				r.DestinationCluster, r.DestinationDir, r.TransferTables.Name(), r.LastError)
		} else {
			fmt.Printf("\t\tвосстанавливаем %s%s/%s --> %s%s/%s\n", r.SourceCluster, r.SourceDir, r.TransferTables.Name(),
				r.DestinationCluster, r.DestinationDir, r.TransferTables.Name())
		}
	} else {
		fmt.Printf("\tНе указан флаг для репликации YT\n")
	}

}

type Plan struct {
	Host
	Backups
}

func (ps Plans) FindPlan(host Host) (Plan, bool) {
	for _, plan := range ps {
		if host.GetHost() == plan.GetHost() {
			return plan, true
		}
	}
	return Plan{}, false
}

func (ps Plans) SizeBackupForHost(host Host) int {
	size := 0
	var foundedPlan Plan
	var ok bool
	if foundedPlan, ok = ps.FindPlan(host); !ok {
		Wrap(logger.Err(f("[analytic/SizeBackupForHost] not found plan per group %s", host)))
		return 0
	}
	for _, backup := range foundedPlan.Backups {
		size += backup.Size
	}
	return size
}

type Plans []Plan

func (ps Plans) PlanPerGroup(group Group) (result Plans) {
	for _, plan := range ps {
		saved := plan
		if saved.HasGroup(group) {
			result = append(result, saved)
		}
	}
	return
}

type Verdict struct {
	Plans
	ComplectName string
	LostBackups  map[Group]Backups
	MyGroup      Group
	Hosts              //копия для получения свободных ресурсов на машине
	Error        error `json:"-"`
	Replicator   YtReplicatorVerdict
}

type YtReplicatorVerdict struct {
	YtReplicator
	Enable         bool
	LastError      string
	TransferTables y.YtTransferTables
	Hostname       string
}

func NewReplicatorVerdict(replicator YtReplicator, enable bool, group Group) YtReplicatorVerdict {
	return YtReplicatorVerdict{
		replicator,
		enable,
		"",
		y.YtTransferTables{},
		fmt.Sprintf("localhost_%s", group.GetGroupName()),
	}
}

func newVerdict(plans Plans, complectName string, groupName Group, lostBackup map[Group]Backups, hosts Hosts,
	err error, rv YtReplicatorVerdict) *Verdict {
	return &Verdict{plans,
		complectName,
		lostBackup,
		groupName,
		hosts,
		err,
		rv}
}

func (v Verdict) GetGroupName() Group {
	return v.MyGroup
}

type StageStatus struct {
	CurrentStage string `json:"current-status"`
	LastStage    string `json:"last-status"`
}

func NewStageStatus(name string) StageStatus {
	return StageStatus{CurrentStage: name}
}

func (ss *StageStatus) SetStage(newstage string) {
	Wrap(logger.Info(f("set stage: new %s, last %s", newstage, ss.GetLastStage())))
	stageMutex.Lock()
	defer stageMutex.Unlock()
	if newstage != ss.CurrentStage {
		ss.LastStage = ss.CurrentStage
		ss.CurrentStage = newstage
	}
}

func (ss StageStatus) GetCurrentStage() string {
	return ss.CurrentStage
}

func (ss StageStatus) GetLastStage() string {
	return ss.LastStage
}

type Instances []string

type Timer struct {
	StartTime int64
	EndTime   int64
}

func NewTimer() *Timer {
	return &Timer{
		StartTime: time.Now().Unix(),
		EndTime:   0,
	}
}

type GroupManager struct {
	MyGroup            Group                 `json:"group"`           //имя группы
	GroupStatus        *StageStatus          `json:"group-status"`    //failed, success, ready_to_move and etc
	Servers            *Hosts                `json:"-"`               //ресурсы серверов: количество инстансов, память, место
	ComplectName       *string               `json:"complect-name"`   //имя коплекта бекапов для восстановления(для sheduler)
	AllowInstances     *Instances            `json:"allow-instances"` //список инстансов для восстановления(для sheduler)
	FinishStage        *string               `json:"finish-stage"`    //на каком stage следует остановиться
	TasksStatus        `json:"task-id-list"` //состояние выполняемых задач
	*Verdict           `json:"verdict"`      //план восстановления(на какие машины что ставить)
	History            map[string]*Timer     `json:"stage-timer"`         //время на прохождение stage из start до finish
	EnableReplicator   *bool                 `json:"enable-replicator"`   //включить заливку репликатора
	EnableBinlogWriter *bool                 `json:"enable-binlogwriter"` //включить обновление статистики у binlogwriter
	//*YtReplicator    `json:"-"`
	*Config `json:"-"`
}

func NewGroupManager(group, currentStage, finishStage string, hosts *Hosts, config Config) GroupManager {
	/*if len(currentStage) == 0 {
		currentStage = "DEFAULT"
	}*/
	stage := StageStatus{currentStage, ""}
	complectName := ""
	allowInstances := Instances([]string{})
	verdict := Verdict{}
	enableReplicator := false
	enableBinlogWriter := false
	var taskList = make(map[string]*TaskStatus)
	var stageTimer = make(map[string]*Timer)
	//fmt.Printf("HOSTS REPLICATOR %+v\n", replicator)
	return GroupManager{Group(group), &stage,
		hosts, &complectName, &allowInstances,
		&finishStage, taskList, &verdict, stageTimer,
		&enableReplicator, &enableBinlogWriter, &config}
}

type GroupJSONStatistic struct {
	GroupName Group `json:"group-name"`
	Servers   Hosts `json:"servers"`
}

//уничтожает все таски, в группе GroupManager
func (gm *GroupManager) ClearGroupTasks() {
	gm.TasksStatus.RemoveAllTasks()
}

func (gm *GroupManager) CheckActiveInstances() bool {
	verdict := *gm.Verdict
	servers := *gm.Servers.FindGroupHosts(gm.MyGroup)
	for _, plan := range verdict.Plans {
		hostnameInPlan := plan.GetHost()
		instancesInPlan := plan.Backups.AllInstances()
		if len(instancesInPlan) == 0 {
			return true
		}
		host, ok := servers.FindHost(hostnameInPlan)
		if !ok {
			Wrap(logger.Crit(f("[analytic/RestoreStages] not found host %s in %v", hostnameInPlan, servers)))
			return false
		}
		activeInstances := (*host).ActiveInstaces()
		for _, instanceName := range instancesInPlan {
			if !activeInstances.HasInstance(instanceName) {
				Wrap(logger.Crit(f("[analytic/RestoreStages] not found active instance %s in %s", instanceName, hostnameInPlan)))
				return false
			}
		}
	}
	return true
}

//выставляет начало времени выполнения stage
func (gm *GroupManager) SetStartTime() {
	status := (*gm).GetGroupStatus()
	if timer, ok := (*gm).History[status]; !ok {
		(*gm).History[status] = NewTimer()
	} else {
		(*timer).StartTime = time.Now().Unix()
	}
}

//выставляет кокончание времени выполнения stage
func (gm *GroupManager) SetEndTime() {
	status := (*gm).GetGroupStatus()
	if timer, ok := (*gm).History[status]; ok {
		(*timer).EndTime = time.Now().Unix()
	}
}

func (gm *GroupManager) CleanHistory() {
	var deleted []string
	for key := range (*gm).History {
		deleted = append(deleted, key)
	}
	for _, key := range deleted {
		delete((*gm).History, key)
	}
}

func (gm GroupManager) GetTimer(nameStage string) *Timer {
	if timer, ok := gm.History[nameStage]; ok {
		return timer
	}
	return nil
}

//выставляет имя комплекта бекапов для восстановления: либо дата в виде стоки(например 20190809), либо "last".
// "last" таже берется при пустой переменной.
func (gm *GroupManager) SetComplectName(name string) {
	if len(name) == 0 {
		name = "last"
	}
	*gm.ComplectName = name
	Wrap(logger.Info(f("[analytic/SetComplectName] set %s", *gm.ComplectName)))
}

//возвращает строку с именем комплекта бекапов
func (gm *GroupManager) GetComplectName() string {
	return *gm.ComplectName
}

//выставляет stage, после которого задача перейдет в paused. Если последний stage равен finish, то выставится success
func (gm *GroupManager) SetFinishStage(fstage string) {
	*gm.FinishStage = fstage
	Wrap(logger.Info(f("[analytic/SetFinishStage] set %s for %s", fstage, gm.MyGroup)))
}

//возврашает строку с именем завершающего stage
func (gm *GroupManager) GetFinishStage() string {
	return *gm.FinishStage
}

//выставляет список mysql баз, которые требуется восстановить в текущей группе.
//Если список пустой - восстанавливаются все известные.
func (gm *GroupManager) SetAllowInstances(instances []string) {
	*gm.AllowInstances = Instances(instances)
}

//возвращает список строк с именами баз
func (gm *GroupManager) GetAllowInstances() []string {
	return *gm.AllowInstances
}

//выставляем stage в зависимости от условий:
//  1. в текущем stage все задачи SUCCESS - выставляется новый newstage
//  2. в текущем stage все задачи SUCCESS и текущий stage равен завершающемуся stage - выставляется paused
//  3. в текущем stage любая задача FAILED - выставляется failed
func (gm *GroupManager) SetStageIfSuccess(newstage string) {
	stage := (*gm).GetGroupStatus()
	if last := (*gm).GetLastStatus(); stage == "failed" && !gm.TasksStageAnyStatus("FAILED", last) {
		Wrap(logger.Info(f("[analytic/SetStageIf] failed tasks %v %v",
			gm.TasksStageAnyStatus("FAILED", last), last)))
		gm.SetGroupStatus(last)
	}
	Wrap(logger.Info(f("[analytic/SetStageIf] all success tasks = %t for group %v",
		gm.TasksStageAllStatus("SUCCESS", stage), stage)))
	if gm.TasksStageAllStatus("SUCCESS", stage) {
		gm.SetEndTime()
		if strings.Contains(newstage, "finish") {
			gm.SetGroupStatus(newstage)
			return
		}
		if strings.Contains(stage, gm.GetFinishStage()) {
			gm.SetGroupStatus("paused")
			return
		}
		gm.SetGroupStatus(newstage)
		return
	}
	Wrap(logger.Info(f("[analytic/SetStageIf] any failed tasks = %t for group %v %v",
		gm.TasksStageAnyStatus("FAILED", stage), stage)))
	if stage != "failed" && gm.TasksStageAnyStatus("FAILED", stage) {
		gm.SetEndTime()
		gm.SetGroupStatus("failed")
	}
}

type HaproxyFormats []HaproxyFormat

type HaproxyFormat struct {
	Host     string `json:"host"`
	Instance string `json:"instance"`
	Port     int    `json:"port"`
	Group    Group  `json:"group"`
}

//выводит json формат плана востановления для генерации haproxy конфига
func (gm *GroupManager) PlanForHaproxy() (data HaproxyFormats) {
	hosts := (*gm).Servers.FindGroupHosts(gm.MyGroup)
	for _, host := range *hosts {
		Wrap(logger.Info(fmt.Sprintln("[analytic/PlanForHaproxy] generate plan ", host.ActiveInstaces())))
		if _, ok := gm.Verdict.FindHost(host); !ok {
			continue
		}
		if plan, ok := gm.Verdict.FindPlan(host); ok {
			for _, instance := range host.ActiveInstaces() {
				if _, ok := plan.FindBackupByInstance(instance.InstanceName); !ok {
					continue
				}
				value := HaproxyFormat{
					Host:     host.GetHost(),
					Instance: instance.InstanceName,
					Port:     instance.DatabasePort,
					Group:    gm.MyGroup,
				}
				data = append(data, value)
			}
		}
	}
	Wrap(logger.Info(f("[analytic/PlanForHaproxy] %v", data)))
	return
}

type ReplicasFormats []ReplicasFormat

type ReplicasFormat struct {
	Host        string      `json:"host"`
	Instance    string      `json:"instance"`
	Port        int         `json:"port"`
	GroupName   GroupName   `json:"group-name"`
	ReplicaName ReplicaName `json:"replica-name"`
}

func (rf ReplicasFormat) ShortName() (name string) {
	name = strings.ReplaceAll(rf.Instance, "ppcdata", "ppc:")
	name = strings.ReplaceAll(name, "ppcmonitor", "monitor")
	return
}

//выводит json формат плана востановления для генерации haproxy конфига
func (gsm *GroupsManager) PlansForReplicas(group Group) (data ReplicasFormats) {
	var plans Plans
	var hosts Hosts
	for _, gm := range *gsm {
		if strings.Contains(gm.MyGroup.ToString(), string(group.GetGroupName())) {
			plans = append(plans, gm.PlanPerGroup(gm.MyGroup)...)
			hosts = append(hosts, *gm.Servers.HostsPerGroup(gm.MyGroup)...)
		}
	}

	for _, host := range hosts {
		Wrap(logger.Info(fmt.Sprintln("[analytic/PlanForHaproxy] generate plan ", host.ActiveInstaces())))
		if plan, ok := plans.FindPlan(host); ok {
			for _, instance := range host.ActiveInstaces() {
				//if instance.InstanceName == "ppcdict" || instance.InstanceName == "sandbox" {
				//	continue
				//}
				if _, ok := plan.FindBackupByInstance(instance.InstanceName); !ok {
					continue
				}
				for _, hg := range host.GetGroups() {
					value := ReplicasFormat{
						Host:        host.GetHost(),
						Instance:    instance.InstanceName,
						Port:        instance.DatabasePort,
						GroupName:   hg.GetGroupName(),
						ReplicaName: hg.GetReplica(),
					}
					data = append(data, value)
				}
			}
		}
	}
	Wrap(logger.Info(f("[analytic/PlanForReplicas] %v", data)))
	return
}

type GroupsManager []*GroupManager //список групп

func NewGroupsManager() *GroupsManager {
	return &GroupsManager{}
}

//проверяет входит ли указанная группа в текущие
func (gsm GroupsManager) HasGroup(name interface{}) bool {
	groupName := FormatGroup(name)
	for _, group := range gsm {
		if group.MyGroup == groupName {
			return true
		}
	}
	return false
}

//добавляет newgroup  в текущий список групп
func (gsm *GroupsManager) AddMangerGroups(newgroup GroupManager) {
	gsm.AddManagerGroup(&newgroup)
}

//удаляет oldgroup из текущего списка групп
func (gsm *GroupsManager) DeleteMangerGroups(oldgroup GroupManager) {
	var deleted []int
	for num, group := range *gsm {
		if oldgroup.MyGroup == group.MyGroup {
			deleted = append(deleted, num)
		}
	}
	if len(deleted) > 0 {
		var cnt, num int
		for cnt, num = range deleted {
			(*gsm)[num] = (*gsm)[len(*gsm)-cnt-1]
		}
		*gsm = (*gsm)[:len(*gsm)-cnt-1]
	}
}

//
func (gsm *GroupsManager) CheckAllStatusGroups(stage Stage) bool {
	for _, group := range *gsm {
		if group.GetGroupStatus() != stage.StageName {
			return false
		}
	}
	return true
}

//принимает список групп и отдает словарь с ключем в виде статуса и списком групп mysql
//map[SUCCESS] = []string{dev7, testload}
func (gsm *GroupsManager) MonitoringStatus(groups Groups) (monstat map[string][]string) {
	monstat = make(map[string][]string)
	for _, gm := range *gsm {
		group := string(gm.MyGroup)
		if !groups.ContainGroup(gm.MyGroup) {
			continue
		}
		if value, ok := monstat[gm.GetGroupStatus()]; !ok {
			monstat[gm.GetGroupStatus()] = []string{group}
		} else {
			monstat[gm.GetGroupStatus()] = append(value, group)
		}
	}
	return
}

//Приводит group в тип Group
func FormatGroup(name interface{}) Group {
	var groupName Group
	switch g := name.(type) {
	case Group:
		groupName = g
	case Groups:
		groupName = g[0]
	case string:
		groupName = Group(g)
	case GroupName:
		groupName = Group(fmt.Sprintf("%v", g))
	default:
		Wrap(logger.Crit(f("[analytic/FormatGroup] error convert value %[0]v type %[0]T", name)))
		return Group("")
	}
	return groupName
}

//возвращает ссылку на группу из списка групп GroupsManager. Если группы нет, то создает ссылку на пустаю структуру.
func (gsm *GroupsManager) GetGroupManager(name interface{}) *GroupsManager {
	var groups GroupsManager
	groupName := FormatGroup(name)
	for _, group := range *gsm {
		if strings.Contains(group.MyGroup.ToString(), groupName.ToString()) {
			groups = append(groups, group)
		}
	}
	Wrap(logger.Info(f("[analytic/GetGroupManager] not found group %s", groupName)))
	return &groups
}

//получить список имен всех групп
func (gsm *GroupsManager) GetAllGroups() (groups Groups) {
	for _, gm := range *gsm {
		groups = append(groups, gm.MyGroup)
	}
	return
}

//добавить группу в список групп. Если группа существует - то ничего не делается.
func (gsm *GroupsManager) AddManagerGroup(group *GroupManager) {
	if (*gsm).HasGroup(group.MyGroup) {
		return
	}
	Wrap(logger.Info(f("[analytic/AddManagerGroup] adding %s into GroupsManager", group.MyGroup)))
	*gsm = append(*gsm, group)
}

//следит за выполнением задач в GroupsManager
func (gsm *GroupsManager) StartMonitorStages() {
	ticker := time.NewTicker(5 * time.Second)
	defer ticker.Stop()

	for range ticker.C {
		for _, group := range *gsm {
			if group.GetGroupStatus() != "default" {
				if err := group.RestoreStages(gsm); err != nil {
					Wrap(logger.Warning(f("[analytic/StartMonitorStages] failed restore: %s", err)))
				} //запускает новую задачу по необходимости и переключает stage из планируемого(ready_*) в запущенный
			}
		}

	}
}

//возвращает строку с указанием текущего статуса по группе: planning, ready_to_move, prepare и т.д.
func (gm *GroupManager) GetGroupStatus() string {
	return (*gm).GroupStatus.GetCurrentStage()
}

//возвращает строку с указанием прошлого статуса по группе. Например можно понять какой stage привел к failed
func (gm *GroupManager) GetLastStatus() string {
	return (*gm).GroupStatus.GetLastStage()
}

//выставляет текущий статус группы. Изменение этого параметра может привести к запуску stage.
//Например ready_prepare_data запустит stage=prepare_data
func (gm *GroupManager) SetGroupStatus(newstatus string) {
	gm.GroupStatus.SetStage(newstatus)
}

func (gm *GroupManager) SetReplicator(enable bool) {
	*gm.EnableReplicator = enable
}

func (gm *GroupManager) SetBinlogWriter(enable bool) {
	*gm.EnableBinlogWriter = enable
}

//вывести текущее состояние группы GroupManager
func (gm GroupManager) PrintGroupStatus() {
	fmt.Printf("[analytic/PrintGroupStatus] group %s stage \"%s\"\n", gm.MyGroup, *gm.GroupStatus)
}

//
func (gm GroupManager) ReplicatorYtCluster() YtReplicator {
	name := gm.MyGroup.GetGroupName()
	return gm.Config.LoadReplicatorConf(name)
}

//
func (gm GroupManager) ReplicatorYtHosts() Servers {
	return gm.ReplicatorYtCluster().ReplicatorHosts
}

//Запускает команды в зависимости от установленного stage
func (gm *GroupManager) RestoreStages(gms *GroupsManager) error {
	switch (*gm).GetGroupStatus() {
	//подготовка к наливке, чистка старых данных
	case "start":
		gm.ClearGroupTasks() //чистка всех задач у группы
		gm.CleanHistory()
		gm.SetGroupStatus("ready_to_clean")
	//чистка машин от старых или неполных бекапов
	case "ready_to_clean":
		verdict := *gm.Verdict
		ss := NewStageStatus("clean")
		gm.SetGroupStatus(ss.GetCurrentStage())
		(*gm).SetStartTime()
		for _, plan := range verdict.Plans {
			gm.RemoveTasks(plan.Hostname, ss)
			task := CleanOldBackup(plan.GetHost(), ss) //задача на удаление старых данных
			Wrap(logger.Info(f("[analytic/RestoreStages] add clean task: %+v", task)))
			(*gm).AddTask(&task) //добавляем в пул задач
		}
	//постройка плана наливки группы машин
	case "planning":
		gm.PrintGroupStatus()
		(*gm).SetStartTime()
		complects := NewComplects()
		if err := complects.CreateComplects(); err != nil {
			Wrap(logger.Crit(f("[analytic/RestoreStages] error getting complects: %s", err)))
		}
		replicator := (*gm).ReplicatorYtCluster()
		planning := NewPlan(gm.Servers, complects, &replicator)
		fmt.Printf("NEW PLAN %+v", replicator)
		Wrap(logger.Info(f("[analytic/RestoreStages] start planning for %s", strings.Join(gm.GetAllowInstances(), ","))))
		verdict := planning.NewPlanRestore(gm.MyGroup, gm.GetComplectName(), gm.GetAllowInstances(), *gm.EnableReplicator)
		*gm.Verdict = *verdict
		(*gm).SetEndTime()
		if verdict.Error != nil {
			Wrap(logger.Crit(f("[analytic/RestoreStages] error verdict %s", verdict.Error.Error())))
			gm.SetGroupStatus("failed")
			return verdict.Error
		}
		gm.SetGroupStatus("ready_to_copy")
	//запуск копирования бекапов из mds
	case "ready_to_copy":
		gm.PrintGroupStatus()
		verdict := *gm.Verdict
		ss := NewStageStatus("copy")
		gm.SetGroupStatus(ss.GetCurrentStage())
		(*gm).SetStartTime()
		for _, plan := range verdict.Plans {
			gm.RemoveTasks(plan.Hostname, ss)
			for _, backup := range plan.Backups {
				task := DownloadBackup(plan.GetHost(), backup.NameBackup, backup.GetInstance(), ss)
				Wrap(logger.Info(f("[analytic/RestoreStages] add download_data task: %+v", task)))
				(*gm).AddTask(&task)
			}
		}
		Wrap(logger.Info(f("[analytic/RestoreStages] verdict: %+v\n hosts: %+v\n", gm.Verdict, gm.Hosts)))
	//восстановление консистентности базы innobackupex
	case "ready_prepare_data":
		verdict := *gm.Verdict
		ss := NewStageStatus("prepare_data")
		gm.SetGroupStatus(ss.GetCurrentStage())
		(*gm).SetStartTime()
		for _, plan := range verdict.Plans {
			gm.RemoveTasks(plan.Hostname, ss)
			for _, backup := range plan.Backups {
				task := PrepareBackup(plan.GetHost(), backup.NameBackup, backup.GetInstance(), ss)
				Wrap(logger.Info(f("[analytic/RestoreStages] add prepare_data task: %+v", task)))
				(*gm).AddTask(&task)
			}
		}
	//сохранение старых копманий для тестирования
	case "ready_teleport_saved":
		ss := NewStageStatus("teleport_saved")
		gm.SetGroupStatus(ss.GetCurrentStage())
		(*gm).SetStartTime()
		hosts := gm.Config.TeleportHosts
		for _, hostname := range hosts {
			gm.RemoveTasks(hostname, ss)
			task := TeleportSavedTask(hostname, ss, gm.MyGroup)
			Wrap(logger.Info(f("[analytic/RestoreStages] add teleport_saved task: %+v", task)))
			(*gm).AddTask(&task)
		}
	//временный запуск новых баз для применения новых грантов mysql
	case "ready_prepare_grants":
		verdict := *gm.Verdict
		ss := NewStageStatus("prepare_grants")
		gm.SetGroupStatus(ss.GetCurrentStage())
		(*gm).SetStartTime()
		for _, plan := range verdict.Plans {
			gm.RemoveTasks(plan.Hostname, ss)
			for _, backup := range plan.Backups {
				task := PrepareTestingGrants(plan.GetHost(), backup.GetInstance(), ss)
				Wrap(logger.Info(f("[analytic/RestoreStages] add prepare_grants task: %+v", task)))
				(*gm).AddTask(&task)
			}
		}
	//установка новых пакетов с конфигами баз и удаление старых
	case "ready_prepare_packages":
		verdict := *gm.Verdict
		ss := NewStageStatus("prepare_packages")
		gm.SetGroupStatus(ss.GetCurrentStage())
		(*gm).SetStartTime()
		for _, plan := range verdict.Plans {
			gm.RemoveTasks(plan.Hostname, ss)
			//если под хост нет бекапов - пропускаем его
			if len(plan.Backups) == 0 {
				continue
			}
			task := PrepareMysqlPackages(plan.GetHost(), plan.AllInstances(), ss, gm.MyGroup)
			Wrap(logger.Info(f("[analytic/RestoreStages] add prepare_packages task: %+v", task)))
			(*gm).AddTask(&task)
		}
		for _, host := range *gm.HostsPerGroup(gm.MyGroup) {
			if _, ok := verdict.Plans.FindPlan(host); !ok {
				task := PrepareMysqlPackages(host.GetHost(), []string{}, ss, gm.MyGroup)
				(*gm).AddTask(&task)
			}
		}
	//остановка старых баз и запуск новых
	case "ready_move_databases":
		verdict := *gm.Verdict
		ss := NewStageStatus("move_databases")
		gm.SetGroupStatus(ss.GetCurrentStage())
		(*gm).SetStartTime()
		for _, plan := range verdict.Plans {
			gm.RemoveTasks(plan.Hostname, ss)
			//если под хост нет бекапов - пропускаем его
			if len(plan.Backups) == 0 {
				continue
			}
			task := MoveMysqlDatabases(plan.GetHost(), gm.MyGroup, plan.AllInstances(), ss)
			Wrap(logger.Info(f("[analytic/RestoreStages] add move_databases task: %+v", task)))
			(*gm).AddTask(&task)
		}
		for _, host := range *gm.HostsPerGroup(gm.MyGroup) {
			if _, ok := verdict.Plans.FindPlan(host); !ok {
				task := MoveMysqlDatabases(host.GetHost(), gm.MyGroup, []string{}, ss)
				(*gm).AddTask(&task)
			}
		}
	//скидываем gtid для установки правильных позиций
	case "ready_set_gtid":
		verdict := *gm.Verdict
		ss := NewStageStatus("set_gtid")
		gm.SetGroupStatus(ss.GetCurrentStage())
		(*gm).SetStartTime()
		for _, plan := range verdict.Plans {
			gm.RemoveTasks(plan.Hostname, ss)
			//если под хост нет бекапов - пропускаем его
			if len(plan.Backups) == 0 {
				continue
			}
			task := PrepareGtidSets(plan.GetHost(), plan.AllInstances(), ss)
			Wrap(logger.Info(f("[analytic/RestoreStages] add prepare_packages task: %+v", task)))
			(*gm).AddTask(&task)
		}
	//восстановление компаний для тестирования
	case "ready_teleport_restore":
		ss := NewStageStatus("teleport_restore")
		gm.SetGroupStatus(ss.GetCurrentStage())
		(*gm).SetStartTime()
		hosts := gm.Config.TeleportHosts
		for _, hostname := range hosts {
			gm.RemoveTasks(hostname, ss)
		}
		rand.Seed(time.Now().Unix())
		i := rand.Intn(len(hosts))
		hostname := hosts[i]
		task := TeleportRestoreTask(hostname, ss, gm.MyGroup)
		Wrap(logger.Info(f("[analytic/RestoreStages] add ready_teleport_saved task: %+v", task)))
		(*gm).AddTask(&task)
	//генерируем новый конфиг haproxy
	case "ready_prepare_haproxy":
		verdict := *gm.Verdict
		ss := NewStageStatus("prepare_haproxy")
		gm.SetGroupStatus(ss.GetCurrentStage())
		(*gm).SetStartTime()
		haproxyPlan := (*gm).PlanForHaproxy()
		var data []byte
		if data, err = json.Marshal(haproxyPlan); err != nil {
			Wrap(logger.Warning(f("error marshal %+v: %s", haproxyPlan, err)))
		}
		for _, plan := range verdict.Plans {
			gm.RemoveTasks(plan.Hostname, ss)
			task := PrepareHaproxyConfig(plan.GetHost(), data, ss)
			Wrap(logger.Info(f("[analytic/RestoreStages] add prepare_haproxy task: %+v", task)))
			(*gm).AddTask(&task)
		}
	//запускаем репликацию между базами mysql
	case "ready_start_replication":
		verdict := *gm.Verdict
		ss := NewStageStatus("start_replication")
		gm.SetGroupStatus(ss.GetCurrentStage())
		(*gm).SetStartTime()
		replicasPlan := (*gms).PlansForReplicas(verdict.MyGroup) //map[Instance]

		var data []byte
		if data, err = json.Marshal(replicasPlan); err != nil {
			Wrap(logger.Warning(f("error marshal %+v: %s", replicasPlan, err)))
		}

		for _, plan := range verdict.Plans {
			gm.RemoveTasks(plan.Hostname, ss)
			if plan.Backups.Len() == 0 {
				continue
			}
			task := PrepareReplication(plan.GetHost(), data, ss)
			Wrap(logger.Info(f("[analytic/RestoreStages] add prepare_replication task: %+v", task)))
			(*gm).AddTask(&task)
		}
	//копируем базы YT  через transfer manager
	case "ready_yt_transfer_tables":
		verdict := *gm.Verdict
		ss := NewStageStatus("yt_transfer_tables")
		gm.SetGroupStatus(ss.GetCurrentStage())
		(*gm).SetStartTime()
		r := verdict.Replicator
		tt := r.TransferTables
		task := DownloadYtTables(r.Hostname, r.SourceCluster, r.DestinationCluster, r.Account,
			fmt.Sprintf("%s/%s", r.SourceDir, tt.Name()), r.DestinationDir, r.YTTokenFile, ss)
		//конструкция нужна для правильной чистки группы с несколькими репликами. Например: dev7@rep1 + dev7@rep2
		for _, group := range *gms.GetGroupManager(gm.MyGroup.GetGroupName()) {
			group.RemoveTasks(r.Hostname, ss)
			(*group).AddTask(&task)
		}
	case "ready_yt_update_gtid":
		verdict := *gm.Verdict
		ss := NewStageStatus("yt_update_gtid")
		gm.SetGroupStatus(ss.GetCurrentStage())
		(*gm).SetStartTime()
		r := verdict.Replicator
		tt := r.TransferTables
		//hosts := gm.HostsPerGroup((*gm).MyGroup)
		hosts := gm.Servers.FindGroupHosts((*gm).MyGroup)
		gtid := hosts.GtidPerInstance()
		Wrap(logger.Info(f("[analytic/GtidPerInstance] %s", gtid)))
		task := UpdateYtGtid(gm.MyGroup, r.Hostname, r.DestinationCluster, r.Account,
			fmt.Sprintf("%s/%s", r.SourceDir, tt.Name()), r.DestinationDir, r.YTTokenFile, ss, gtid)
		for _, group := range *gms.GetGroupManager(gm.MyGroup.GetGroupName()) {
			group.RemoveTasks(r.Hostname, ss)
			(*group).AddTask(&task)
		}
	case "ready_yt_convert_tables":
		verdict := *gm.Verdict
		ss := NewStageStatus("yt_convert_tables")
		gm.SetGroupStatus(ss.GetCurrentStage())
		(*gm).SetStartTime()
		r := verdict.Replicator
		tt := r.TransferTables
		task := ConvertYtTables(r.Hostname, r.DestinationCluster, r.Account,
			fmt.Sprintf("%s/%s", r.SourceDir, tt.Name()), r.DestinationDir, r.YTTokenFile, ss)
		for _, group := range *gms.GetGroupManager(gm.MyGroup.GetGroupName()) {
			group.RemoveTasks(r.Hostname, ss)
			(*group).AddTask(&task)
		}
	case "ready_yt_update_current_link":
		verdict := *gm.Verdict
		ss := NewStageStatus("yt_update_current_link")
		gm.SetGroupStatus(ss.GetCurrentStage())
		(*gm).SetStartTime()
		r := verdict.Replicator
		tt := r.TransferTables
		task := UpdateYtCurrentLink(r.Hostname, r.DestinationCluster, r.Account,
			fmt.Sprintf("%s/%s", r.SourceDir, tt.Name()), r.DestinationDir, r.YTTokenFile, ss)
		for _, group := range *gms.GetGroupManager(gm.MyGroup.GetGroupName()) {
			group.RemoveTasks(r.Hostname, ss)
			(*group).AddTask(&task)
		}
	case "ready_yt_replicator_restart":
		verdict := *gm.Verdict
		ss := NewStageStatus("yt_replicator_restart")
		gm.SetGroupStatus(ss.GetCurrentStage())
		(*gm).SetStartTime()
		r := verdict.Replicator
		tt := r.TransferTables
		for _, hostname := range gm.ReplicatorYtHosts() {
			task := RestartReplicatorDaemon(hostname, ss, tt.Name())
			for _, group := range *gms.GetGroupManager(gm.MyGroup.GetGroupName()) {
				group.RemoveTasks(hostname, ss)
				(*group).AddTask(&task)
			}
		}
	case "ready_yt_check_replication":
		verdict := *gm.Verdict
		ss := NewStageStatus("yt_check_replication")
		gm.SetGroupStatus(ss.GetCurrentStage())
		(*gm).SetStartTime()
		replicasPlan := (*gms).PlansForReplicas(verdict.MyGroup) //map[Instance]

		var replicas []byte
		if replicas, err = json.Marshal(replicasPlan); err != nil {
			Wrap(logger.Warning(f("error marshal replicas plan %+v: %s", replicasPlan, err)))
		}
		fmt.Printf("REPLICAS %s %+v\n", replicas, replicasPlan)
		r := verdict.Replicator
		cnf := gm.Config
		task := CheckYtReplication(gm.MyGroup, r.Hostname, r.DestinationCluster, r.Account,
			r.DestinationDir, r.YTTokenFile, cnf.MysqlTokenFile, cnf.MysqlUser, replicas, ss)
		Wrap(logger.Info(f("[analytic/RestoreStages] add ready_yt_check_replication task: %+v", task)))
		for _, group := range *gms.GetGroupManager(gm.MyGroup.GetGroupName()) {
			group.RemoveTasks(r.Hostname, ss)
			(*group).AddTask(&task)
		}
	case "ready_yt_clean_directory":
		verdict := *gm.Verdict
		ss := NewStageStatus("yt_clean_directory")
		gm.SetGroupStatus(ss.GetCurrentStage())
		(*gm).SetStartTime()
		r := verdict.Replicator
		tt := r.TransferTables
		task := RemoveYtOldDatabases(r.Hostname, r.DestinationCluster, r.Account,
			fmt.Sprintf("%s/%s", r.SourceDir, tt.Name()), r.DestinationDir, r.YTTokenFile, ss)
		for _, group := range *gms.GetGroupManager(gm.MyGroup.GetGroupName()) {
			group.RemoveTasks(r.Hostname, ss)
			(*group).AddTask(&task)
		}
	case "ready_binlogwriter_update":
		ss := NewStageStatus("binlogwriter_update")
		gm.SetGroupStatus(ss.GetCurrentStage())
		(*gm).SetStartTime()
		cnf := (*gm).Config
		ytcnf := (*gm).Config.Replicators

		replicasPlan := (*gms).PlansForReplicas((*gm).MyGroup)
		gn := (*gm).MyGroup.GetGroupName()
		fmt.Println("CONFIG", ytcnf, gn)
		execHost := (*gm).MyGroup.GenerateLocalhost()

		Wrap(logger.Info(f("[analytic/GtidPerInstance] %s", replicasPlan)))
		for _, group := range *gms.GetGroupManager(gm.MyGroup.GetGroupName()) {
			group.RemoveTasks(execHost, ss)
		}
		for _, group := range *gms.GetGroupManager(gm.MyGroup.GetGroupName()) {
			for _, brokerType := range []string{"logbrokerwriter", "logbrokerwriter-json"} {
				task := UpdateBinlogwriterGtid(gm.MyGroup, execHost, ytcnf.GetBinlogWriterCluster(gn),
					ytcnf.GetYTAccount(gn), ytcnf.GetBinlogWriterDir(gn), ytcnf.GetYTTokenFile(gn), ss, replicasPlan,
					brokerType, cnf.MysqlUser, cnf.MysqlTokenFile)
				(*group).AddTask(&task)
			}
		}
	case "ready_binlogwriter_start":
		ss := NewStageStatus("binlogwriter_start")
		gm.SetGroupStatus(ss.GetCurrentStage())
		(*gm).SetStartTime()
		cnf := (*gm).Config.Replicators
		gn := (*gm).MyGroup.GetGroupName()
		fmt.Println(cnf.GetBinlogWriterHosts(gn), gn)
		for _, hostname := range cnf.GetBinlogWriterHosts(gn) {
			task := StartBinlogwriterDaemon(hostname, ss)
			for _, group := range *gms.GetGroupManager(gn) {
				group.RemoveTasks(hostname, ss)
				(*group).AddTask(&task)
			}
		}

	case "ready_binlogwriter_stop":
		ss := NewStageStatus("binlogwriter_stop")
		gm.SetGroupStatus(ss.GetCurrentStage())
		(*gm).SetStartTime()
		cnf := (*gm).Config.Replicators
		gn := (*gm).MyGroup.GetGroupName()
		for _, hostname := range cnf.GetBinlogWriterHosts(gn) {
			task := StopBinlogwriterDaemon(hostname, ss)
			for _, group := range *gms.GetGroupManager(gn) {
				group.RemoveTasks(hostname, ss)
				(*group).AddTask(&task)
			}
		}
	case "failed":
		Wrap(logger.Crit(f("[analytic/RestoreStages] ход восстановления БД для %s завершилась неудачно", gm.MyGroup)))
	case "success":
		Wrap(logger.Info(f("[analytic/RestoreStages] восстановление БД для %s комплекта %s завершено", gm.MyGroup, gm.ComplectName)))
	default:
		gm.PrintGroupStatus()
		Wrap(logger.Crit("[analytic/RestoreStages] not found stage"))
	}
	return nil
}

//мониторит текущее выполнение задач в GroupsManager и переводит группы в новый статус:
// при успешном выполнении - в указанный stage или 'paused'
// при неуспешном - 'failed'
func (gsm *GroupsManager) StartMonitoringGroupTasks() {
	ticker := time.NewTicker(5 * time.Second)
	defer ticker.Stop()

	for range ticker.C {
		for _, groupManager := range *gsm {
			go groupManager.UpdateTasksStatus()
			switch groupManager.GetGroupStatus() {
			case "clean":
				groupManager.SetStageIfSuccess("planning")
			case "copy":
				groupManager.SetStageIfSuccess("ready_prepare_data")
			case "prepare_data":
				groupManager.SetStageIfSuccess("ready_teleport_saved")
			case "teleport_saved":
				groupManager.SetStageIfSuccess("ready_prepare_grants")
			case "prepare_grants":
				groupManager.SetStageIfSuccess("ready_prepare_packages")
			case "prepare_packages":
				groupManager.SetStageIfSuccess("ready_move_databases")
			case "move_databases":
				groupManager.SetStageIfSuccess("ready_check_databases")
			case "ready_check_databases":
				ss := NewStageStatus("check_databases")
				groupManager.SetGroupStatus(ss.GetCurrentStage())
				(*groupManager).SetStartTime()
				if (*groupManager).CheckActiveInstances() {
					groupManager.SetGroupStatus("ready_set_gtid")
					(*groupManager).SetEndTime()
				} else {
					groupManager.SetGroupStatus("failed")
				}
			case "set_gtid":
				groupManager.SetStageIfSuccess("ready_teleport_restore")
			case "teleport_restore":
				groupManager.SetStageIfSuccess("ready_prepare_haproxy")
			case "prepare_haproxy":
				groups := *gsm.GetGroupManager(groupManager.MyGroup.GetGroupName())
				if len(groups) == 1 {
					Wrap(logger.Info(f("/: %v. Skip replication", groups)))
					if groupManager.Replicator.Enable {
						groupManager.SetStageIfSuccess("waiting_yt_prepare")
					} else {
						groupManager.SetStageIfSuccess("success")
					}
				} else if len(groups) == 0 {
					Wrap(logger.Info(f("not found groups: %v. Paused", groups)))
					groupManager.SetStageIfSuccess("paused")
				} else {
					groupManager.SetStageIfSuccess("waiting_start_replication")
				}
			case "waiting_start_replication":
				//проверяем что на общей группе(реплики одной группы) все готово к применению репликации
				groups := *gsm.GetGroupManager(groupManager.MyGroup.GetGroupName())
				if groups.CheckAllStatusGroups(NewStage("", "waiting_start_replication")) {
					for _, group := range groups {
						//выставляем новый статус для всех подгрупп-реплик общей группы
						group.SetGroupStatus("ready_start_replication")
					}
				}
			case "start_replication":
				if groupManager.Replicator.Enable {
					groupManager.SetStageIfSuccess("waiting_yt_prepare")
				} else {
					groupManager.SetStageIfSuccess("success")
				}
			case "waiting_yt_prepare":
				groups := *gsm.GetGroupManager(groupManager.MyGroup.GetGroupName())
				if groups.CheckAllStatusGroups(NewStage("", "waiting_yt_prepare")) {
					for i, group := range groups {
						//выставляем новый статус для всех подгрупп-реплик общей группы
						if i == 0 {
							group.SetGroupStatus("ready_yt_transfer_tables")
						} else {
							group.SetGroupStatus("running_yt_prepare")
						}
					}
				}
			case "yt_transfer_tables":
				groupManager.SetStageIfSuccess("ready_yt_update_gtid")
			case "yt_update_gtid":
				groupManager.SetStageIfSuccess("ready_yt_convert_tables")
			case "yt_convert_tables":
				groupManager.SetStageIfSuccess("ready_yt_update_current_link")
			case "yt_update_current_link":
				groupManager.SetStageIfSuccess("ready_yt_replicator_restart")
			case "yt_replicator_restart":
				groupManager.SetStageIfSuccess("ready_yt_check_replication")
			case "yt_check_replication":
				groupManager.SetStageIfSuccess("ready_yt_clean_directory")
			case "yt_clean_directory":
				groupManager.SetStageIfSuccess("running_yt_prepare")
			case "running_yt_prepare":
				groups := *gsm.GetGroupManager(groupManager.MyGroup.GetGroupName())
				if groups.CheckAllStatusGroups(NewStage("", "running_yt_prepare")) {
					for _, group := range groups {
						//выставляем новый статус для всех подгрупп-реплик общей группы
						group.SetGroupStatus("waiting_binlogwriter_prepare")
					}
				}
			case "waiting_binlogwriter_prepare":
				groups := *gsm.GetGroupManager(groupManager.MyGroup.GetGroupName())
				if groups.CheckAllStatusGroups(NewStage("", "waiting_binlogwriter_prepare")) {
					for i, group := range groups {
						//выставляем новый статус для всех подгрупп-реплик общей группы
						if i == 0 {
							group.SetGroupStatus("ready_binlogwriter_stop")
						} else {
							group.SetGroupStatus("running_binlogwriter_prepare")
						}
					}
				}
			case "binlogwriter_stop":
				groupManager.SetStageIfSuccess("ready_binlogwriter_update")
			case "binlogwriter_update":
				groupManager.SetStageIfSuccess("ready_binlogwriter_start")
			case "binlogwriter_start":
				groupManager.SetStageIfSuccess("running_binlogwriter_prepare")
			case "running_binlogwriter_prepare":
				groups := *gsm.GetGroupManager(groupManager.MyGroup.GetGroupName())
				if groups.CheckAllStatusGroups(NewStage("", "running_binlogwriter_prepare")) {
					for _, group := range groups {
						//выставляем новый статус для всех подгрупп-реплик общей группы

						group.SetGroupStatus("success")
					}
				}
			case "failed":
				groupManager.SetStageIfSuccess(groupManager.GetLastStatus()) //пробуем выбраться из failed, если задача выполнится
			default:
			}
		}
	}
}

//перезапуск всех задач во всех группах со статусом FAILED и возрастом больше 2 минут
func (gsm *GroupsManager) RestartFailedTasks() {
	for _, gm := range *gsm {
		stage := gm.GroupStatus.GetCurrentStage()
		if strings.Contains(stage, "failed") {
			stage = gm.GroupStatus.GetLastStage()
		}
		stageTasks := gm.TasksStatus.GetStageTasks(stage)
		failedTasks := stageTasks.FailedTaskStatus()
		Wrap(logger.Info(f("[analytic/RestartFailedTasks] restart failed tasks. "+
			"last stage %s, stage tasks %+v, failed tasks: %+v", stage, stageTasks, failedTasks)))
		for hashID, task := range failedTasks {
			if time.Since(time.Unix(task.StartTime, 0)) < MINTASKAGE {
				Wrap(logger.Info(f("[analytic/RestartFailedTasks] task %s smaller %s. Skip restart him.", hashID, MINTASKAGE)))
				continue
			}
			gm.TasksStatus.RemoveTask(hashID)
			Wrap(logger.Info(f("[analytic/RestartFailedTasks] start task %v with stage %s", task.RemoteRequest, stage)))
			newTask := TaskExecute(task.RemoteRequest, stage) //пробуем перевыложить сломанный task
			gm.AddTask(&newTask)
		}
		Wrap(logger.Info(f("[analytic/RestartFailedTasks] restart failed tasks. new tasks %+v",
			gm.TasksStatus.GetStageTasks(stage))))
	}
}

func (gsm *GroupsManager) RemoveEmptyGroups() {
L:
	for {
		for i, gm := range *gsm {
			if hosts := gm.Servers.HostsPerGroup(gm.MyGroup); len(*hosts) == 0 {
				(*gsm)[i] = (*gsm)[len(*gsm)-1]
				*gsm = (*gsm)[:len(*gsm)-1]
				continue L
			}
		}
		break L
	}
}

//вывести в stdout текущее состояние задач в GroupsManager
func (gsm GroupsManager) PrintGroupManagerStatus(groups Groups) {
	for _, groupManager := range gsm {
		if !groups.ContainGroup(groupManager.MyGroup) {
			continue
		}
		var suffix string
		if len(groupManager.GetLastStatus()) > 0 && groupManager.GetLastStatus() != groupManager.GetGroupStatus() {
			suffix = fmt.Sprintf(" предыдущее состояние %s", groupManager.GetLastStatus())
		}
		fmt.Printf("\nДля группы %s текущее состояние: %s%s. Завершающий stage: %s\n", groupManager.MyGroup,
			groupManager.GetGroupStatus(), suffix, *groupManager.FinishStage)

		hosts := *groupManager.FindGroupHosts(groupManager.MyGroup)
		fmt.Printf("\tНа текущий момент доступно %d серверов со свободным местом:\n", len(hosts))
		for _, host := range hosts {
			fmt.Printf("\t\t%s доступно %s\n", host.GetHost(), host.GetFreeSpace().SizeHuman())
		}
		fmt.Println("")
		groupManager.PrintPlan()
		fmt.Println("")
		if groupManager.LenTasks() == 0 {
			continue
		}

		fmt.Printf("\tТаски выполнения:\n")
		for _, stage := range STAGES {
			tasks := groupManager.GetListTasks(stage)
			if len(tasks) > 0 {
				sort.Sort(tasks)
				for _, task := range tasks {
					var suffix string
					msg := task.GetTaskMessage()
					if len(msg) > 0 {
						suffix = fmt.Sprintf(", последнее сooбщение: %s", msg)
					}
					fmt.Printf("\t\t(%s) %s статус: %s%s\n", task.StageName, task.Hostname, task.GetTaskStatus(), suffix)
				}
			} else {
				fmt.Printf("\t\t(%s) не выполнялся\n", stage)
			}
		}

		fmt.Printf("\n\tВремя выполнения:\n")
		var workingTime int64
		for _, stage := range STAGES {
			var timer *Timer
			if timer = groupManager.GetTimer(stage); timer != nil {
				var suffix string
				if timer.EndTime == 0 {
					suffix = "еще не завершен"
				} else {
					workingTime += timer.EndTime - timer.StartTime
					endTime := time.Unix(timer.EndTime, 0)
					suffix = fmt.Sprintf("завершен в %s (%s)", endTime, endTime.Sub(time.Unix(timer.StartTime, 0)))
				}
				fmt.Printf("\t\t (%s) начался в %s %s\n", stage, time.Unix(timer.StartTime, 0), suffix)
			}
		}
		durationTime := time.Unix(workingTime, 0)
		duration := durationTime.Sub(time.Unix(0, 0))
		fmt.Printf("\n\tНа восстановление затрачено %s\n", duration)
	}
}

//запрос, для генерации плана восстановления баз в verdict
type PlanRequest struct {
	MyGroup            Group    `json:"group"`
	NameComplect       string   `json:"complect-name"`
	InstanceNames      []string `json:"instance-names"`
	FinishStage        string   `json:"finish-stage"`
	EnableReplicator   bool     `json:"enable-replicator"`
	EnableBinlogWriter bool     `json:"enable-binlogwriter"`
}

func NewPlanRequest(group, complect, fstage string, instances []string, replicator, binlogwriter bool) PlanRequest {
	return PlanRequest{
		MyGroup:            Group(group),
		NameComplect:       complect,
		InstanceNames:      instances,
		FinishStage:        fstage,
		EnableReplicator:   replicator,
		EnableBinlogWriter: binlogwriter,
	}
}

func (pr PlanRequest) GetNameComplect() string {
	if len(pr.NameComplect) == 0 {
		return "last"
	}
	return pr.NameComplect
}

func formatting(value interface{}) string {
	newval := strings.ToLower(fmt.Sprintf("%s", value))
	newval = strings.ReplaceAll(newval, "-", "_")
	return newval
}

func checkStageName(stageName string) error {
	stages := append(STAGES, StagesStat...)
	for _, stage := range stages {
		if strings.Contains(stageName, stage) || strings.Contains(stageName, "ready_"+stage) {
			return nil
		}
	}
	return errors.New(f("newstage name wrong: cur %s need %s or %s ", stageName,
		strings.Join(stages, ","),
		"ready_"+strings.Join(stages, ",ready_")))
}

func (pr *PlanRequest) Normalize() error {
	plan := *pr
	plan.MyGroup = Group(formatting(pr.MyGroup))
	plan.NameComplect = formatting(pr.NameComplect)
	plan.FinishStage = formatting(pr.FinishStage)
	if err := checkStageName(plan.FinishStage); err != nil {
		return err
	}
	*pr = plan
	return nil
}

func (pr PlanRequest) GetInstanceNames() []string {
	if len(pr.InstanceNames) == 0 {
		return []string{}
	}
	return pr.InstanceNames
}

//план восстановления баз с указанием машин и бекапов. В Error пробрасывается ошибка при планировании.
type PlanRestore struct {
	BackupPerHost map[string]Backups `json:"backup-per-host"`
	Error         error              `json:"error"`
}

//группа и ее статус
type Stage struct {
	GroupName Group  `json:"group-name"`
	StageName string `json:"stage-name"`
}

func NewStage(group, stage string) Stage {
	return Stage{GroupName: Group(group), StageName: stage}
}

func (s Stage) GetCurrentStage() string {
	return s.StageName
}

func (s Stage) GetLastStage() string {
	return ""
}

func Wrap(err error) {
	if err != nil {
		fmt.Printf("[wrapper] error: %s\n", err)
	}
}
