package support

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"sort"
	"strings"
	"time"

	rr "a.yandex-team.ru/direct/infra/dt-db-manager/pkg/sshrequest"
)

func CleanOldBackup(serverName string, stage StageStatus) TaskStatus {
	command := "sudo /usr/local/bin/direct-async-run --run-shell -j start '/bin/rm -rf /opt/mysql.*.{old,new,tmp}'"
	request := rr.NewRemouteRequest(serverName, command)
	return TaskExecute(request, stage.GetCurrentStage())
}

func DownloadBackup(serverName, nameBackup, instance string, stage StageStatus) TaskStatus {
	command := fmt.Sprintf("/usr/bin/pkill -9 -f /opt/mysql.%s.new", instance)
	request := rr.NewRemouteRequest(serverName, command)
	TaskExecute(request, stage.GetCurrentStage())
	command = fmt.Sprintf("sudo /usr/bin/mysql-backup-mds download -n %s --remove-data -e production -o /opt/mysql.%s.new", nameBackup, instance)
	request = rr.NewRemouteRequest(serverName, command)
	return TaskExecute(request, stage.GetCurrentStage())
}

func PrepareBackup(serverName, nameBackup, instance string, stage StageStatus) TaskStatus {
	command := fmt.Sprintf("/usr/bin/pkill -9 -f /opt/mysql.%s.new", instance)
	request := rr.NewRemouteRequest(serverName, command)
	TaskExecute(request, stage.GetCurrentStage())
	command = fmt.Sprintf("sudo /usr/bin/mysql-backup-mds prepare -n %s -e production --memory-prepare 2G -o /opt/mysql.%s.new", nameBackup, instance)
	request = rr.NewRemouteRequest(serverName, command)
	return TaskExecute(request, stage.GetCurrentStage())
}

func PrepareTestingGrants(serverName, instance string, stage StageStatus) TaskStatus {
	command := fmt.Sprintf("sudo /usr/local/bin/direct-async-run --run-shell -j start '/usr/local/bin/sync-db-prepare-db-testing.py %s.new'", instance)
	request := rr.NewRemouteRequest(serverName, command)
	return TaskExecute(request, stage.GetCurrentStage())
}

func PrepareMysqlPackages(serverName string, instances []string, stage StageStatus, group Group) TaskStatus {
	command := fmt.Sprintf("sudo /usr/local/bin/direct-async-run --run-shell -j start '/usr/local/bin/sync-db-prepare-mysql-packages.py --doit --group %s %s'", group.GetGroupName(), strings.Join(instances, " "))
	request := rr.NewRemouteRequest(serverName, command)
	return TaskExecute(request, stage.GetCurrentStage())
}

func MoveMysqlDatabases(serverName string, group Group, instances []string, stage StageStatus) TaskStatus {
	command := fmt.Sprintf("sudo /usr/local/bin/direct-async-run --run-shell -j start '/usr/local/bin/sync-db-prepare-start-new-databases.py --doit --group %s %s'", group.GetGroupName(), strings.Join(instances, " "))
	request := rr.NewRemouteRequest(serverName, command)
	return TaskExecute(request, stage.GetCurrentStage())
}

func PrepareGtidSets(serverName string, instances []string, stage StageStatus) TaskStatus {
	command := fmt.Sprintf("sudo /usr/local/bin/direct-async-run --run-shell -j start '/usr/local/bin/sync-db-purged-gtid.py --doit %s'", strings.Join(instances, " "))
	request := rr.NewRemouteRequest(serverName, command)
	return TaskExecute(request, stage.GetCurrentStage())
}

func PrepareHaproxyConfig(serverName string, data []byte, stage StageStatus) TaskStatus {
	const jsonTmpFile = "/tmp/haproxy.json"
	command := fmt.Sprintf("/bin/echo '%s' > %s", data, jsonTmpFile)
	request := rr.NewRemouteRequest(serverName, command)
	TaskExecute(request, stage.GetCurrentStage())
	command = fmt.Sprintf("sudo /usr/local/bin/direct-async-run --run-shell -j start '/usr/local/bin/sync-db-prepare-haproxy.py --doit --generate-initd --file %s'", jsonTmpFile)
	request = rr.NewRemouteRequest(serverName, command)
	return TaskExecute(request, stage.GetCurrentStage())
}

func PrepareReplication(serverName string, data []byte, stage StageStatus) TaskStatus {
	const jsonTmpFile = "/tmp/replication.json"
	command := fmt.Sprintf("/bin/echo '%s' > %s", data, jsonTmpFile)
	request := rr.NewRemouteRequest(serverName, command)
	TaskExecute(request, stage.GetCurrentStage())
	command = fmt.Sprintf("sudo /usr/local/bin/direct-async-run --run-shell -j start '/usr/local/bin/sync-db-prepare-replication.py --doit --file %s'", jsonTmpFile)
	request = rr.NewRemouteRequest(serverName, command)
	return TaskExecute(request, stage.GetCurrentStage())
}

func DownloadYtTables(serverName, clusterBackup, clusterRestore, account, dirBackup, dirRestore, tokenFile string, stage StageStatus) TaskStatus {
	command := fmt.Sprintf("/usr/local/bin/direct-async-run --run-shell -j start '/usr/local/bin/yt-transfer-client "+
		"-cluster-backup %s -cluster-restore %s -account %s -directory-backup %s -directory-restore %s -yt-token %s -command download-tables'",
		clusterBackup, clusterRestore, account, dirBackup, dirRestore, tokenFile)
	request := rr.NewRemouteRequest(serverName, command)
	return TaskExecute(request, stage.GetCurrentStage())
}

func UpdateYtGtid(group Group, serverName, clusterRestore, account, dirBackup, dirRestore, tokenFile string, stage StageStatus, gtidData GtidMysqlBinlog) TaskStatus {
	gtidFile := fmt.Sprintf("/tmp/gtid_status_%s.json", group.GetGroupName())
	data, _ := json.Marshal(gtidData)
	_ = ioutil.WriteFile(gtidFile, data, 0644)
	command := fmt.Sprintf("/usr/local/bin/direct-async-run --run-shell -j start '/usr/local/bin/yt-transfer-client "+
		"-cluster-restore %s -account %s -directory-backup %s -directory-restore %s -yt-token %s -command update-gtid -gtid-file %s'", clusterRestore,
		account, dirBackup, dirRestore, tokenFile, gtidFile)
	request := rr.NewRemouteRequest(serverName, command)
	return TaskExecute(request, stage.GetCurrentStage())
}

func ConvertYtTables(serverName, clusterRestore, account, dirBackup, dirRestore, tokenFile string, stage StageStatus) TaskStatus {
	command := fmt.Sprintf("/usr/local/bin/direct-async-run --run-shell -j start '/usr/local/bin/yt-transfer-client "+
		"-cluster-restore %s -account %s -directory-backup %s -directory-restore %s -yt-token %s -command convert-tables'", clusterRestore,
		account, dirBackup, dirRestore, tokenFile)
	request := rr.NewRemouteRequest(serverName, command)
	return TaskExecute(request, stage.GetCurrentStage())
}

func UpdateYtCurrentLink(serverName, clusterRestore, account, dirBackup, dirRestore, tokenFile string, stage StageStatus) TaskStatus {
	command := fmt.Sprintf("/usr/local/bin/direct-async-run --run-shell -j start '/usr/local/bin/yt-transfer-client "+
		"-cluster-restore %s -account %s -directory-backup %s -directory-restore %s -yt-token %s -command update-link'", clusterRestore,
		account, dirBackup, dirRestore, tokenFile)
	request := rr.NewRemouteRequest(serverName, command)
	return TaskExecute(request, stage.GetCurrentStage())
}

func CheckYtReplication(group Group, serverName, clusterRestore, account, dirRestore, tokenYtFile, tokenMysqlFile, mysqlUser string, data []byte, stage StageStatus) TaskStatus {
	jsonTmpFile := fmt.Sprintf("/tmp/replication_%s.json", group.GetGroupName())
	_ = ioutil.WriteFile(jsonTmpFile, data, 0644)
	command := fmt.Sprintf("/usr/local/bin/direct-async-run --run-shell -j start '/usr/local/bin/replicator-check "+
		"-cluster-restore %s -account %s -directory-restore %s -yt-token %s -mysql-token %s -mysql-user %s -instances-file %s'", clusterRestore, account, dirRestore,
		tokenYtFile, tokenMysqlFile, mysqlUser, jsonTmpFile)
	request := rr.NewRemouteRequest(serverName, command)
	return TaskExecute(request, stage.GetCurrentStage())
}

func RemoveYtOldDatabases(serverName, clusterRestore, account, dirBackup, dirRestore, tokenFile string, stage StageStatus) TaskStatus {
	command := fmt.Sprintf("/usr/local/bin/direct-async-run --run-shell -j start '/usr/local/bin/yt-transfer-client "+
		"-cluster-restore %s -account %s -directory-restore %s -yt-token %s -command remove-old-databases'", clusterRestore,
		account, dirRestore, tokenFile)
	request := rr.NewRemouteRequest(serverName, command)
	return TaskExecute(request, stage.GetCurrentStage())
}

func RestartReplicatorDaemon(execHost string, stage StageStatus, backupName string) TaskStatus {
	versionFile := "/etc/direct/binlog-to-yt-sync/sync-path.version"
	command := fmt.Sprintf("/bin/echo '%s' > %s", backupName, versionFile)
	request := rr.NewRemouteRequest(execHost, command)
	TaskExecute(request, stage.GetCurrentStage())
	command = "sudo /usr/local/bin/direct-async-run --run-shell -j start '/usr/bin/sv restart binlog-to-yt-sync'"
	request = rr.NewRemouteRequest(execHost, command)
	return TaskExecute(request, stage.GetCurrentStage())
}

func UpdateBinlogwriterGtid(group Group, execHost, ytCluster, account, ytDir, tokenFile string, stage StageStatus,
	replicasData ReplicasFormats, brokerType, mysqlUser, mysqlToken string) TaskStatus {
	replicasFile := fmt.Sprintf("/tmp/replicas_status_%s.json", group.GetGroupName())
	data, _ := json.Marshal(replicasData)
	_ = ioutil.WriteFile(replicasFile, data, 0644)
	command := fmt.Sprintf("/usr/local/bin/direct-async-run --run-shell -j start '/usr/local/bin/binlogwriter-update "+
		"-cluster %s -account %s -yt-directory %s -yt-token %s -instances-file %s -group %s -broker-type %s "+
		"-mysql-user %s -mysql-token %s'", ytCluster, account, ytDir, tokenFile, replicasFile, group.GetGroupName(),
		brokerType, mysqlUser, mysqlToken)
	request := rr.NewRemouteRequest(execHost, command)
	return TaskExecute(request, stage.GetCurrentStage())
}

func StopBinlogwriterDaemon(serverName string, stage StageStatus) TaskStatus {
	command := "sudo /usr/local/bin/direct-async-run --run-shell -j start '/usr/bin/sv stop direct-binlog-logbrokerwriter'"
	request := rr.NewRemouteRequest(serverName, command)
	return TaskExecute(request, stage.GetCurrentStage())
}

func StartBinlogwriterDaemon(serverName string, stage StageStatus) TaskStatus {
	command := "sudo /usr/local/bin/direct-async-run --run-shell -j start '/usr/bin/sv start direct-binlog-logbrokerwriter'"
	request := rr.NewRemouteRequest(serverName, command)
	return TaskExecute(request, stage.GetCurrentStage())
}

func TeleportSavedTask(serverName string, stage StageStatus, group Group) TaskStatus {
	command := "sudo /usr/local/bin/direct-async-run --run-shell -j start 'date'"
	request := rr.NewRemouteRequest(serverName, command)
	return TaskExecute(request, stage.GetCurrentStage())
}

func TeleportRestoreTask(serverName string, stage StageStatus, group Group) TaskStatus {
	command := "sudo /usr/local/bin/direct-async-run --run-shell -j start 'date'"
	request := rr.NewRemouteRequest(serverName, command)
	return TaskExecute(request, stage.GetCurrentStage())
}

type Backup struct {
	Date       string `json:"date"`
	Size       int    `json:"size"`
	NameBackup string `json:"name_backup"`
}

func NewBackup() Backup {
	return Backup{}
}

func (b Backup) GetInstance() string {
	var instance string
	value := strings.Split(b.NameBackup, "_")
	if len(value) != 0 {
		instance = value[0]
	}
	return instance
}

func (b Backup) GetPrognosedSize() Byte {
	return Byte(b.Size * 8)
}

type Backups []Backup

func NewBackups() Backups {
	return Backups{}
}

func (bs Backups) Len() int      { return len(bs) }
func (bs Backups) Swap(i, j int) { bs[i], bs[j] = bs[j], bs[i] }
func (bs Backups) Less(i, j int) bool {
	prev, next := len(bs[i].GetInstance()), len(bs[j].GetInstance())
	if prev < next {
		return true
	} else if prev == next && bs[i].GetInstance() < bs[j].GetInstance() {
		return true
	}
	return false
}

func (bs Backups) FindBackupByInstance(instance string) (Backup, bool) {
	for _, backup := range bs {
		if instance == backup.GetInstance() {
			return backup, true
		}
	}
	return NewBackup(), false
}

func (bs Backups) ComplectName() string {
	if len(bs) > 0 {
		return strings.Split(bs[0].NameBackup, "_")[1]
	}
	return "NONE"

}

func (bs Backups) AllInstances() (instances []string) {
	for _, backup := range bs {
		instances = append(instances, backup.GetInstance())
	}
	return
}

func (bs Backups) TotalPrognosedSize() (size Byte) {
	for _, backup := range bs {
		size += backup.GetPrognosedSize()
	}
	return
}

func (bs *Backups) FilterBackupsPerInstance(instances []string) {
	var newBackups Backups
	Wrap(logger.Info(f("filter backup for instances %s", instances)))
	for _, instance := range instances {
		if backup, ok := bs.FindBackupByInstance(instance); ok {
			newBackups = append(newBackups, backup)
			Wrap(logger.Info(f("%+v", newBackups)))
		}
	}
	*bs = newBackups
}

//key может быть instance, а может str(date)
type GroupBackups map[string]Backups

//Получает список бекапов из mds
func ListBackups() (groupBackups GroupBackups, err error) {
	var line []byte
	groupBackups = make(GroupBackups)
	command := f("sudo /usr/bin/mysql-backup-mds list -j -e production")
	if line, err = LocalExecute(command); err != nil {
		return
	}
	if err = json.Unmarshal(line, &groupBackups); err != nil {
		return
	}
	return
}

type Complects GroupBackups

func (cs Complects) MaxSize() (maxSize int) {
	for _, backups := range cs {
		size := len(backups)
		if size > maxSize {
			maxSize = size
		}
	}
	return
}

func NewComplects() *Complects {
	complect := make(Complects)
	return &complect
}

//Получаем список имен комплектов бекапов
func (cs Complects) ListComplectNames() (complectNames []string) {
	for name := range cs {
		complectNames = append(complectNames, name)
	}
	sort.Strings(complectNames)
	return
}

//На основе доступных бекапов, создаем комплекты. По структуре это хеш, с ключем=дате и значениями=именам бекапов.
// Например: {backup_date1: [ppcdata1, ppcdata2], backup_date2: [ppcdata1, ppcdata2]}
func (cs *Complects) CreateComplects() (err error) {
	complects := NewComplects()
	var groupBackups GroupBackups
	if groupBackups, err = ListBackups(); err != nil {
		return
	}
	for _, backups := range groupBackups {
		for _, backup := range backups {
			var (
				saved Backups
				ok    bool
			)
			date := backup.Date
			if saved, ok = (*complects)[date]; !ok {
				(*complects)[date] = NewBackups()
			}
			if _, ok := saved.FindBackupByInstance(backup.GetInstance()); ok {
				continue
			}
			saved = append(saved, backup)
			(*complects)[date] = saved
		}
	}
	*cs = *complects
	return
}

//На основе полученных из mds комплектов, выводит полные наборы(все бекапы за день).
func (cs Complects) FullComplects() Complects {
	for retry := 0; retry < 3; retry++ {
		if len(cs) > 0 {
			break
		}
		time.Sleep(5 * time.Second) //исключаем попадание в момент обновления данных о бекапах
	}
	var deletedKeys []string //ключи для удаления неполных наборов
	for date, backups := range cs {
		if len(backups) != cs.MaxSize() {
			deletedKeys = append(deletedKeys, date)
		}
	}
	for _, date := range deletedKeys {
		delete(cs, date) //тут идет работа с копией Complects
	}
	return cs
}

func (cs Complects) HasFullComplects(complectName string) (Backups, bool) {
	if backups, ok := cs.FullComplects()[complectName]; ok {
		return backups, true
	}
	Wrap(logger.Crit(f("комплект %s не полный", complectName)))
	return nil, false
}

func (cs Complects) LastFullComplects() Backups {
	var lastComplect Backups
	if cs.LenComplects() > 0 {
		for _, name := range cs.KeysComplects() {
			if complect, ok := cs.HasFullComplects(name); ok {
				lastComplect = complect
				break
			}
		}
	}
	return lastComplect
}

func (cs Complects) KeysComplects() []string {
	var keys []string
	for k := range cs {
		keys = append(keys, k)
	}
	sort.Sort(sort.Reverse(sort.StringSlice(keys)))
	return keys
}

func (cs Complects) LenComplects() int {
	return len(cs)
}

func (cs *Complects) StartMonitoringMdsBackups() {
	ticker := time.NewTicker(10 * time.Second)
	defer ticker.Stop()

	for range ticker.C {
		if err := cs.CreateComplects(); err != nil {
			fmt.Printf("error update mds backup list: %s", err)
		}
	}
}
