package support

import (
	"fmt"
	"io/ioutil"
	"log/syslog"
	"os"
	"strings"

	"gopkg.in/yaml.v2"

	"a.yandex-team.ru/direct/infra/dt-db-manager/pkg/zookeeper"
)

const (
	CONFIG      = "/etc/yandex/dt-db-manager/config.yaml"
	LISTENPORT  = 8080
	DEFZKPATH   = "/dt-db-manager"
	DEFZKSERVER = "ppcdiscord.yandex.ru,ppclogshutter01f.ppc.yandex.ru"
	LISTENHOST  = "localhost"
	MYSQLTOKEN  = ""
	MYSQLUSER   = "default"
)

var logger *syslog.Writer
var err error
var f = fmt.Sprintf

func (yr YtReplicator) SetError(err error) {
	*yr.Error = err
}

func (yrs YtReplicators) GetBinlogWriterHosts(name GroupName) Servers {
	if bw, ok := yrs[name]; ok {
		return bw.WriterHosts
	}
	return Servers{}
}

func (yrs YtReplicators) GetBinlogWriterCluster(name GroupName) string {
	if bw, ok := yrs[name]; ok {
		return bw.WriterYTCluster
	}
	return ""
}

func (yrs YtReplicators) GetBinlogWriterDir(name GroupName) string {
	if bw, ok := yrs[name]; ok {
		return bw.WriterYTDir
	}
	return ""
}

func (yrs YtReplicators) GetYTAccount(name GroupName) string {
	if bw, ok := yrs[name]; ok {
		return bw.Account
	}
	return ""
}
func (yrs YtReplicators) GetYTTokenFile(name GroupName) string {
	if bw, ok := yrs[name]; ok {
		return bw.YTTokenFile
	}
	return ""
}

func init() {
	priority := syslog.LOG_LOCAL3 | syslog.LOG_DEBUG
	if logger, err = syslog.New(priority, "dt-db-manager"); err != nil {
		fmt.Printf("error start syslog: %s", err)
	}

}

func ParseConfig(configPath string) Config {
	var data []byte
	if len(configPath) == 0 {
		Wrap(logger.Debug(f("use default config %s\n", configPath)))
		configPath = CONFIG
	}
	if data, err = ioutil.ReadFile(configPath); err != nil {
		Wrap(logger.Crit(f("error read file %s: %s\n", configPath, err)))
		os.Exit(2)
	}
	config := Config{
		ZookeeperServers: strings.Split(DEFZKSERVER, ","),
		ZookeeperPath:    DEFZKPATH,
		ListenPort:       LISTENPORT,
		ListenHost:       LISTENHOST,
		Logger:           logger,
		Replicators:      make(YtReplicators),
		TeleportHosts:    Servers{},
		MysqlTokenFile:   MYSQLTOKEN,
		MysqlUser:        MYSQLUSER,
	}

	if err := yaml.Unmarshal(data, &config); err != nil {
		Wrap(logger.Crit(f("error parse config %s: %s\n", configPath, err)))
		os.Exit(2)
	}

	Wrap(logger.Debug(f("load config %s: %+v", configPath, config)))
	return config
}

func (c Config) ZkPathList(zc *zookeeper.ZkConnect, prefix string) map[string][]string {
	result := make(map[string][]string)
	for _, group := range zc.ListPath(c.ZookeeperPath) {
		if !strings.Contains(group, prefix) {
			continue
		}
		path := fmt.Sprintf("%s/%s", c.ZookeeperPath, group)
		for _, node := range zc.ListPath(path) {
			result[group] = append(result[group], fmt.Sprintf("%s/%s", path, node))
		}
	}
	return result
}

func (c Config) ZkPathGroupGenerate(group string) string {
	return fmt.Sprintf("%s/db_%s/servers", c.ZookeeperPath, group)
}

func (c Config) ZkPathStatusGenerate(group string) string {
	return fmt.Sprintf("%s/db_%s/status", c.ZookeeperPath, group)
}

func (c Config) LoadReplicatorConf(name interface{}) (replicator YtReplicator) {
	group := FormatGroup(name)
	var err error
	var ok bool
	replicator, ok = c.Replicators[group.GetGroupName()]
	replicator.Error = &err
	if !ok {
		msg := fmt.Errorf("not found group %s for yt replicators %+v", group.GetGroupName(), c.Replicators)
		replicator.SetError(msg)
	}
	return
}

type Servers []string
