package support

import (
	"encoding/json"
	"errors"
	"fmt"
	"sort"
	"strings"
	"sync"
	"time"

	rr "a.yandex-team.ru/direct/infra/dt-db-manager/pkg/sshrequest"
)

type Host struct {
	*Groups  `json:"groups"`
	Hostname string `json:"hostname"`
	*Resources
	Error    *error      `json:"-"`
	Lock     *sync.Mutex `json:"-"`
	TaskList TasksStatus
}

func (h Host) GetHost() string {
	return h.Hostname
}

func (h *Host) GetResources() Resources {
	if h.Resources == nil {
		return Resources{}
	}
	return *h.Resources
}

func (h *Host) GetGroups() Groups {
	if h.Groups == nil {
		return Groups{}
	}
	return *h.Groups
}

func (h *Host) GetError() error {
	if h.Error == nil {
		return nil
	}
	return *h.Error
}

func (h *Host) SetResource(resource Resources) {
	*h.Resources = resource
}

func NewHost(hostname string, groupname interface{}) Host {
	var groups Groups
	if groupname == nil {
		groups = Groups{}
	} else {
		groups = Groups{FormatGroup(groupname)}
	}
	err := errors.New("")

	return Host{
		Groups:    &groups,
		Hostname:  hostname,
		Resources: &Resources{},
		Error:     &err,
		Lock:      &sync.Mutex{},
		TaskList:  NewTasksStatus(),
	}
}

type Byte int

//общий и занимаемы объем диска
type StorageSpace struct {
	TotalSpace Byte `json:"total_size"`
	FreeSpace  Byte `json:"free_size"`
}

func (ss StorageSpace) GetFreeSpace() Byte {
	return ss.FreeSpace
}

func (b Byte) SizeHuman() string {
	const unit = 1024
	if b < unit {
		return fmt.Sprintf("%dB", b)
	}
	div, exp := int64(unit), 0
	for n := b / unit; n > unit; n /= unit {
		div *= unit
		exp++
	}
	return fmt.Sprintf("%.2f%cB", float64(b)/float64(div), "KMGTPE"[exp])
}

type DBVersion float32

func (db DBVersion) Human() string {
	createTime := time.Unix(int64(db), 0)
	location, _ := time.LoadLocation("Europe/Moscow")
	return createTime.In(location).String()
}

type MysqlBinlog struct {
	Position string `json:"position"`
	Name     string `json:"name"`
	Gtid     string `json:"gtid"`
}

//статистика ресурсов, занимаемых на машине, и версия БД
type MysqlInstance struct {
	DatabaseSize     Byte        `json:"size_database"`
	VersionDatabase  DBVersion   `json:"version_database"`
	UsageMemory      Byte        `json:"usage_memory"`
	InstanceName     string      `json:"name"`
	DatabasePort     int         `json:"port"`
	InstanceRunning  string      `json:"running"`
	XtrabackupBinlog MysqlBinlog `json:"xtrabackup_binlog"`
}

func (mi MysqlInstance) TypeInstance() string {
	name := strings.Split(mi.InstanceName, ".")
	return name[0]
}

func (mi MysqlInstance) VersionInstance() DBVersion {
	return mi.VersionDatabase
}

func (mi MysqlInstance) XtrabackupGtid() string {
	return mi.XtrabackupBinlog.Gtid
}

func (mi MysqlInstance) RunningInstance() string {
	status := strings.ToLower(mi.InstanceRunning)
	if !strings.Contains(status, "yes") &&
		!strings.Contains(status, "no") {
		return "unknown"
	}
	return status
}

type MysqlInstances []MysqlInstance

func (mis MysqlInstances) Len() int      { return len(mis) }
func (mis MysqlInstances) Swap(i, j int) { mis[i], mis[j] = mis[j], mis[i] }
func (mis MysqlInstances) Less(i, j int) bool {
	return mis[1].InstanceName < mis[j].InstanceName
}

func (mis MysqlInstances) HasInstance(name string) bool {
	for _, instance := range mis {
		if instance.InstanceName == name {
			return true
		}
	}
	return false
}

//список установленных пакетов mysql
type MysqlPackages []string

//место на диске, статистика по установленным БД
type Resources struct {
	StorageSpace   `json:"server_space"`
	MysqlInstances map[string]MysqlInstance `json:"instances"`
	MysqlPackages  `json:"installed_dbconfigs"`
}

func (r *Resources) RecheckInstanceName() {
	for name, instance := range (*r).MysqlInstances {
		if len(instance.InstanceName) == 0 {
			instance.InstanceName = name
		}
		(*r).MysqlInstances[name] = instance
	}
}

func (h *Host) UpdateResources() bool {
	command := "sudo /usr/local/bin/get-resources-host.py"
	request := rr.NewRemouteRequest(h.GetHost(), command)

	var data []byte
	var err error
	if data, err = rr.RemoteExecute(request); err != nil {
		Wrap(logger.Crit(f("error execute request %+v: %s", request, err)))
		*h.Error = err
		return false
	}

	var resources Resources
	if err = json.Unmarshal(data, &resources); err != nil {
		Wrap(logger.Crit(f("error unmarshal response: %s for request %+v: %s \n", data, request, err)))
		*h.Error = err
		return false
	}

	resources.RecheckInstanceName()
	h.SetResource(resources)
	return true
}

func (h Host) ActiveInstaces() MysqlInstances {
	var activeInstances MysqlInstances
	for name, instance := range h.MysqlInstances {
		if strings.Contains(name, ".new") || strings.Contains(name, ".old") {
			continue
		}
		activeInstances = append(activeInstances, instance)
	}
	sort.Sort(activeInstances)
	return activeInstances
}

func (h Host) PrepareInstances() MysqlInstances {
	var prepareInstances MysqlInstances
	for name, instance := range h.MysqlInstances {
		if strings.Contains(name, ".new") {
			prepareInstances = append(prepareInstances, instance)
			continue
		}
	}
	sort.Sort(prepareInstances)
	return prepareInstances
}

type Hosts []Host

func NewHosts() *Hosts {
	return &Hosts{}
}

func (hs Hosts) LenHosts() int {
	return len(hs)
}

func (hs Hosts) HasHost(newHost Host) bool {
	for _, savedHost := range hs {
		if savedHost.GetHost() == newHost.GetHost() {
			return true
		}
	}
	return false
}

func (hs Hosts) MaxFreeSpaceHost() Host {
	var maxHost = NewHost("", "")
	for _, host := range hs {
		if host.GetFreeSpace() > maxHost.GetFreeSpace() {
			maxHost = host
		}
	}
	return maxHost
}

func (hs Hosts) PrintHosts() {
	groups := hs.AllGroups()
	if len(groups) == 0 {
		fmt.Println("не найдено ни одной группы с хостами")
		return
	}
	for _, group := range hs.AllGroups() {
		fmt.Printf("для группы %s найдены хосты:\n", group)
		for _, host := range *hs.HostsPerGroup(group) {
			fmt.Printf("\t%s со свободным местом %s \n", host.Hostname, host.GetFreeSpace().SizeHuman())
			for _, instance := range host.ActiveInstaces() {
				fmt.Printf("\t\tCURRENT %s:%d running:%s version:%s\n", instance.TypeInstance(),
					instance.DatabasePort, instance.RunningInstance(), instance.VersionInstance().Human())
			}
			for _, instance := range host.PrepareInstances() {
				fmt.Printf("\t\tPREPARE %s running:%s version:%s\n", instance.TypeInstance(),
					instance.RunningInstance(), instance.VersionInstance().Human())
			}
		}
	}
}

//Удаление хоста
func (hs *Hosts) RemoveHost(removedHosts Hosts) {
L:
	for {
		for i, host := range *hs {
			if removedHosts.HasHost(host) {
				(*hs)[i] = (*hs)[len(*hs)-1]
				*hs = (*hs)[:len(*hs)-1]
				continue L
			}
		}
		break L
	}
	Wrap(logger.Err(f("[monitor/RemoveHost] remove host %v", *hs)))
}

func (hs *Hosts) UpdateHosts(newHosts Hosts) {
	for _, newHost := range newHosts {
		if ok := hs.HasHost(newHost); !ok {
			startHost := NewHost(newHost.Hostname, newHost.GetGroups())
			Wrap(logger.Info(f("[monitor/UpdateHostshosts] update group %v, %s", newHost.GetGroups(), newHost.Hostname)))
			*hs = append(*hs, startHost)
			continue
		}

		finded, _ := hs.FindHost(newHost)
		for _, newGroup := range newHost.GetGroups() {
			if finded.HasGroup(newGroup) {
				continue
			}
			*finded.Groups = append(*finded.Groups, newGroup)
			Wrap(logger.Info(f("[monitor/UpdateHostshosts] add group %s to %+v", newGroup, *finded.Groups)))
		}

	}
	Wrap(logger.Debug(f("[monitor/UpdateHostshosts] ALL GROUPS3", hs)))
}

func (hs *Hosts) GetResources() map[string]*Resources {
	var result = make(map[string]*Resources)
	for _, host := range *hs {
		result[host.Hostname] = host.Resources
	}
	return result
}

type GtidMysqlBinlog map[string]MysqlBinlog

func (hs *Hosts) GtidPerInstance() (result GtidMysqlBinlog) {
	result = make(GtidMysqlBinlog)
	for _, host := range *hs {
		for _, instance := range host.ActiveInstaces() {
			result[instance.InstanceName] = instance.XtrabackupBinlog
		}
	}
	return
}

func (gmb GtidMysqlBinlog) ShortName() (result GtidMysqlBinlog) {
	result = make(GtidMysqlBinlog)
	for name, value := range gmb {
		name = strings.ReplaceAll(name, "ppcdata", "ppc:")
		name = strings.ReplaceAll(name, "ppcmonitor", "monitor")
		result[name] = value
	}
	return
}

func (gmb GtidMysqlBinlog) Print() {
	for instance, binlog := range gmb.ShortName() {
		fmt.Printf("(%s): %s\n", instance, binlog.Gtid)
	}
}

func (g Group) ToString() string {
	return string(g)
}

func (g Group) GenerateLocalhost() string {
	return fmt.Sprintf("localhost_%s", g.GetGroupName())
}

func (g Group) GetGroupName() GroupName {
	if strings.Contains(g.ToString(), "@") {
		return GroupName(strings.Split(string(g), "@")[0])
	}
	return GroupName(g)
}

type ReplicaName string

func (g Group) GetReplica() ReplicaName {
	if strings.Contains(g.ToString(), "@") {
		return ReplicaName(strings.Split(g.ToString(), "@")[1])
	}
	return ReplicaName("")
}

func (gs Groups) ContainGroup(newGroup Group) bool {
	for _, group := range gs {
		if group == newGroup {
			return true
		}
	}
	return false
}

//true если имя группы и реплики совпадает
func (h Host) HasGroup(group Group) bool {
	for _, savedGroup := range *h.Groups {
		if strings.Contains(savedGroup.ToString(), group.ToString()) {
			return true
		}
	}
	return false
}

//true если совпадает имя группы текущей и сравниваемой
func (h Host) HasGroupName(group Group) bool {
	for _, savedGroup := range *h.Groups {
		if savedGroup.GetGroupName() == group.GetGroupName() {
			return true
		}
	}
	return false
}

func (hs *Hosts) ReplicsPerGroupName(value interface{}) (result []Group) {
	var group Group
	switch v := value.(type) {
	case Group:
		group = v
	case GroupName:
		group = Group(v)
	default:
		group = Group("")
	}
	tmp := make(map[Group]int)
	for _, host := range *hs {
		if host.HasGroupName(group) {
			for _, g := range *host.Groups {
				if _, ok := tmp[g]; !ok {
					tmp[g] = 1
				} else {
					tmp[g] += 1
				}
			}
		}
	}
	for replica := range tmp {
		result = append(result, replica)
	}
	return
}

//получить список ссылок хостов на указанное имя группы и реплики
func (hs *Hosts) FindGroupHosts(searched interface{}) *Hosts {
	groupName := FormatGroup(searched)
	var result Hosts
	group := Group(groupName)
	for _, host := range *hs {
		if len(group.GetReplica()) > 0 {
			if host.HasGroup(group) {
				result = append(result, host)
			}
		} else {
			if host.HasGroupName(group) {
				result = append(result, host)
			}
		}
	}
	return &result
}

//получить список ссылок хостов только на указанное имя группы
func (hs *Hosts) FindGroupNameHosts(searched interface{}) *Hosts {
	groupName := FormatGroup(searched)
	var result Hosts
	group := Group(groupName)
	for _, host := range *hs {
		if host.HasGroupName(group) {
			result = append(result, host)
		}
	}
	return &result
}

//находит хост по имени и возвращает на него ссылку *Host
func (hs *Hosts) FindHost(newHost interface{}) (*Host, bool) {
	switch v := newHost.(type) {
	case Host:
		for _, host := range *hs {
			if host.GetHost() == v.GetHost() {
				return &host, true
			}
		}
	case string:
		for _, host := range *hs {
			if host.GetHost() == v {
				return &host, true
			}
		}
	}
	return &Host{}, false
}

func (hs *Hosts) StartMonitoringResources() {
	ticker := time.NewTicker(5 * time.Second)
	defer ticker.Stop()

	for range ticker.C {
		for _, host := range *hs {
			Wrap(logger.Debug(f("START MONITOR %s\n%+v", host.GetHost(), host.GetResources())))
			host.UpdateResources()
		}
	}
}

func (hs *Hosts) AllGroups() Groups {
	var result Groups
	for _, host := range *hs {
		Wrap(logger.Debug(f("ALL GROUPS", host, host.GetGroups())))
		for _, group := range host.GetGroups() {
			if ok := result.ContainGroup(group); !ok {
				result = append(result, group)
			}
		}
	}
	return result
}

func (hs *Hosts) HostsPerGroup(searchGroup Group) *Hosts {
	var result = &Hosts{}
	for _, host := range *hs {
		for _, group := range host.GetGroups() {
			if searchGroup == group {
				*result = append(*result, host)
			}
		}
	}
	return result
}
