package support

import (
	"bytes"
	"crypto/sha1"
	"encoding/hex"
	"encoding/json"
	"fmt"
	"math/rand"
	"os/exec"
	"reflect"
	"strings"
	"time"

	rr "a.yandex-team.ru/direct/infra/dt-db-manager/pkg/sshrequest"
)

type State struct {
	Status string `json:"status"`
	Errors string `json:"message"`
}

func NewState(status, msg string) State {
	return State{status, msg}
}

func (s State) GetState() string {
	if len(s.Status) > 0 {
		return s.Status
	} else if len(s.Errors) > 0 {
		return "FAILED"
	}
	return "NEW"
}

type TaskStatus struct {
	TaskID    string `json:"taskid"`
	Message   string `json:"msg"`
	HashID    string `json:"hashid"`
	Code      int    `json:"code"`
	StageName string `json:"stage-name"`
	StartTime int64  `json:"start-time"`
	State
	rr.RemoteRequest
}

func (ts TaskStatus) GetTaskStatus() string {
	if ts.Code == 0 {
		return ts.GetState()
	}
	return "FAILED"
}

func (ts TaskStatus) GetTaskMessage() string {
	if ts.Code == 0 {
		return ts.Errors
	}
	return ts.Message
}

func (ts TaskStatus) GetStartTime() time.Time {
	var start int64
	v := reflect.ValueOf(ts)
	if v.FieldByName("StartTime").IsValid() {
		start = ts.StartTime
	}
	return time.Unix(start, 0)
}

func (ts TaskStatus) GetHashID() string {
	return ts.HashID
}

func (ts *TaskStatus) UpdateTaskState() bool {
	if ts.GetState() == "SUCCESS" || ts.GetState() == "OK" {
		return true
	}

	var command string
	if strings.HasPrefix(ts.GetHashID(), "/") {
		command = fmt.Sprintf("/bin/cat %s", ts.GetHashID())
	} else {
		command = fmt.Sprintf("/usr/local/bin/direct-async-run -j get-status %s", ts.GetHashID())
	}
	request := rr.RemoteRequest{
		Hostname: ts.GetHost(),
		Command:  command,
	}

	var line []byte
	var err error
	var state State

	if line, err = rr.RemoteExecute(request); err != nil {
		msg := fmt.Sprintf("error execute %s: %s", request.GetCmd(), err)
		state = State{Status: "FAILED", Errors: msg}
	} else {
		if err := json.Unmarshal(line, &state); err != nil {
			msg := fmt.Sprintf("error unmarshal %s: %s", line, err)
			state = State{Status: "FAILED", Errors: msg}
		}
	}

	if time.Since(ts.GetStartTime()) > (time.Duration(10)*time.Minute) &&
		strings.Contains(state.Status, "NEW") {
		state = State{Status: "FAILED", Errors: "timeout state 'NEW'"}
	}

	Wrap(logger.Info(f("[sheduler/UpdateTaskState] hostname: %s, "+
		"stage: %s, hashid: %s, state: %+v", ts.Hostname, ts.StageName, ts.GetHashID(), ts.State)))

	ts.State = state
	return err == nil
}

type TasksStatus map[string]*TaskStatus

func (tss TasksStatus) TasksStageAllStatus(check, stage string) bool {
	tasks := tss.GetStageTasks(stage)
	if len(tasks) == 0 {
		return false
	}
	for _, task := range tasks {
		Wrap(logger.Debug(f("TASK %s STATUS %s", task, task.GetTaskStatus())))
		if task.GetTaskStatus() != check {
			return false
		}
	}
	return true
}

func (tss TasksStatus) TasksStageAnyStatus(check, stage string) bool {
	tasks := tss.GetStageTasks(stage)
	if len(tasks) == 0 {
		return false
	}
	for _, task := range tasks {
		Wrap(logger.Debug(f("TASK %s STATUS %s", task, task.GetTaskStatus())))
		if task.GetTaskStatus() == check {
			return true
		}
	}
	return false
}

func (tss *TasksStatus) UpdateTasksStatus() {
	for _, task := range *tss {
		task.UpdateTaskState()
		Wrap(logger.Debug(f("UPDATE TASK %+v", task)))
	}
}

func (tss TasksStatus) FailedTaskStatus() TasksStatus {
	failedTasks := make(map[string]*TaskStatus)
	for hashID, task := range tss {
		if task.GetTaskStatus() == "FAILED" {
			failedTasks[hashID] = task
		}
	}
	return failedTasks
}

func (tss *TasksStatus) AddTask(task *TaskStatus) {
	(*tss)[task.TaskID] = task
}

func (tss *TasksStatus) RemoveAllTasks() {
	var indexDeleted []string
	for taskID := range *tss {
		indexDeleted = append(indexDeleted, taskID)
	}
	for _, index := range indexDeleted {
		delete(*tss, index)
	}
}

func (tss *TasksStatus) RemoveTasks(hostname string, stage StageInterface) {
	var indexDeleted []string
	for taskID, task := range *tss {
		if task.Hostname == hostname && task.StageName == stage.GetCurrentStage() {
			indexDeleted = append(indexDeleted, taskID)
		}
	}
	for _, index := range indexDeleted {
		delete(*tss, index)
	}
}

func (tss *TasksStatus) RemoveTask(hash string) {
	var indexDeleted []string
	for taskID := range *tss {
		if taskID == hash {
			indexDeleted = append(indexDeleted, taskID)
		}
	}
	for _, index := range indexDeleted {
		delete(*tss, index)
	}
}

func (tss *TasksStatus) LenTasks() int {
	return len(*tss)
}

func HasKey(list []string, finded string) bool {
	for _, value := range list {
		if finded == value {
			return true
		}
	}
	return false
}

func (tss *TasksStatus) GetAllStages() (stages []string) {
	for _, task := range *tss {
		if HasKey(stages, task.StageName) {
			continue
		}
		stages = append(stages, task.StageName)
	}
	return
}

func (tss *TasksStatus) GetStageTasks(stage string) TasksStatus {
	tasks := make(TasksStatus)
	for name, task := range *tss {
		if task.StageName != stage {
			continue
		}
		tasks[name] = task
	}
	return tasks
}

func (tss TasksStatus) GetListTasks(stage string) ListsTaskStatus {
	var list ListsTaskStatus
	for _, task := range tss {
		if task.StageName != stage {
			continue
		}
		list = append(list, *task)
	}
	return list
}

func NewTasksStatus() TasksStatus {
	return make(map[string]*TaskStatus)
}

func LocalExecute(command string) ([]byte, error) {
	cmd := strings.Split(command, " ")
	return exec.Command(cmd[0], cmd[1:]...).Output()
}

func TaskExecute(request rr.RemoteRequest, stageName string) TaskStatus {
	var output []byte
	var err error

	taskid := GetTaskID()
	if output, err = rr.RemoteExecute(request); err != nil {
		msg := fmt.Sprintf("error execute %s: %s", request.GetCmd(), err)
		return TaskStatus{TaskID: taskid,
			Message:       msg,
			Code:          2,
			StageName:     stageName,
			RemoteRequest: request,
			StartTime:     time.Now().Unix()}
	}

	var execStatus TaskStatus
	execStatus.TaskID = GetTaskID()
	execStatus.StageName = stageName
	lines := bytes.Split(output, []byte("\n"))
	for _, line := range lines {
		if bytes.HasPrefix(line, []byte("{")) {
			err = json.Unmarshal(line, &execStatus)
			if err != nil {
				execStatus.Message = fmt.Sprintf("error unmarshal %s: %s", line, err)
				execStatus.Code = 3
				execStatus.RemoteRequest = request
				return execStatus
			} else {
				execStatus.RemoteRequest = request
			}
			if ok := execStatus.UpdateTaskState(); !ok {
				Wrap(logger.Warning(f("update task state %+v\n", execStatus.State)))
				execStatus.RemoteRequest = request
			}
			return execStatus
		}
	}
	execStatus.Message = fmt.Sprintf("not found json in output: %s", output)
	execStatus.Code = 404
	execStatus.RemoteRequest = request
	return execStatus
}

type SimpleStatus struct {
	Status string `json:"status"`
}

func GetStatusDone() string {
	result := SimpleStatus{Status: "DONE"}
	output, _ := json.Marshal(result)
	return string(output)
}

func GetStatusFailed() string {
	result := SimpleStatus{Status: "FAILED"}
	output, _ := json.Marshal(result)
	return string(output)
}

func GetTaskID() string {
	line := fmt.Sprintf("%d_%d", time.Now().Unix(), rand.Int63())
	id := sha1.New()
	_, _ = id.Write([]byte(line))
	return hex.EncodeToString(id.Sum(nil))
}

type ListsTaskStatus []TaskStatus

func (lts ListsTaskStatus) Len() int           { return len(lts) }
func (lts ListsTaskStatus) Swap(i, j int)      { lts[i], lts[j] = lts[j], lts[i] }
func (lts ListsTaskStatus) Less(i, j int) bool { return lts[i].TaskID < lts[j].TaskID }

type StageInterface interface {
	GetCurrentStage() string
	GetLastStage() string
}
