package support

import (
	"bytes"
	"fmt"
	"io/ioutil"
	"os"
	"path/filepath"
	"strings"

	y "a.yandex-team.ru/direct/infra/dt-db-manager/pkg/yttransfer"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yt"
)

type YtConnector struct {
	YtReplicator
	SourceClient      y.YtConnect `json:"-"`
	DestinationClient y.YtConnect `json:"-"`
}

func NewGroupYtReplicator(r YtReplicator) (ytConnector YtConnector, err error) {
	sourceClient, err := y.NewYtConnect(r.SourceCluster, r.Account, r.YTTokenFile)
	if err != nil {
		err = fmt.Errorf("error connect %s: %s", r.SourceCluster, err)
		return
	}
	destinationClient, err := y.NewYtConnect(r.DestinationCluster, r.Account, r.YTTokenFile)
	if err != nil {
		err = fmt.Errorf("error connect %s: %s", r.DestinationCluster, err)
		return
	}
	ytConnector = YtConnector{
		YtReplicator:      r,
		SourceClient:      sourceClient,
		DestinationClient: destinationClient,
	}
	return
}

func (yc YtConnector) GetDestinationCluster() string {
	return yc.DestinationCluster
}

func (yc YtConnector) GetSourceCluster() string {
	return yc.SourceCluster
}

func (yc YtConnector) GetDestinationDirectory() string {
	return strings.TrimRight(yc.DestinationDir, "/")
}

func (yc YtConnector) GetSourceDirectory() string {
	return strings.TrimRight(yc.SourceDir, "/")
}

func (yc YtConnector) NewTransferYtBackup() (backup y.YtTransferTables, err error) {
	backups, err := yc.ListYtSourceDirectory()
	if err != nil {
		err = fmt.Errorf("error get backups %s: %s", yc.SourceDir, err)
		return
	}
	backup = backups.GoodBackup()

	fmt.Printf("good backups %+v\n", backup)

	var backupAttr y.ResourceDisk
	for _, table := range backup {
		backupAttr, err = yc.SourceClient.ResourceUsage(table.SourceNodePath())
		if err != nil {
			fmt.Printf("error resource usage for %s: %s", table.SourceNodePath(), err)
		}
		fmt.Printf("yt backup size %s: %+v\n", table, backupAttr)
	}

	restoreFree, err := yc.DestinationClient.ResourceLimit()
	if err != nil {
		err = fmt.Errorf("error resource limit for %s: %s", yc.Account, err)
		return
	}
	fmt.Printf("restore size %s: %+v\n", yc.Account, restoreFree)

	if backupAttr.DiskSpace > restoreFree.DiskSpace {
		err = fmt.Errorf("для текущего копировнаия не хватает места. BackupSize: %d, FreeSpace: %d(%s)",
			backupAttr.DiskSpace, restoreFree.DiskSpace, yc.DestinationCluster)
		return
	}
	return
}

func (yc YtConnector) RunTransferYtBackups(tts y.YtTransferTables) (tasks y.YtTransferTasks, err error) {
	client := yc.DestinationClient
	//предварительно удаляем данные на yt приемнике, если они есть
	for _, table := range tts.GoodBackup() {
		tablePath := table.DestinationNodePath()
		if ok, _ := client.NodeYtExists(tablePath); ok {
			fmt.Printf("[RunTransferYtBackups/remove] %s\n", tablePath)
			err = client.RemoveYtNode(tablePath, false, false)
			if err != nil {
				return
			}
		}
	}
	if token, err := ioutil.ReadFile(yc.YTTokenFile); err == nil {
		_ = os.Setenv("YT_TOKEN", string(token))
	}
	tasks = tts.StartTransferTables()
	//прерываем скрипт если есть незапущенные таски
	for _, task := range tasks {
		err = *task.LastError
		fmt.Printf("[RunTransferYtBackups] start task %v error %s\n", task, err.Error())
		_, ok := task.GetTaskID()
		if len(err.Error()) != 0 && !ok {
			tasks.AbortTasks()
			return
		}
	}
	err = tasks.MonitorTransferTasks()
	return
}

func (yc YtConnector) RunConvertTables(tts y.YtTransferTables) (err error) {
	//convert static tables
	var convertTables []string
	client := yc.DestinationClient
	for _, tt := range tts {
		if strings.Contains(tt.DestinationNodePath(), "_static") {
			convertTables = append(convertTables, tt.DestinationNodePath())
		}
	}

	for _, table := range convertTables {
		Wrap(logger.Info(f("dynamic convert yt %s", table)))
	}

	convertsStatus := client.StartConvertStaticTable(convertTables)

	if failed := convertsStatus.ConvertsFailed(); len(failed) > 0 {
		return fmt.Errorf("found failed converts yt %v", failed)
	}

	if err := client.UpdateSyncVersion(convertsStatus); err != nil {
		return fmt.Errorf("error update sync version: %s", err)
	}

	if err := client.RemoveConvertedTables(convertsStatus); err != nil {
		return fmt.Errorf("error remove converted yt tables: %s", err)
	}
	return
}

func (yc YtConnector) RunConvertDynamicTables(tts y.YtTransferTables) (err error) {
	var convertTables []string
	client := yc.SourceClient
	for _, table := range tts {
		var dynamic bool
		if err = client.GetAttribute(table.SourceNodePath(), "dynamic", &dynamic); err != nil && dynamic {
			convertTables = append(convertTables, table.SourceNodePath())
			Wrap(logger.Info(f("static convert yt %s", table)))
		}
	}

	convertsStatus := client.StartConvertDynamicTable(convertTables)
	if failed := convertsStatus.ConvertsFailed(); len(failed) > 0 {
		return fmt.Errorf("found failed converts yt %v", failed)
	}

	if failed := convertsStatus.ConvertsFailed(); len(failed) > 0 {
		return fmt.Errorf("found failed converts yt %v", failed)
	}
	return
}

func (yc YtConnector) UpdateYtMysqlSyncStates(gtids GtidMysqlBinlog, tt y.YtTransferTables) (err error) {
	client := yc.DestinationClient
	fmt.Printf("[UpdateYtMysqlSyncStates] tables %+v\n", tt)
	path, ok := tt.FindTransferTable("mysql-sync-states_static")
	if !ok {
		return fmt.Errorf("not found table 'mysql-sync-states_static'")
	}
	oldpath := path.DestinationNodePath()
	r, err := client.Client.ReadTable(client.Cntx, ypath.Path(oldpath), nil)
	if err != nil {
		return fmt.Errorf("error read table %s: %s", oldpath, err)
	}
	defer func() { _ = r.Close() }()

	var mysqlSyncStat MysqlSyncDatas

	var errmsg bytes.Buffer

	for r.Next() {
		var line MysqlSyncData
		err = r.Scan(&line)
		if err != nil {
			errmsg.WriteString(err.Error())
		}
		mysqlSyncStat = append(mysqlSyncStat, &line)
	}

	if r.Err() != nil {
		return fmt.Errorf("error read table %s: %s", oldpath, r.Err())
	}

	if errmsg.Len() != 0 {
		return fmt.Errorf(errmsg.String())
	}

	for alias, value := range gtids.ShortName() {
		mysqlSyncStat.UpdateGtidPerDbname(alias, value.Gtid)
	}

	ok, err = client.Client.NodeExists(client.Cntx, ypath.Path(oldpath), nil)
	if err != nil {
		return fmt.Errorf("error node exists %s: %s", oldpath, err)
	}

	if !ok {
		_, err := yt.CreateTable(client.Cntx, client.Client, ypath.Path(oldpath), yt.WithInferredSchema(&MysqlSyncStat{}))
		if err != nil {
			return fmt.Errorf("error create table %s: %s", oldpath, err)
		}
	}

	w, err := client.Client.WriteTable(client.Cntx, ypath.Path(oldpath), nil)
	if err != nil {
		return fmt.Errorf("error write table %s: %s", oldpath, err)
	}

	for _, line := range mysqlSyncStat {
		if err := w.Write(&line); err != nil {
			errmsg.WriteString(err.Error())
		}
	}
	if err := w.Commit(); err != nil {
		return err
	}
	return
}

func (yc YtConnector) ChangeYtCurrentLink(tt y.YtTransferTables) (err error) {
	client := yc.DestinationClient
	ytdir := yc.GetDestinationDirectory()
	target := f("%s/%s", ytdir, tt.Name())
	link := f("%s/%s", ytdir, "current")
	return client.UpdateYtLink(link, target)
}

//Удаляет все старые привезенные базы в указанной дирректории. Новой базой считается слинкованная с current.
func (yc YtConnector) RemoveOldTables() (err error) {
	var removed []string
	client := yc.DestinationClient
	folders, err := client.ListYtNode(yc.GetDestinationDirectory())
	if err != nil {
		return fmt.Errorf("[RemoveOldTables/ListYtNode] %s: %s", yc.DestinationDir, err)
	}
	var excludeDir string
	for _, name := range folders {
		if strings.Contains(name, "current") {
			var target string
			path := fmt.Sprintf("%s/%s", yc.GetDestinationDirectory(), name)
			err = client.GetLinkAttribute(path, "target_path", &target)
			if err != nil {
				return err
			}
			excludeDir = filepath.Base(target)
			if len(excludeDir) == 0 {
				/*если параметр в target link пустой, то это говорит о битой ссылке.
				Надо руками посмотреть что не так в YT: либо обновить линку, либо удалить ее.
				*/
				return fmt.Errorf("target link empty: %s --> %s", path, excludeDir)
			}
			break
		}
	}
	for _, name := range folders {
		if strings.Contains(name, "current") ||
			strings.Contains(name, "v.") {
			continue
		}
		if strings.Contains(name, "20") &&
			!strings.Contains(name, excludeDir) {
			r := fmt.Sprintf("%s/%s", yc.GetDestinationDirectory(), name)
			removed = append(removed, r)
		}
	}
	for _, name := range removed {
		fmt.Printf("[RemoveOldTables] remove old directory: %v\n", name)
		if err = client.RemoveYtNode(name, true, true); err != nil {
			return fmt.Errorf("RemoveOldTables/RemoveYtNode] %s: %s", name, err)
		}
	}
	return nil
}

//Принимает дирректорию для бекапа и возвращает список со стуктурой вида {YtConnect, YtDir, YtPath}.
func (yc YtConnector) ListYtSourceDirectory() (backups y.YtTransferTables, err error) {
	client := yc.SourceClient
	dirs := strings.TrimRight(yc.SourceDir, "/")
	var errmsg bytes.Buffer
	if ok, _ := client.NodeHasType(dirs, "map_node"); !ok {
		err = fmt.Errorf("type node: not map_node")
		return
	}
	listnode, err := client.ListYtNode(dirs) //получаем сначала только первый
	if err != nil {
		err = fmt.Errorf("list node: %s", err)
		return
	}
	for _, name := range listnode {
		if strings.Contains(name, "mysql-sync-states_static") {
			listnode = []string{filepath.Base(dirs)}
			dirs = "/" + filepath.Dir(dirs)
		}
	}

	for _, name := range listnode {
		ytpath := fmt.Sprintf("%s/%s", dirs, name)
		if tables, err := client.RecurseListYtNode(ytpath); err == nil {
			for _, table := range tables {
				//FIX
				//if !strings.Contains(table, "mysql-sync-states_static") {
				//	continue
				//}
				ft := fmt.Sprintf("%s%s", ytpath, table)
				version, err := client.DatabaseVersion(ft)
				if err != nil {
					fmt.Printf("error get version %s: %s", ft, err)
				}
				tt := y.NewYtTransferTable(yc, name, table, version)
				backups = append(backups, tt)
			}
		} else {
			errmsg.WriteString(err.Error())
		}
	}
	if errmsg.Len() > 0 {
		err = fmt.Errorf(errmsg.String())
	}
	return
}

type MysqlSyncStat struct {
	MysqlSyncData
	ImportedTables string `yson:"imported_tables,omitempty"`
}

type MysqlSyncData struct {
	Dbname        string `yson:"dbname"`
	GtidSet       string `yson:"gtid_set"`
	ServerSchema  string `yson:"server_schema"`
	LastTimestamp int    `yson:"last_timestamp"`
	IsImported    bool   `yson:"is_imported"`
}

type MysqlSyncDatas []*MysqlSyncData

func (msd *MysqlSyncDatas) UpdateGtidPerDbname(dbname, value string) {
	for _, data := range *msd {
		mydata := *data
		if data.Dbname == dbname {
			mydata.GtidSet = value
			*data = mydata
		}
	}
}
