package yttransfer

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"log"
	"net/http"
	"os"
	"sort"
	"strings"
	"time"
)

const (
	YtTransferAPI      = "http://transfer-manager.yt.yandex.net/api/v1%s"
	MaxIdleConn        = 20
	RequestTimeout     = 60
	MaxRetryFailedTask = 2
)

var (
	httpClient *http.Client
	ytToken    string
	ff         = fmt.Sprintf
)

func init() {
	httpClient = CreateHTTPClient()
}

type YtMeta interface {
	GetSourceDirectory() string
	GetDestinationDirectory() string
	GetSourceCluster() string
	GetDestinationCluster() string
}

type YtTransferTable struct {
	YtMeta        `json:"-"`
	NameBackup    string
	Path          string
	VersionBackup ReplicatorVersion
}

type YtTransferTask struct {
	YtTransferTable `json:"transfer_table"`
	TaskID          string `json:"id"`
	State           string `json:"state"`
	CreateTime      string `json:"creation_time"`
	User            string `json:"user"`
	LastError       *error `json:"error"`
	Retry           *int   `json:"-"`
}

func NewYtTransferTask(ytt YtTransferTable) YtTransferTask {
	retry := 0
	err := fmt.Errorf("")
	return YtTransferTask{
		Retry:           &retry,
		LastError:       &err,
		YtTransferTable: ytt,
	}
}

func (ytt YtTransferTask) GetLastErrorString() string {
	return (*ytt.LastError).Error()
}

func (ytt YtTransferTask) HasLastError() bool {
	return len(ytt.GetLastErrorString()) > 0
}

func (ytt *YtTransferTask) SetLastError(msg interface{}) {
	switch v := msg.(type) {
	case string:
		*ytt.LastError = fmt.Errorf("%s", v)
	case error:
		*ytt.LastError = v
	}
}

func (ytt YtTransferTask) BodyDataForCopyTask() map[string]string {
	return map[string]string{
		"destination_cluster": ytt.GetDestinationCluster(),
		"source_cluster":      ytt.GetSourceCluster(),
		"destination_table":   ytt.DestinationNodePath(),
		"source_table":        ytt.SourceNodePath(),
	}
}

func (ytt YtTransferTask) TaskDone() bool {
	if len(ytt.State) > 0 && strings.Contains("completed", ytt.State) {
		return true
	}
	return false
}

func (ytt YtTransferTask) GetTaskID() (string, bool) {
	if len(ytt.TaskID) == 0 {
		return "", false
	}
	return ytt.TaskID, true
}

func (ytt YtTransferTask) TaskFailed() bool {
	if len(ytt.GetLastErrorString()) > 0 {
		return true
	}
	if len(ytt.State) > 0 && strings.Contains("failed", ytt.State) {
		return true
	}
	return false
}

func (ytt YtTransferTask) ToString() string {
	return ff("%v %s %s %s %s %v %d", ytt.YtTransferTable,
		ytt.TaskID, ytt.State, ytt.CreateTime, ytt.User, *ytt.LastError, *ytt.Retry)
}

func NewYtTransferTable(ym YtMeta, nameBackup, path string, version ReplicatorVersion) YtTransferTable {
	return YtTransferTable{
		YtMeta:        ym,
		NameBackup:    nameBackup,
		Path:          path,
		VersionBackup: version,
	}
}

func (tt YtTransferTable) SourceNodePath() string {
	if strings.Contains(tt.GetSourceDirectory(), tt.NameBackup) {
		return fmt.Sprintf("%s%s", tt.GetSourceDirectory(), tt.Path)
	}
	return fmt.Sprintf("%s/%s%s", tt.GetSourceDirectory(), tt.NameBackup, tt.Path)
}

func (tt YtTransferTable) DestinationNodePath() string {
	if strings.Contains(tt.GetDestinationDirectory(), tt.NameBackup) {
		return fmt.Sprintf("%s%s", tt.GetDestinationDirectory(), tt.Path)
	}
	return fmt.Sprintf("%s/%s%s", tt.GetDestinationDirectory(), tt.NameBackup, tt.Path)
}

func (tt YtTransferTable) BackupName() string {
	return tt.NameBackup
}

func (tt YtTransferTable) StartTransferTable() *YtTransferTask {
	task := NewYtTransferTask(tt)
	fmt.Println("START", task.BodyDataForCopyTask())
	body, err := json.Marshal(task.BodyDataForCopyTask())
	log.Printf("[StartTransferTable] run task %v\n", task.BodyDataForCopyTask())
	if err != nil {
		msg := ff("error marshal %v: %s", task, err)
		task.SetLastError(msg)
		return &task
	}
	address := ff(YtTransferAPI, "/tasks/")
	var data []byte
	for retry := 0; retry < 3; retry++ {
		data, err = runHTTP("POST", address, body)
		if err == nil || len(data) > 0 {
			break
		}
	}
	if err != nil {
		task.SetLastError(err)
		fmt.Printf("ERROR %s\n", err)
		return &task
	}
	if len(data) == 0 {
		msg := ff("empty taskid")
		task.SetLastError(msg)
		return &task
	}
	task.TaskID = ff("%s", data)
	log.Printf("[StartTransferTable] finish task %v with id %s\n", task.BodyDataForCopyTask(), task.TaskID)
	return &task
}

type YtTransferTasks []*YtTransferTask

func (tts *YtTransferTasks) UpdateTasksStatus() {
	for _, task := range *tts {
		id, ok := task.GetTaskID()
		if !ok {
			continue
		}
		if *task.Retry > MaxRetryFailedTask {
			continue
		}
		if task.TaskDone() {
			continue
		}
		address := ff(YtTransferAPI, ff("/tasks/%s/", id))
		data, err := runHTTP("GET", address, nil)
		if err != nil {
			task.SetLastError(err)
			continue
		}

		//fmt.Printf("%s", data) debug
		if err := json.Unmarshal(data, task); err != nil {
			task.SetLastError(ff("error unmarshal %s: %s", data, err))
		}
	}
}

func (tts *YtTransferTasks) GetTransferTables() (result YtTransferTables) {
	for _, task := range *tts {
		result = append(result, task.YtTransferTable)
	}
	return
}

func (tts *YtTransferTasks) GetTasksFailed() (tasks YtTransferTasks) {
	for _, status := range *tts {
		if status.TaskFailed() || status.HasLastError() {
			tasks = append(tasks, status)
		}
	}
	return
}

func (tts *YtTransferTasks) GetRealyFailedTasks() (tasks YtTransferTasks) {
	for _, task := range *tts {
		if task.TaskFailed() && (*task.Retry > MaxRetryFailedTask) {
			tasks = append(tasks, task)
		}
	}
	return
}

func (tts *YtTransferTasks) RestartFailedTask() {
	for _, task := range tts.GetRealyFailedTasks() {
		*task.Retry += 1
		id, ok := task.GetTaskID()
		fmt.Printf("[RestartFailedTask] failed task %s retry %d\n", id, *task.Retry)
		if !ok {
			continue
		}
		address := ff(YtTransferAPI, ff("/tasks/%s/restart/", id))
		if _, err := runHTTP("POST", address, nil); err != nil {
			task.SetLastError(err)
		}
	}
}

func (tts *YtTransferTasks) AbortTasks() {
	for _, task := range *tts {
		id, ok := task.GetTaskID()
		if !ok {
			continue
		}
		address := ff(YtTransferAPI, ff("/tasks/%s/abort/", id))
		if _, err := runHTTP("POST", address, nil); err != nil {
			task.SetLastError(err)
		}
	}
}

func (tts *YtTransferTasks) GetTasksNotDone() (tasks YtTransferTasks) {
	for _, status := range *tts {
		if status.TaskDone() {
			continue
		}
		tasks = append(tasks, status)
	}
	return
}

func (tts *YtTransferTasks) CheckAllTasksDone() bool {
	for _, status := range *tts {
		if !status.TaskDone() {
			return false
		}
	}
	return true
}

type YtTransferTables []YtTransferTable

func (ytts YtTransferTables) StartTransferTables() (tasks YtTransferTasks) {
	for _, table := range ytts {
		task := table.StartTransferTable()
		tasks = append(tasks, task)
	}
	return
}

func (ytts YtTransferTables) ShowTransferTables() (tasks YtTransferTasks) {
	for _, table := range ytts {
		newtask := NewYtTransferTask(table)
		tasks = append(tasks, &newtask)
	}
	return
}

func (ytts YtTransferTables) GetBackupNames() (result []string) {
	for _, tt := range ytts {
		result = append(result, tt.BackupName())
	}
	sort.Sort(sort.Reverse(sort.StringSlice(result)))
	return
}

func (ytts YtTransferTables) GetBackupByName(name string) (result YtTransferTables) {
	for _, tt := range ytts {
		if strings.Contains(tt.BackupName(), name) {
			result = append(result, tt)
		}
	}
	return
}

//Возвращает наиболее полный свежий бекап.
func (ytts YtTransferTables) GoodBackup() (result YtTransferTables) {
	for _, name := range ytts.GetBackupNames() {
		tables := ytts.GetBackupByName(name)
		//проверять тут mysql_sync_state
		if len(tables) > len(result) {
			result = tables
			continue
		}
		if len(tables) == len(result) {
			break
		}
	}
	return
}

func (ytts YtTransferTables) Name() (result string) {
	for _, table := range ytts {
		return table.NameBackup
	}
	return
}

func (ytts YtTransferTables) FindTransferTable(name string) (tt YtTransferTable, ok bool) {
	for _, table := range ytts {
		if strings.Contains(table.Path, name) {
			return table, true
		}
	}
	return
}

func (ytts YtTransferTables) FindSourceNodePath(name string) (tt YtTransferTable, ok bool) {
	for _, table := range ytts {
		if strings.Contains(table.SourceNodePath(), name) {
			return table, true
		}
	}
	return
}

func (tts *YtTransferTasks) MonitorTransferTasks() (err error) {
	for {
		(*tts).UpdateTasksStatus()
		(*tts).RestartFailedTask()

		if (*tts).CheckAllTasksDone() {
			break
		}

		for _, task := range tts.GetTasksNotDone() {
			fmt.Printf("[NotDoneTasks] taskid: %s, state: %s, retry: %d, error: %s\n", task.TaskID, task.State, *task.Retry, task.GetLastErrorString())
		}

		if tasks := tts.GetRealyFailedTasks(); len(tasks) > 0 {
			return fmt.Errorf("any tasks failed: %v", tasks)
		}
		time.Sleep(10 * time.Second)
	}
	return
}

func CreateHTTPClient() *http.Client {
	client := &http.Client{
		Transport: &http.Transport{
			MaxIdleConnsPerHost: MaxIdleConn,
		},
		Timeout: time.Duration(RequestTimeout) * time.Second,
	}
	return client
}

func GetYtToken() (string, error) {
	if token := os.Getenv("YT_TOKEN"); len(token) != 0 {
		if strings.HasPrefix(token, "AQAD") {
			return strings.Trim(token, "\n"), nil
		} else {
			return "", fmt.Errorf("invalid value token: '%s'", token)
		}
	}
	//fmt.Printf("ERR YT token not found")
	return "", fmt.Errorf("empty YT_TOKEN")
}

func setHeaders(req *http.Request) error {
	if token, err := GetYtToken(); err == nil {
		req.Header.Set("Authorization", fmt.Sprintf("OAuth %s", token))
	} else {
		return err
	}
	req.Header.Set("Accept-Type", "application/json")
	req.Header.Set("Content-Type", "application/json")
	return nil
}

func NewPostRequest(args ...interface{}) (*http.Request, error) {
	address, ok := args[0].(string)
	if !ok {
		return nil, fmt.Errorf("error convert %v to string", args[0])
	}
	body, ok := args[1].([]byte)
	if !ok {
		return nil, fmt.Errorf("error convert 'body' %v to []byte", args[1])
	}
	req, err := http.NewRequest("POST", address, bytes.NewReader(body))
	if err != nil {
		return nil, fmt.Errorf("error request %s %s: %s", address, body, err)
	}
	if err := setHeaders(req); err != nil {
		return nil, fmt.Errorf("error set  Authorization: %s", err)
	}
	return req, nil
}

func NewGetRequest(args ...interface{}) (*http.Request, error) {
	address, ok := args[0].(string)
	if !ok {
		return nil, fmt.Errorf("error convert method %v to string", args[0])
	}
	req, err := http.NewRequest("GET", address, nil)
	if err != nil {
		return nil, fmt.Errorf("error GET request %s: %s", address, err)
	}
	if err := setHeaders(req); err != nil {
		return nil, fmt.Errorf("error set  Authorization: %s", err)
	}
	return req, nil
}

func requestToString(req *http.Request) string {
	if strings.Contains(req.Method, "POST") {
		return ff("%s %s %v", req.Method, req.URL, req.PostForm)
	}
	return ff("%s %s", req.Method, req.URL)
}

func runHTTP(method, address string, body []byte) ([]byte, error) {
	var requestFunc func(...interface{}) (*http.Request, error)
	if strings.Contains(method, "POST") {
		requestFunc = NewPostRequest
	} else {
		requestFunc = NewGetRequest
	}
	request, err := requestFunc(address, body)
	if err != nil {
		return nil, fmt.Errorf("error (%s %s) create request: %s", method, address, err)
	}
	log.Printf("send request %s address %s, %s\n", method, address, body)
	response, err := httpClient.Do(request)
	if err != nil {
		return nil, fmt.Errorf("error (%s) response: %s", requestToString(request), err)
	}
	defer func() { _ = response.Body.Close() }()
	if response.StatusCode != http.StatusOK {
		return nil, fmt.Errorf("error (%s) response code: %d, status: %s", requestToString(request),
			response.StatusCode, response.Status)
	}
	answer, err := ioutil.ReadAll(response.Body)
	if err != nil {
		return nil, fmt.Errorf("error (%s) read response body: %s", requestToString(request), err)
	}
	return answer, nil
}
