package zookeeper

import (
	"bytes"
	"errors"
	"fmt"
	"log/syslog"
	"path/filepath"
	"strings"
	"time"

	"github.com/go-zookeeper/zk"
)

var logger *syslog.Writer
var err error
var f = fmt.Sprintf

type ZkConnect struct {
	*zk.Conn
	Event   <-chan zk.Event
	Servers []string
	Logger  *syslog.Writer
}

func Connect(servers []string, newLogger *syslog.Writer) (ZkConnect, error) {
	var conn *zk.Conn
	var session <-chan zk.Event
	logger = newLogger
	for i := 0; i < 3; i++ {
		conn, session, err = zk.Connect(servers, 10*time.Second)
		if err != nil {
			continue
		}
		return ZkConnect{conn, session, servers, logger}, nil
	}
	msg := fmt.Sprintf("error connect %s: %s", servers, err)
	Wrap(logger.Crit(msg))
	return ZkConnect{}, errors.New(msg)
}

func (zc *ZkConnect) CheckConnect() bool {
	if zc.State() == zk.State(-1) {
		if newconn, err := Connect(zc.Servers, zc.Logger); err != nil {
			fmt.Println(err)
			return false
		} else {
			zc = &newconn
		}
	}
	return true
}

type Node struct {
	Group   string
	Path    string
	Stat    *zk.Stat
	Data    []byte
	Deleted int
}

func NewZkNode(group, path string) *Node {
	stat := zk.Stat{}
	data := []byte{}
	return &Node{Group: fmt.Sprintf("db_%s", group), Path: path, Stat: &stat, Data: data, Deleted: 0}
}

func (n Node) GetGroup() string {
	return strings.ReplaceAll(n.Group, "db_", "")
}

func (n Node) GetNodeVersion() int32 {
	if n.Stat == nil {
		return 0
	}
	return n.Stat.Version
}

type Nodes []*Node

func (ns Nodes) GetNodeVersion(path string) int32 {
	for _, node := range ns {
		if node.Path == path {
			return node.Stat.Version
		}
	}
	return 0
}

func (ns *Nodes) DeleteNode(rmnode Node) {
	var indexDeleted []int
	for num, node := range *ns {
		if node.Path == rmnode.Path {
			indexDeleted = append(indexDeleted, num)
		}
	}
	if len(indexDeleted) > 0 {
		var cnt, num int
		for cnt, num = range indexDeleted {
			(*ns)[num] = (*ns)[len(*ns)-cnt-1]
		}
		*ns = (*ns)[:len(*ns)-cnt-1]
	}
}

func (ns *Nodes) AddNode(newnode Node) {
	*ns = append(*ns, &newnode)
}

func (ns *Nodes) UpdateNode(newnode Node) {
	if !ns.ExistNode(newnode) {
		ns.AddNode(newnode)
		return
	}
	for _, node := range *ns {
		if node.Group == newnode.Group && node.Path == newnode.Path {
			node.Data = newnode.Data
			node.Stat.Version++
		}
	}
}

func (ns *Nodes) FindGroup(group, status string) *Node {
	for _, node := range *ns {
		if strings.Contains(node.Group, group) && strings.Contains(node.Path, status) {
			return node
		}
	}
	return nil
}

func (ns Nodes) ExistNode(finded Node) bool {
	for _, node := range ns {
		if node.Path == finded.Path {
			return true
		}
	}
	return false
}

func (zc *ZkConnect) ExistNode(path string) (bool, error) {
	if ok, _, err := zc.Exists(path); err != nil {
		return false, err
	} else if !ok {
		msg := fmt.Sprintf("[ERROR] node %s dosn't exist", path)
		Wrap(logger.Info(msg))
		return false, errors.New(msg)
	}
	return true, nil
}

func (zc *ZkConnect) ListPath(path string) []string {
	var (
		zklist []string
	)

	if ok, err := zc.ExistNode(path); err == nil && ok {
		if zklist, _, err := zc.Children(path); err == nil {
			return zklist
		}
		Wrap(logger.Debug(f("list zkpath %s children: %s", path, err)))
	}
	return zklist
}

func (zc *ZkConnect) CleanPath(node Node) (bool, error) {
	msg := bytes.NewBuffer([]byte(""))
	var removedNodes []string
	baseZkPath := filepath.Dir(node.Path)
	for _, path := range zc.ListPath(baseZkPath) {
		removedNodes = append(removedNodes, filepath.Join(baseZkPath, path), baseZkPath)
	}
	Wrap(logger.Info(f("start clean zkdir %s", baseZkPath)))
	for _, path := range removedNodes {
		_, stat, _ := zc.Get(path)
		Wrap(logger.Info(f("remove node %s", path)))
		if err := zc.Delete(path, stat.Version); err != nil {
			msg.WriteString(fmt.Sprintf("%s, ", err))
		}
	}
	if msg.Len() != 0 {
		Wrap(logger.Crit(msg.String()))
	}
	return msg.Len() == 0, errors.New(msg.String())
}

func (zc *ZkConnect) LoadNode(group, path string) (Node, error) {
	var (
		ok   bool
		err  error
		data []byte
		stat *zk.Stat
	)
	Wrap(logger.Info(f("start load zknode %s", path)))
	if ok, err = zc.ExistNode(path); err == nil && ok {
		if data, stat, err = zc.Get(path); err != nil {
			Wrap(logger.Crit(f("[ERROR] load zknode %s: %s", path, err)))
		}
	}
	node := Node{
		Group: group,
		Path:  path,
		Stat:  stat,
		Data:  data,
	}
	Wrap(logger.Info(f("done load zknode %s", path)))
	return node, err
}

func (zc *ZkConnect) SaveNode(node Node) error {
	var (
		err  error
		stat *zk.Stat
	)

	Wrap(logger.Debug(f("start save zknode %s", node.Path)))
	tree := strings.Split(node.Path, "/")
	for i := range tree {
		path := "/" + filepath.Join(tree[:i+1]...)
		if ok, _ := zc.ExistNode(path); len(path) > 1 && !ok {
			if _, err := zc.Create(path, []byte{}, 0, zk.WorldACL(zk.PermAll)); err != nil {
				Wrap(logger.Crit(f("[zookeeper/SaveNode] error create node %s: %s", path, err)))
			}
		}
	}

	_, stat, _ = zc.Get(node.Path)

	Wrap(logger.Debug(f("zknode %s current/save id %d/%d", node.Path, stat.Version, node.GetNodeVersion())))
	if stat.Version == 0 || stat.Version < node.GetNodeVersion() {
		if _, err = zc.Set(node.Path, node.Data, stat.Version); err != nil {
			Wrap(logger.Crit(f("save zknode %s", node.Path)))
			return err
		}
	}
	Wrap(logger.Debug(f("done save zknode %s", node.Path)))
	return nil
}

func (zc *ZkConnect) DeleteNode(node Node) error {
	var (
		err  error
		stat *zk.Stat
	)
	_, stat, _ = zc.Get(node.Path)
	Wrap(logger.Info(f("start delete zknode %s", node.Path)))
	if stat.Version == 0 || stat.Version < node.GetNodeVersion() {
		if err = zc.Delete(node.Path, stat.Version); err != nil {
			Wrap(logger.Crit(f("delete node %s: %s", node.Path, err)))
			return err
		}
	}
	Wrap(logger.Info(f("done delete zknode %s", node.Path)))
	return nil
}

func Wrap(err error) {
	if err != nil {
		fmt.Printf("[wrapper] error: %s\n", err)
	}
}
