package main

import (
	"bufio"
	"context"
	"crypto/tls"
	"crypto/x509"
	"flag"
	"fmt"
	"io/ioutil"
	"math/rand"
	"net"
	"net/http"
	_ "net/http/pprof"
	"os"
	"runtime"
	"runtime/debug"
	"strconv"
	"strings"
	"sync/atomic"
	"time"

	"a.yandex-team.ru/direct/infra/dt-graphite-ch-cacher/internal/consistent"
	"a.yandex-team.ru/direct/infra/dt-graphite-ch-cacher/internal/delta"
	"a.yandex-team.ru/direct/infra/dt-graphite-ch-cacher/internal/mylib"
)

const (
	DEBUG  = 1
	layout = "Jan 2, 2006 at 15:04:02"
)

var (
	confFile     *string = flag.String("config", "config.ini", "config file")
	listenPort   *string = flag.String("port", "8765", "listen port for incoming data")
	metricSearch *string = flag.String("metricSearch", "9876", "port for delta handler")
	conAlive     int64   = 0
)

func Wrap(err error) {
	if err != nil {
		fmt.Printf("[wrapper] error: %s\n", err)
	}
}

func processConnection(local net.Conn, boss mylib.Boss, mon *mylib.Mmon) {
	defer func() { _ = local.Close() }()
	r := bufio.NewReader(local)
	lines := make(chan string, 10000)
	defer close(lines)
	go lineReader(r, lines)
	lastRcv := time.Now()
	ticker := time.NewTicker(30 * time.Second)
	// tag for loop to break it properly
L:
	for {
		select {
		case line := <-lines:
			//            line := <-lines
			if line == "__end_of_data__" {
				break L
			}
			// process data
			atomic.AddInt32(&mon.Rcv, 1)
			parse(line, boss)
			lastRcv = time.Now()
		case <-ticker.C:
			if time.Since(lastRcv).Seconds() > 60 {
				fmt.Printf("closing connection after read timeout 60sec, %s\n", local.RemoteAddr().String())
				break L
			}
		}
	}
	ticker.Stop()
	atomic.AddInt64(&conAlive, -1)
}

func lineReader(r *bufio.Reader, lines chan string) {
	for {
		line, _, err := r.ReadLine()
		if err != nil {
			// my oh my, such an ugly way to stop accepting data via channels :(
			lines <- "__end_of_data__"
			break
		}
		lines <- string(line)
	}
}

func parse(input string, boss mylib.Boss) {
	if len(input) < 15 {
		// input str must be at least 15 chars
		return
	}
	input = strings.Trim(input, " ")

	var metric, data, ts string
	//format: metric.path value timestamp
	arr := strings.Fields(input)
	if len(arr) == 3 {
		metric, data, ts = arr[0], arr[1], arr[2]
		if strings.Count(metric, "'") > 0 {
			fmt.Printf("Failed to parse. Metric contains a quotation mark: %v\n", metric)
			return
		}
		// convert timestamp to int64
		tsInt, err := strconv.ParseInt(ts, 0, 64)
		if err != nil {
			fmt.Println("Failed to parse to int")
			return
		}
		t := time.Unix(tsInt, 0)
		currTime := time.Now()

		// just check if data is a real float number
		// but pass to clickhouse as a string to avoid useless convertation
		_, err = strconv.ParseFloat(data, 32)
		if err != nil {
			fmt.Printf("Failed to parse data to float, bogus string [ %s ]\n", input)
			return
		}

		if boss.Single == 1 {
			w := singleSender(boss.Senders)
			w.Pipe <- fmt.Sprintf("('%s', %s, %d, '%s', %d)\n", metric, data, t.Unix(), t.Format("2006-01-02"), currTime.Unix())
		} else {
			r, err := boss.Ring.GetN(metric, boss.Rf)
			if err != nil {
				fmt.Printf("Failed to get caches for metric %s, err %v\n", metric, err)
				return
			}

			for _, item := range r {
				w := getSender(item, boss.Senders)
				w.Pipe <- fmt.Sprintf("('%s', %s, %d, '%s', %d)\n", metric, data, t.Unix(), t.Format("2006-01-02"), currTime.Unix())
			}
		}
		// send metric to deltaManager

		select {
		case boss.DeltaChan <- metric:
		default:
		}
	} else {
		fmt.Printf("[Error] Bad formated input: %s\n", input)
	}
}

func sender(sender mylib.Sender, sendMon *int32) {
	fmt.Printf("Started sender with options: [%s:%d instance %d]\n", sender.Host, sender.Port, sender.Index)

	ticker := time.NewTicker(1 * time.Second)
	defer ticker.Stop()
	lastSend := time.Now()
	for range ticker.C {
		if queueLen := len(sender.Pipe); queueLen > 0 {
			if time.Since(lastSend) > 10*time.Second || queueLen > 1500000 {
				sendData(sender)
				lastSend = time.Now()
				atomic.AddInt32(sendMon, int32(queueLen))
				fmt.Printf("sending %d metrics\n", queueLen)
			}
		} else {
			lastSend = time.Now()
		}
	}
}

func sendData(c mylib.Sender) {
	// an ugly hack to handle conn and write timeouts

	var data []string
L:
	for {
		select {
		case i := <-c.Pipe:
			data = append(data, i)
		default:
			break L
		}
	}
	raw := strings.Join(data, ", ")

	pool, _ := x509.SystemCertPool()
	mytls := &tls.Config{
		RootCAs:            pool,
		InsecureSkipVerify: true,
	}

	transport := http.Transport{
		Dial:            mylib.DialTimeout,
		TLSClientConfig: mytls,
	}

	client := http.Client{
		Transport: &transport,
	}

	url := fmt.Sprintf("https://%s:%d", c.Host, c.Port)

	ctx, cancel := context.WithTimeout(context.Background(), 300*time.Second)
	defer cancel()

	for retry := 0; retry < 3; retry++ {
		req := strings.NewReader(fmt.Sprintf("INSERT INTO %s.graphite VALUES %s", c.Database, raw))
		hr, _ := http.NewRequest("POST", url, req)
		hr = hr.WithContext(ctx)
		hr.Header.Add("X-ClickHouse-User", c.User)
		hr.Header.Add("X-ClickHouse-Key", c.Password)
		hr.Header.Add("Content-Type", "text/xml")
		//resp, err := client.Post(url, "text/xml", req)
		startTime := time.Now()

		resp, err := client.Do(hr)
		if err != nil {
			fmt.Printf("Failed to send data to %s:%d, retries left %d, going to sleep for 1s, error: %v\n", c.Host, c.Port, retry, err)
			time.Sleep(1000 * time.Millisecond)
			continue
		}
		body, err := ioutil.ReadAll(resp.Body)
		if err != nil {
			fmt.Printf("Error read Body: %s %s\n", body, err)
		}
		if resp.StatusCode != 200 {
			fmt.Printf("Got not 200 response status for %s:%d, retries left %d, going to sleep for 1s, status: %s\n", c.Host, c.Port, retry, resp.Status)
			time.Sleep(1000 * time.Millisecond)
			_ = resp.Body.Close()
			continue
		}
		_ = resp.Body.Close()
		fmt.Printf("sending %d bytes, count metrics %d, sending time %s\n", len(raw), len(data), time.Since(startTime))
		//fmt.Printf("sending %s\n", raw[:10])
		break
	}
}

func singleSender(senders []mylib.Sender) mylib.Sender {
	// if we just have multiply senders for one host, than senders will be
	// randomly rotated
	return senders[rand.Intn(len(senders))]
}

func getSender(r string, senders []mylib.Sender) mylib.Sender {
	var senderArr []mylib.Sender
	for _, w := range senders {
		if r == w.Host {
			senderArr = append(senderArr, w)
		}
	}
	randIndex := rand.Intn(len(senderArr))

	return senderArr[randIndex]
}

// functions for monitoring internals
func monitor(mon *mylib.Mmon, boss mylib.Boss) {
	// just pick up first sender all the time, kiss
	sender := boss.Senders[0]
	ticker := time.Tick(1000 * time.Millisecond)
	last := time.Now()
	for range ticker {
		//			log("debug", fmt.Sprintf("sending to %s..", sender.host))
		currTime := time.Now()
		if currTime.Unix() > last.Unix() {
			sendMonData(atomic.SwapInt32(&mon.Send, 0), atomic.SwapInt32(&mon.Rcv, 0), atomic.SwapInt32(&mon.Conn, 0), boss, currTime, sender)
			last = currTime
		}
	}
}

func sendMonData(m int32, r int32, c int32, boss mylib.Boss, ts time.Time, sender mylib.Sender) {
	// get memstats
	mem := new(runtime.MemStats)
	runtime.ReadMemStats(mem)
	data := map[string]int64{
		"metrics_send": int64(m),
		"metrics_rcvd": int64(r),
		"conn":         int64(c),
		"conn_alive":   atomic.LoadInt64(&conAlive),
		"heap_sys":     int64(mem.HeapSys),
		"heap_idle":    int64(mem.HeapIdle),
		"heap_inuse":   int64(mem.HeapInuse),
		"alloc":        int64(mem.Alloc),
		"sys":          int64(mem.Sys),
	}

	//out := ""
	tsF := ts.Format("2006-01-02")
	for key, val := range data {
		metricName := fmt.Sprintf("one_sec.int_%s.%s", boss.Port, key)
		out := fmt.Sprintf("('%s', %d, %d, '%s', %d)", metricName, val, ts.Unix(), tsF, ts.Unix())
		// all monitor metrics must exist in metricsearch index
		boss.DeltaChan <- metricName
		sender.Pipe <- out
	}
}

func startWorkers(config mylib.Config, r *consistent.Consistent, mon *mylib.Mmon) []mylib.Sender {
	workers := make([]mylib.Sender, 0)
	index := 0
	for _, st := range config.Storages {
		for j := 0; j <= st.Num; j++ {
			var w mylib.Sender
			w.Host = st.Host
			w.Port = st.Port
			w.User = config.User
			w.Password = config.Password
			w.Database = config.Database
			w.Pipe = make(chan string, config.ChanLimit)
			w.Index = index
			r.Add(st.Host)
			index++
			workers = append(workers, w)
			go sender(w, &mon.Send)
		}
	}
	return workers
}

func freemem() {
	ticker := time.Tick(1000 * time.Millisecond)
	for range ticker {
		fmt.Println("just checking")
	}
}

type AliveType struct {
	Alive bool `json:"alive"`
}

func checkSenders(boss mylib.Boss, status *bool) {
	/*pool, _ := x509.SystemCertPool()
	mytls := &tls.Config{
		RootCAs:            pool,
		InsecureSkipVerify: true,
	}

	transport := http.Transport{
		Dial: mylib.DialTimeout,
		//TLSClientConfig: mytls,
	}

	client := http.Client{
		Transport: &transport,
	}

	// Вариант проверки живости через ручку /alive-pool у прокси кликхауса.
	ticker := time.NewTicker(1 * time.Second)
	for range ticker.C {
		w := boss.Senders[0]

		url := fmt.Sprintf("http://%s:%d/alive-pool", w.Host, 8085)
		resp, err := client.Get(url)
		if err != nil {
			fmt.Printf("error check: %s, error %s\n", url, err)
			*status = false
		}
		data, _ := ioutil.ReadAll(resp.Body)
		d := AliveType{Alive: false}
		_ = json.Unmarshal(data, &d)
		if resp.StatusCode == 200 && d.Alive {
			*status = true
		} else {
			fmt.Printf("error check: %s, status %s, data: %s\n", url, resp.Status, data)
			*status = false
		}
		_ = resp.Body.Close()
	}*/

	/* старый кусок данных, в котором проверялась работа кликхауса по всем хостам.
	for _, w := range boss.Senders {
		url := fmt.Sprintf("https://%s:%d", w.Host, w.Port)
		resp, err := client.Get(url)
		if err == nil {
			defer func() { _ = resp.Body.Close() }()
			_, _ = ioutil.ReadAll(resp.Body)
			if resp.StatusCode == 200 {
				return true
			}
			fmt.Printf("error check: %s, status %s\n", url, resp.Status)
		} else {
			fmt.Printf("error check: %s, error %s\n", url, err)
		}
	}*/

	w := boss.Senders[0]
	ticker := time.NewTicker(2 * time.Second)
	for range ticker.C {
		conn, err := net.DialTimeout("tcp", net.JoinHostPort(w.Host, fmt.Sprint(w.Port)), 5*time.Second)
		if err != nil {
			fmt.Printf("error check: %s:%d: %s\n", w.Host, w.Port, err)
			*status = false
			continue
		}
		if conn != nil {
			_ = conn.Close()
			*status = true
			continue
		}
		*status = false
	}
}

func main() {
	runtime.GOMAXPROCS(4)
	debug.SetGCPercent(90)
	//	go freemem()

	rand.Seed(time.Now().Unix())
	// parse config
	flag.Parse()
	if flag.NFlag() != 3 {
		fmt.Printf("usage: cacher -config config_file -port listen_port -metricSearch metric_search_host\n")
		flag.PrintDefaults()
		os.Exit(1)
	}

	var config mylib.Config
	if _, err := os.Stat(*confFile); os.IsNotExist(err) {
		fmt.Printf("no such file: %s, loading default\n", *confFile)
		config = mylib.Load("")
	} else {
		config = mylib.Load(*confFile)
		fmt.Printf("using %s as config file\n", *confFile)
	}

	// set hash ring object
	r := consistent.New()
	// set up monitoring
	mon := new(mylib.Mmon)
	// spawn db writers and fill hash ring object
	workers := startWorkers(config, r, mon)

	var boss mylib.Boss
	deltaChan := make(chan string, 5000000)
	// create Boss var (used to hide tons of vars in functions stack)
	boss.Senders = workers
	boss.Rf = config.Rf
	boss.Ring = r
	boss.Single = 0
	boss.Port = *listenPort
	boss.DeltaChan = deltaChan
	// if we have a single host, than we can ignore hash ring mess
	// and do simple rr rotation of senders
	if len(boss.Ring.Members()) == 1 {
		boss.Single = 1
	}
	// start delta manager
	if config.EnableDelta > 0 {
		metricServers := strings.Split(*metricSearch, ",")
		go delta.DeltaManager(deltaChan, workers, metricServers, boss)
		go func() {
			Wrap(http.ListenAndServe(":5"+*listenPort, nil))
		}()
	} else {
		go delta.BogusDelta(deltaChan)
	}

	go monitor(mon, boss)

	var status bool
	go checkSenders(boss, &status)

	for {
		//при живом приёмнике открываем сокет на входящие соединения
		if status {
			ln, err := net.Listen("tcp", ":"+*listenPort)
			fmt.Printf("Started on %s port\n", *listenPort)
			fmt.Printf("worker chanLimit %d\n", config.ChanLimit)
			if err != nil {
				fmt.Printf("Unable to start listener, %v\n", err)
			}

			//принимаем входящие соедиения(на всякий еще раз проверяем приёмник)
			for {
				conn, err := ln.Accept()
				if err == nil {
					if status {
						go processConnection(conn, boss, mon)
						// received new connection
						atomic.AddInt32(&mon.Conn, 1)
						atomic.AddInt64(&conAlive, 1)
					} else {
						fmt.Printf("Graphite servers died, %v\n", boss.Senders)
						Wrap(ln.Close())
						time.Sleep(5 * time.Second)
						break
					}
				} else {
					fmt.Printf("Failed to accept connection, %v\n", err)
				}
			}
		}
	}
}
