package main

import (
	"bytes"
	"crypto/tls"
	"crypto/x509"
	"flag"
	"fmt"
	"io/ioutil"
	"log"
	"net"
	"net/http"
	"os"
	"os/exec"
	"regexp"
	"strings"
	"time"

	"gopkg.in/ini.v1"

	"a.yandex-team.ru/direct/infra/dt-graphite-ch-cacher/internal/mylib"
)

type MyError struct {
	Value string
}

type MyConfig struct {
	webAddr          string
	cacherAddr       string
	storageAddr      []string
	clickhouseToken  string
	clickhousePasswd string
	clickhouseUser   string
	metricReadyFile  string
}

var (
	confFile *string = flag.String("config", "/etc/cacher/checks.ini", "config file for checks")
)

func (e MyError) Error() string {
	return fmt.Sprintf("%v", e.Value)
}

func iptruler(s string) {
	check := exec.Command("iptruler", "dump")
	output, _ := check.Output()
	closed := bytes.Contains(output, []byte("REJECT"))
	if strings.Contains(s, "down") && closed {
		fmt.Printf("Iptables already closed. Skip it.\n")
		return
	}
	if strings.Contains(s, "up") && !closed {
		return
	}
	cmd := exec.Command("iptruler", "all", s)
	var out bytes.Buffer
	cmd.Stdout = &out
	err := cmd.Run()
	if err != nil {
		fmt.Printf("Error closed iptables. Msg: %s\n", err)
	} else {
		fmt.Printf("Run command: iptruler all %s. %s\n", s, out.String())
	}
}

func Wrap(err error) {
	if err != nil {
		fmt.Printf("[wrapper] error: %s\n", err)
	}
}

func httpInterface(s string) error {
	client := http.Client{}
	var httpHeader = regexp.MustCompile(`^http:`)
	if len(httpHeader.FindStringSubmatch(s)) == 0 {
		s = "http://" + s
	}
	req, _ := http.NewRequest("GET", s, nil)
	resp, err := client.Do(req)

	if err == nil {
		// fmt.Printf("Web graphite status code: %v %s\n", resp.StatusCode, resp.Status)
		defer func() { _ = resp.Body.Close() }()
		_, _ = ioutil.ReadAll(resp.Body)
		if resp.StatusCode == 200 {
			return nil
		} else {
			return MyError{fmt.Sprintf("status code not 200: %s", resp.Status)}
		}
	} else {
		return MyError{fmt.Sprintf("failed to connect %s", err)}
	}
}

func httpsInterface(address, user, passwd string) error {
	pool, _ := x509.SystemCertPool()
	mytls := &tls.Config{
		RootCAs:            pool,
		InsecureSkipVerify: true,
	}

	transport := http.Transport{
		Dial:            mylib.DialTimeout,
		TLSClientConfig: mytls,
	}

	client := http.Client{
		Transport: &transport,
	}

	url := fmt.Sprintf("https://%s", address)

	for retry := 0; retry < 3; retry++ {
		hr, _ := http.NewRequest("GET", url, nil)
		hr.Header.Add("X-ClickHouse-User", user)
		hr.Header.Add("X-ClickHouse-Key", passwd)
		hr.Header.Add("Content-Type", "text/xml")
		resp, err := client.Do(hr)
		if err != nil {
			fmt.Printf("Failed connect %s, retries left %d, going to sleep for 1s, error: %v\n", address, retry, err)
			time.Sleep(1 * time.Second)
			continue
		}
		defer func() { _ = resp.Body.Close() }()
		body, err := ioutil.ReadAll(resp.Body)
		if err != nil {
			fmt.Printf("Error read Body: %s %s\n", body, err)
		}
		if resp.StatusCode != 200 {
			fmt.Printf("Got not 200 response status for %s, retries left %d, going to sleep for 1s, status: %s\n", address, retry, resp.Status)
			time.Sleep(1000 * time.Millisecond)
			continue
		}
		return nil
	}
	return fmt.Errorf("failed connect clickhouse")
}

func checkCacher(s string) error {
	conn, err := net.Dial("tcp", s)
	if err == nil {
		Wrap(conn.Close())
	}
	return err
}

func checkReadyMetricsearch(s string) error {
	if _, err := os.Stat(s); err != nil {
		return fmt.Errorf("error stat file %s: %s", s, err)
	}
	return nil
}

func main() {

	flag.Parse()

	config := &MyConfig{
		webAddr:          "http://localhost/browser/header",
		cacherAddr:       "localhost:2024",
		storageAddr:      []string{"127.0.0.1:8123"},
		clickhouseUser:   "default",
		clickhouseToken:  "/tmp/token",
		clickhousePasswd: "",
		metricReadyFile:  "/var/cache/merticsearch_firstinit",
	}

	if _, err := os.Stat(*confFile); os.IsNotExist(err) {
		fmt.Printf("no such file: %s, loading default\n", *confFile)
	} else {
		fmt.Printf("using %s as config file\n", *confFile)
		cfg, err := ini.Load(*confFile)
		if err != nil {
			fmt.Printf("Error load ini config %s.", *confFile)
		}
		if cfg.Section("main").HasKey("webAddr") {
			config.webAddr = cfg.Section("main").Key("webAddr").String()
		}
		if cfg.Section("main").HasKey("cacherAddr") {
			config.cacherAddr = cfg.Section("main").Key("cacherAddr").String()
		}
		if cfg.Section("main").HasKey("storageAddr") {
			config.storageAddr = cfg.Section("main").Key("storageAddr").Strings(",")
		}
		if cfg.Section("main").HasKey("clickhouseUser") {
			config.clickhouseUser = cfg.Section("main").Key("clickhouseUser").String()
		}
		if cfg.Section("main").HasKey("clickhouseToken") {
			config.clickhouseToken = cfg.Section("main").Key("clickhouseToken").String()
		}
		if cfg.Section("main").HasKey("metricReadyFile") {
			config.clickhouseToken = cfg.Section("main").Key("clickhouseToken").String()
		}
	}

	if pass, err := ioutil.ReadFile(config.clickhouseToken); err == nil {
		config.clickhousePasswd = string(pass)
	} else {
		iptruler("down")
		log.Fatalf("error read file %s: %s", config.clickhouseToken, err)
	}

	alive := false
	for {
		aliveHTTP := httpInterface(config.webAddr)
		aliveCacher := checkCacher(config.cacherAddr)
		readyMetricSearch := checkReadyMetricsearch(config.metricReadyFile)

		var grSliceAlive []error
		for _, value := range config.storageAddr {
			if grAlive := httpsInterface(value, config.clickhouseUser, config.clickhouseToken); grAlive != nil {
				grSliceAlive = append(grSliceAlive, grAlive)
			}
		}

		if (aliveHTTP == nil) && (aliveCacher == nil) && (len(grSliceAlive) == 0) && (readyMetricSearch == nil) {
			if alive {
				iptruler("up")
			}
			alive = true
		} else {
			if aliveHTTP != nil {
				fmt.Printf("Graphite-web %s\n", aliveHTTP)
			}
			if aliveCacher != nil {
				fmt.Printf("Graphite-ch-cacher %s\n", aliveCacher)
			}
			if len(grSliceAlive) != 0 {
				fmt.Printf("Clickhouse storages %s\n", grSliceAlive)
			}
			if readyMetricSearch != nil {
				fmt.Printf("Merticasearch not ready. Not found %s\n", readyMetricSearch)
			}
			if !alive {
				iptruler("down")
			}
			alive = false
		}
		time.Sleep(10 * time.Second)
	}
}
