package main

import (
	"context"
	"crypto/tls"
	"crypto/x509"
	"flag"
	"fmt"
	"io"
	"io/ioutil"
	"log"
	"net/http"
	"os"
	"os/exec"
	"time"

	"github.com/gofrs/flock"
	"gopkg.in/ini.v1"
)

const (
	DEBUG     = false
	CONFILE   = "/etc/cacher/checks.ini"
	QUERY     = "SELECT distinct(Path) FROM graphite_d"
	TMPFILE   = "/tmp/metrics.dat"
	FIRSTINIT = "/var/cache/merticsearch_firstinit"
	LOCKFILE  = "/var/lock/metricsearch-init.lock"
)

var (
	confile, tmpfile string
	force, debug     bool
)

type MainSection struct {
	Config `ini:"main"`
}

type Config struct {
	User            string `ini:"clickhouseUser"`
	Token           string `ini:"clickhouseToken"`
	Password        string `ini:"-"`
	Database        string `ini:"clickhouseDatabase"`
	Address         string `ini:"storageAddr"`
	MaxRequestHours int    `ini:"maxRequestHours"`
}

func LoadConfig(path string) (cnf MainSection, err error) {
	cnf = MainSection{
		Config{
			User:            "default",
			Token:           "",
			Password:        "",
			Database:        "graphite",
			Address:         "localhost:8123",
			MaxRequestHours: 12,
		},
	}
	data, err := ioutil.ReadFile(path)
	if err != nil {
		return cnf, fmt.Errorf("error read file %s: %s", path, err)
	}
	cfg, err := ini.Load(data)
	if err != nil {
		return cnf, fmt.Errorf("error load ini: %s", err)
	}
	if debug {
		fmt.Println(cfg.GetSection("main"))
	}
	if err := cfg.MapTo(&cnf); err != nil {
		return cnf, fmt.Errorf("error mapping ini: %s", err)
	}
	if len(cnf.Token) > 0 {
		data, err := ioutil.ReadFile(cnf.Token)
		if err != nil {
			return cnf, fmt.Errorf("error read %s: %s", cnf.Token, err)
		}
		cnf.Password = string(data)
	}
	return cnf, nil
}

func ExecQuery(query string, cnf Config, savedFile string) error {
	pool, _ := x509.SystemCertPool()
	mytls := &tls.Config{
		RootCAs:            pool,
		InsecureSkipVerify: true,
	}

	transport := http.Transport{
		TLSClientConfig: mytls,
	}

	ctx, cancel := context.WithCancel(context.Background())

	stop := time.AfterFunc(time.Duration(cnf.MaxRequestHours)*time.Hour, func() {
		log.Printf("read metrics CH more than %d hours. Cancel request\n", cnf.MaxRequestHours)
		cancel()
	})

	defer func() { _ = stop.Stop() }()

	client := http.Client{
		Transport: &transport,
	}

	address := fmt.Sprintf("https://%s/", cnf.Address)
	hr, _ := http.NewRequest("GET", address, nil)
	hr = hr.WithContext(ctx)
	values := hr.URL.Query()
	values.Add("query", query)
	values.Add("database", cnf.Database)
	hr.URL.RawQuery = values.Encode()
	hr.Header.Add("X-ClickHouse-User", cnf.User)
	hr.Header.Add("X-ClickHouse-Key", cnf.Password)
	hr.Header.Add("Content-Type", "text/xml")

	if debug {
		fmt.Printf("request: %+v\n", hr)
	}

	for retry := 0; retry < 3; retry++ {
		resp, err := client.Do(hr)
		if err != nil {
			log.Printf("failed to send data to %s, retries left %d, going to sleep for 1s, error: %v\n", address, retry, err)
			time.Sleep(1000 * time.Millisecond)
			continue
		}
		if resp.StatusCode != 200 {
			log.Printf("got not 200 response status for %s, retries left %d, going to sleep for 1s, status: %s\n", address, retry, resp.Status)
			time.Sleep(1000 * time.Millisecond)
			_ = resp.Body.Close()
			continue
		}
		fd, err := os.OpenFile(savedFile, os.O_CREATE|os.O_WRONLY, 0644)
		if err != nil {
			log.Printf("error open file %s for saved: %s", savedFile, err)
			_ = resp.Body.Close()
			continue
		}
		if _, err = io.Copy(fd, resp.Body); err != nil {
			_ = fd.Close()
			_ = resp.Body.Close()
			continue
		}
		_ = fd.Close()
		_ = resp.Body.Close()
		return nil
	}
	return fmt.Errorf("error exec query %s", query)
}

func main() {
	flag.StringVar(&confile, "config", CONFILE, "config for connect")
	flag.StringVar(&tmpfile, "tmpfile", TMPFILE, "tmpfile for dump metricsearch")
	flag.BoolVar(&force, "force", false, "force initialization")
	flag.BoolVar(&debug, "debug", false, "debug mode")
	flag.Parse()

	fileLock := flock.New(LOCKFILE)

	locked, err := fileLock.TryLock()
	if err != nil {
		log.Fatalf("error trylock: %s", err)
	}

	if !locked {
		if debug {
			log.Printf("cannot get lock for %s", LOCKFILE)
		}
		os.Exit(1)
	}

	defer func() { _ = fileLock.Unlock() }()

	if !force {
		if _, err := os.Stat(FIRSTINIT); err == nil {
			if debug {
				log.Printf("exists first initialization file %s, exit", FIRSTINIT)
			}
			os.Exit(1)
		}
	}

	cnf, err := LoadConfig(confile)
	if err != nil {
		log.Fatalf("%s", err)
	}

	err = ExecQuery(QUERY, cnf.Config, tmpfile)
	if err != nil {
		log.Fatalf("error download dump %s: %s", tmpfile, err)
	}

	if err = ioutil.WriteFile(FIRSTINIT, []byte("DONE"), 0644); err != nil {
		log.Fatalf("error saved file %s: %s", FIRSTINIT, err)
	}

	cmd := exec.Command("/usr/sbin/metricsearch", fmt.Sprintf("-reindex=%s", tmpfile))
	if err := cmd.Run(); err != nil {
		log.Fatalf("error reindes metricsearch: %s", err)
	}

	_ = os.Remove(tmpfile)

	cmd = exec.Command("/etc/init.d/metricsearch", "restart")
	if err := cmd.Run(); err != nil {
		log.Fatalf("error restart metricsearch: %s", err)
	}

	fmt.Printf("FINISH")
}
