package delta

import (
	"bufio"
	"fmt"
	"net"
	"net/http"
	"strings"
	"time"

	"a.yandex-team.ru/direct/infra/dt-graphite-ch-cacher/internal/mylib"
	"a.yandex-team.ru/direct/infra/dt-graphite-ch-cacher/internal/siphash"
)

type CacheItem struct {
	metric  string
	storage string
}

var (
	key0, key1 uint64
)

func deltaSender(metricSearch string, delta chan CacheItem) {
	for {
		item := <-delta
		transport := http.Transport{
			Dial: mylib.DialTimeout,
		}
		client := http.Client{
			Transport: &transport,
		}

		for retry := 0; retry < 3; retry++ {
			resp, err := client.Get("http://" + metricSearch + ":7000/add?name=" + item.metric)
			if err != nil {
				fmt.Printf("Error: failed to add metric %s, retries left %d,  err [ %v ]", item, retry, err)
				continue
			}
			if resp.StatusCode != 200 {
				fmt.Printf("Error: failed to add metric %s, retries left %d, err [ %v ]", item, retry, err)
				continue
			}
			fmt.Printf("Debug: server %s added metric %s\n", metricSearch, item)
			_ = resp.Body.Close()
			break
		}
	}
}

// it is the same function as in mylib, but deadline is set to 10min
func DialTimeoutLong(network, addr string) (net.Conn, error) {
	c, err := net.DialTimeout(network, addr, time.Duration(60*time.Second))
	if err != nil {
		return c, err
	}
	_ = c.SetDeadline(time.Now().Add(time.Duration(600 * time.Second)))
	return c, err
}

func Wrap(err error) {
	if err != nil {
		fmt.Printf("[wrapper] error: %s\n", err)
	}
}

func loadCache(metricSearch string, senders []mylib.Sender, cache map[uint64]int) {
	// load cache from file
	resp, err := http.Get("http://" + metricSearch + ":7000/dump")
	if err != nil {
		fmt.Printf("Error: failed to get index from metricsearch, err [ %v ]", err)
		return
	}
	defer func() { _ = resp.Body.Close() }()

	scanner := bufio.NewScanner(resp.Body)
	for scanner.Scan() {
		m := strings.TrimSpace(scanner.Text())
		if m == "" {
			continue
		}
		mhash := siphash.Hash(key0, key1, []byte(m))
		cache[mhash] = 1
	}
	if err := scanner.Err(); err != nil {
		fmt.Printf("something went wrong while scanning through the index, err %v", err)
	}
}

func DeltaManager(metrics chan string, senders []mylib.Sender, metricServers []string, boss mylib.Boss) {
	metricaSender := make(map[string]chan CacheItem)
	for _, serverName := range metricServers {
		if len(serverName) == 0 {
			continue
		}
		server := serverName
		delta := make(chan CacheItem, 100000)
		metricaSender[server] = delta
		go deltaSender(server, delta)
	}
	//delta := make(chan CacheItem, 100000)
	cache := make(map[uint64]int)
	//go deltaSender(metricSearch, delta)
	fmt.Printf("loading cache\n")
	loadCache(metricServers[0], senders, cache)
	fmt.Printf("loaded %d\n", len(cache))
	for {
		m := <-metrics
		mhash := siphash.Hash(key0, key1, []byte(m))
		_, ok := cache[mhash]
		if !ok {
			// every new metric in deltaManager must have a storage
			storage := ""
			// if there is a single storage everything is easy
			if boss.Single == 1 {
				storage = boss.Senders[0].Host
			} else {
				// but sometimes we can use multiply storage, than it's a hashring task to choose the right one
				r, err := boss.Ring.GetN(m, boss.Rf)
				if err != nil {
					fmt.Printf("Failed to get caches for metric %s, err %v\n", m, err)
					continue
				}
				if len(r) > 0 {
					// daleta manager doesn't inquire about replication and awlays will show the first
					// storage in the ring, dublication because of replication is handled by main sphinx index - path
					storage = r[0]
				} else {
					fmt.Println("Failed to get storage for some reason :(")
				}
			} // single storage vs multistorage
			for server, delta := range metricaSender {
				select {
				case delta <- CacheItem{m, storage}:
					fmt.Printf("send metric to %s\n", server)
				default:
					continue
				}
			}
			cache[mhash] = 1
		} // if !ok
	}
}

func BogusDelta(input chan string) {
	for {
		<-input
	}
}
