package mylib

import (
	"fmt"
	"io/ioutil"
	"net"
	"os"
	"strconv"
	"strings"
	"time"

	"gopkg.in/ini.v1"
)

type Storage struct {
	Host string
	Port int
	Num  int
}

type Config struct {
	LogLevel    string `ini:"logLevel"`
	Storage     string `ini:"storage"`
	Storages    []*Storage
	ChanLimit   int    `ini:"chanLimit"`
	Rf          int    `ini:"Rf"`
	EnableDelta int    `ini:"enableDelta"`
	User        string `ini:"user"`
	Token       string `ini:"token"`
	Password    string `ini:"-"`
	Database    string `ini:"database"`
}

type configFile struct {
	Main Config
}

const defaultConfig = `
    [main]
    logLevel = 10
    storage = 127.0.0.1:8123:1
    Rf = 1
    user = default
    token =
    database = default
`

func parseStorages(storage string) []*Storage {
	stArr := strings.Split(storage, ",")
	storages := make([]*Storage, 0)
	for _, val := range stArr {
		if len(val) < 5 {
			fmt.Printf("storage definition must be atleast 5 chars: a:1:1, %s\n", val)
			continue
		}
		val = strings.Trim(val, " ")
		stOpts := strings.Split(val, ":")
		if len(stOpts) < 3 {
			fmt.Printf("Bad format for storage, not enought options (host:port:num): %s\n", val)
			os.Exit(1)
		}
		host := stOpts[0]
		port, err := strconv.ParseInt(stOpts[1], 0, 32)
		if err != nil {
			fmt.Printf("Bad format for storage definition, port must be int, %s\n", val)
			os.Exit(1)
		}
		num, err := strconv.ParseInt(stOpts[2], 0, 32)
		if err != nil {
			fmt.Printf("Bad format for storage definition, num must be int, %s\n", val)
			os.Exit(1)
		}
		// convert 0,1,2.. to 1,2,3
		num = num - 1
		storages = append(storages, &Storage{host, int(port), int(num)})
	}
	return storages
}

type INI struct {
	Main Config `ini:"main"`
}

func Load(cfgFile string) Config {
	var err error
	var content []byte
	var fileIni *ini.File
	//var section *ini.Section
	//var key *ini.Key

	if cfgFile != "" {
		if content, err = ioutil.ReadFile(cfgFile); err != nil {
			fmt.Printf("Failed read file %s, %v\n", cfgFile, err)
			os.Exit(2)
		}
		fileIni, err = ini.Load(content)
	} else {
		fileIni, err = ini.Load([]byte(defaultConfig))
	}
	if err != nil {
		fmt.Printf("Failed to init config, %v\n", err)
		os.Exit(1)
	}

	cfg := INI{}

	if err = fileIni.MapTo(&cfg); err != nil {
		fmt.Printf("Failed parse init config, %v\n", err)
		os.Exit(2)
	}

	cfg.Main.Storages = parseStorages(cfg.Main.Storage)

	if len(cfg.Main.Token) > 0 {
		if data, err := ioutil.ReadFile(cfg.Main.Token); err == nil {
			cfg.Main.Password = strings.Trim(string(data), "\n")
		} else {
			fmt.Printf("Failed read token %s: %s\n", cfg.Main.Token, err)
			os.Exit(2)
		}
	}

	fmt.Printf("CONFIG %v\n", cfg.Main)

	return cfg.Main
}

func DialTimeout(network, addr string) (net.Conn, error) {
	c, err := net.DialTimeout(network, addr, time.Duration(180*time.Second))
	if err != nil {
		return c, err
	}
	_ = c.SetDeadline(time.Now().Add(time.Duration(180 * time.Second)))
	return c, err
}
