package main

import (
	"encoding/json"
	"flag"
	"fmt"
	"log"
	"os"
	"strings"
	"time"

	api "a.yandex-team.ru/direct/infra/dt-haproxy-cloud/internal/cloudapi"
	"a.yandex-team.ru/direct/infra/dt-haproxy-cloud/internal/config"
)

var (
	configPath, sourceShardName, destinationShardName, envType, taskID string //путь до конфига с настройками подключения
	debug, help, checkDiskStatus, duplicate, waitTask                  bool
	err                                                                error
)

const (
	Address                = "http://localhost:8085/%s"
	DefaultSourceShardName = "shard001"
	DefaultEnvPath         = "/etc/yandex/environment.type"
)

//создание шарда по аналогии с текущим
func RunDuplicateShard(cloudKey, clusterID, sourceShardName, destinationShardName string) {
	token, err := api.NewCloudToken(cloudKey)
	if err != nil {
		log.Fatalf("error token %s: %s\n", cloudKey, err)
	}

	task, err := token.RequestDuplicateShard(clusterID, sourceShardName, destinationShardName)
	if err != nil {
		log.Fatalf("error creating a new shard %s: %s\n", destinationShardName, err)
	}

	log.Printf("in cluster %s, a task has been started with taskid: %s\n", clusterID, task.ID)

	if waitTask {
		log.Println("monitoring of task execution starts. From now on, the script can be interrupted.")
		MonitoringCloudTask(cloudKey, task.ID)
	}
}

func PrintCloudTaskStatus(cloudKey, taskID string) (finish bool) {
	token, err := api.NewCloudToken(cloudKey)
	if err != nil {
		log.Fatalf("error token %s: %s\n", cloudKey, err)
		return true
	}
	for i := 0; i < 3; i++ {
		task, err := token.RequestOperationID(taskID)
		if err != nil {
			log.Printf("error task status: %s, count retry: %d", err, i)
			continue
		}
		if task.Done {
			if task.Error.Code == 0 {
				log.Printf("task completed successfully(taskid: %s, done: %t, code: %d)\n", task.ID, task.Done, task.Error.Code)
			} else {
				log.Printf("task completed with errors(taskid: %s, done: %t, code: %d)\n", task.ID, task.Done, task.Error.Code)
				log.Println(task.Error.Message, task.Error.Details)
			}
			return true
		} else {
			log.Printf("the task is still in progress(taskid: %s, done: %t)\n", task.ID, task.Done)
		}
		break
	}
	return
}

//мониторинг запущенных задач
func MonitoringCloudTask(cloudKey, taskID string) {
	ticker := time.NewTicker(10 * time.Second)
	for range ticker.C {
		if finish := PrintCloudTaskStatus(cloudKey, taskID); finish {
			break
		}
	}
}

func main() {
	flag.BoolVar(&checkDiskStatus, "disk-status", false, "показать статистику по дискам в кластере")
	flag.StringVar(&configPath, "config", "", "путь до конфига с настроками")
	flag.BoolVar(&debug, "debug", false, "дебаг режим")
	flag.StringVar(&sourceShardName, "source-shard", DefaultSourceShardName, "имя шарда, с которого будет копирование настроек")
	flag.StringVar(&destinationShardName, "destination-shard", "", "имя шарда, который будет создан")
	flag.StringVar(&envType, "enviroment-type", "", "тип среды: production, testing, dev")
	flag.BoolVar(&duplicate, "duplicate-shard", false, "создать дубликат шарда в кластере")
	flag.BoolVar(&help, "help", false, "показать списки команд с примерами")
	flag.BoolVar(&waitTask, "wait-task", false, "ждать выполнения задачи в облаке")
	flag.StringVar(&taskID, "taskid", "", "показать статус выполнения задачи")
	flag.Parse()

	api.SetDebug(debug)

	if help {
		flag.PrintDefaults()
		fmt.Println(helptext)
		return
	}

	if len(envType) == 0 {
		err = config.LoadEnviromentType(DefaultEnvPath)
	} else {
		err = os.Setenv("ENVTYPE", strings.ToLower(envType))
	}
	if err != nil {
		log.Fatalf("error set ENVTYPE (current: %s): %s", envType, err)
	}

	if len(configPath) == 0 {
		if configPath, err = config.SearchConfigByEnv("clickhouse", "manager"); err != nil {
			log.Fatalf("%s", err)
		}
	}

	cnf, err := config.LoadConfig(configPath)
	if err != nil {
		log.Fatalf("error loading config %s: %s", configPath, err)
	}

	if checkDiskStatus {
		var diskStatus api.SpaceDiskByHosts
		method := fmt.Sprintf(Address, "used-space-statistics")
		resp := api.Send(method, "", "GET", nil)
		if resp.Err != nil {
			log.Fatalf("error request %s: %s", method, err)
		}
		if err := json.Unmarshal(resp.Data, &diskStatus); err != nil {
			log.Fatalln(err)
		}
		diskStatus.Print()
	}
	if duplicate {
		RunDuplicateShard(cnf.CloudKey, cnf.ClusterID, sourceShardName, destinationShardName)
	}
	if len(taskID) > 0 {
		_ = PrintCloudTaskStatus(cnf.CloudKey, taskID)
	}
}

var helptext string = fmt.Sprintf(`
Клиент для более удобного управления кластером mdb-clickhouse.

Примеры команд:
1. Просмотр занятого места по шардам:
 %[1]s -disk-status
2. Добавить новый шард с именем shard028 по аналогии с существующим(по умолчанию берется shard001). Выводится taskid задачи.
 %[1]s -duplicate-shard -destination-shard shard028
или
 %[1]s -duplicate-shard -destination-shard shard028 -source-shard shard001
3. Добавить новый шард и ждать выполнения задачи
 %[1]s -duplicate-shard -destination-shard shard028 -wait-task
4. Проверить статус выполнения задачи
 %[1]s -taskid 8dsdsadddfejhh
`, os.Args[0])
