package main

import (
	"bytes"
	"fmt"
	"io/ioutil"
	"os"
	"os/exec"
	"strconv"
	//"syscall"

	"a.yandex-team.ru/direct/infra/dt-haproxy-cloud/internal/cloudapi"
	"a.yandex-team.ru/direct/infra/dt-haproxy-cloud/internal/config"
	"a.yandex-team.ru/direct/infra/dt-haproxy-cloud/internal/mylog"
)

var block1 = `
global
	log /dev/log	local0
	log /dev/log	local1 notice
	chroot /var/lib/haproxy
	stats socket /run/haproxy/admin.sock mode 660 level admin
	stats timeout 30s
	user ppc
	group ppc
	daemon

	# Default SSL material locations
	ca-base /etc/ssl/certs
	crt-base /etc/ssl/private

	tune.ssl.default-dh-param 2048

	# Default ciphers to use on SSL-enabled listening sockets.
	# For more information, see ciphers(1SSL). This list is from:
	#  https://hynek.me/articles/hardening-your-web-servers-ssl-ciphers/
	ssl-default-bind-ciphers ECDH+AESGCM:DH+AESGCM:ECDH+AES256:DH+AES256:ECDH+AES128:DH+AES:ECDH+3DES:DH+3DES:RSA+AESGCM:RSA+AES:RSA+3DES:!aNULL:!MD5:!DSS
	ssl-default-bind-options no-sslv3

defaults
	log	global
	mode	http
	option	httplog
	option	dontlognull
        timeout connect 5000
        timeout client  500000
        timeout server  500000
	errorfile 400 /etc/haproxy/errors/400.http
	errorfile 403 /etc/haproxy/errors/403.http
	errorfile 408 /etc/haproxy/errors/408.http
	errorfile 500 /etc/haproxy/errors/500.http
	errorfile 502 /etc/haproxy/errors/502.http
	errorfile 503 /etc/haproxy/errors/503.http
	errorfile 504 /etc/haproxy/errors/504.http

frontend mdb-clickhouse-write-https
	mode tcp
	bind :::%d ssl crt %s
	option tcplog
	use_backend nodes-write

frontend mdb-clickhouse-write-http
	mode tcp
	bind :::%d
	option tcplog
	use_backend nodes-write

backend nodes-write
	#option ssl-hello-chk
	balance roundrobin
`

var block2 = `
frontend mdb-clickhouse-read-https
	mode tcp
	bind :::%d ssl crt %s
	option tcplog
	use_backend nodes-read

frontend mdb-clickhouse-read-http
	mode tcp
	bind :::%d
	option tcplog
	use_backend nodes-read

backend nodes-read
	#option ssl-hello-chk
	balance roundrobin
`

var body = "	server clickhouse-%d %s:%[3]d ssl check port %[3]d inter 12000 rise 5 fall 5 ca-file %s\n"

//генерирует haproxy конфиг и записывает результат в файл по пути path
func generateHaproxyConfig(pool cloudapi.StatusHosts, path string) (err error) {
	if monitoring {
		return fmt.Errorf("skip generate haproxy config because used -monitoring flag")
	}
	writeHosts := cloudapi.ClickhouseReadyWriteHosts(pool, cnf.MaxPercentUsedSpace)
	readHosts := cloudapi.ClickhouseReadyReadHosts(pool)
	if len(writeHosts) == 0 && len(readHosts) == 0 {
		return fmt.Errorf("ignore saved haproxy config because have empty host list")
	}
	var buffer bytes.Buffer
	//если указан сертификат, то добавляем запись cert <name cert>
	buffer.WriteString(fmt.Sprintf(block1, config.CHHTTPS, cnf.Certificate, config.CHHTTP))
	var i int
	for hostname := range writeHosts {
		buffer.WriteString(fmt.Sprintf(body, i, hostname, config.CHHTTPS, cnf.CAcert))
		i++
	}
	buffer.WriteString(fmt.Sprintf(block2, config.ROHTTPS, cnf.Certificate, config.ROHTTP))
	var j int
	for hostname := range readHosts {
		buffer.WriteString(fmt.Sprintf(body, j, hostname, config.CHHTTPS, cnf.CAcert))
		j++
	}
	fd, err := os.OpenFile(path, os.O_CREATE|os.O_WRONLY|os.O_TRUNC, 0644)
	if err != nil {
		return fmt.Errorf("error open file %s: %s", path, err)
	}
	num, err := fd.Write(buffer.Bytes())
	if err != nil {
		return err
	}
	mylog.Notice("saved %d bytes in file %s\n", num, path)
	return
}

//отправляет hup сигнал haproxy по пути pid файла
func reloadHaproxy(path string) (err error) {
	if monitoring {
		return fmt.Errorf("skip restart haproxy config because used -monitoring flag")
	}
	data, err := ioutil.ReadFile(path)
	if err != nil {
		mylog.Warn("error read file %s: %s\n", path, err)
	}
	if err != nil || len(data) == 0 {
		mylog.Warn("empty haproxy pid file %s", path)
		cmd := exec.Command("sudo", "/etc/init.d/haproxy", "restart")
		err := cmd.Run()
		if err != nil {
			return fmt.Errorf("restart haproxy daemon are failed: %s", err)
		}
	}
	data = bytes.Trim(data, "\n")
	pid, err := strconv.ParseInt(string(data), 10, 64)
	if err != nil {
		return fmt.Errorf("data: %s, error: %s", data, err)
	}
	//err = syscall.Kill(int(pid), syscall.SIGUSR1)
	cmd := exec.Command("sudo", "/etc/init.d/haproxy", "reload")
	if err := cmd.Run(); err != nil {
		mylog.Warn("failed reload haproxy for pid: %d, error: %s\n", pid, err)
		cmd := exec.Command("sudo", "/etc/init.d/haproxy", "restart")
		err := cmd.Run()
		mylog.Warn("try restart haproxy")
		if err != nil {
			mylog.Warn("failed restart haproxy, error: %s\n", err)
			return fmt.Errorf("failed reload and restart haproxy: %s", err)
		}
		mylog.Notice("restart haproxy success\n")
	} else {
		mylog.Notice("reload haproxy with pid %d are success\n", pid)
	}
	return
}
