package main

import (
	"encoding/json"
	"flag"
	"net/http"
	"os"
	"os/signal"
	"syscall"
	"time"

	cloud "a.yandex-team.ru/direct/infra/dt-haproxy-cloud/internal/cloudapi"
	"a.yandex-team.ru/direct/infra/dt-haproxy-cloud/internal/config"
	"a.yandex-team.ru/direct/infra/dt-haproxy-cloud/internal/mylog"
)

var (
	confpath          string
	debug, monitoring bool
	cnf               config.Config
	err               error
)

//мануал по работе демона: https://wiki.yandex-team.ru/users/dspushkin/logshatter-in-mds/
func main() {
	flag.BoolVar(&debug, "debug", false, "debug mode")
	flag.StringVar(&confpath, "config", "", "путь до конфига с настроками")
	flag.BoolVar(&monitoring, "monitoring", false, "только мониторить состояние, не генерировать haproxy конфиг")
	flag.Parse()

	if debug {
		_ = os.Setenv("DEBUG", "true")
	}

	mylog.LoadRsyslogConfig()
	mylog.Notice("start daemon at %s\n", time.Now())
	defer func() { mylog.Notice("stop daemon at %s\n", time.Now()) }()

	if err = config.LoadEnviromentType(config.ENVFILE); err != nil {
		mylog.Crit("error load enviroment: %s", err)
	}

	if len(confpath) == 0 {
		if confpath, err = config.SearchConfigByEnv("clickhouse", "viewer"); err != nil {
			mylog.Crit("error search config byenv: %s", err)
		}
	}

	mylog.Debug("load config %s\n", confpath)

	if cnf, err = config.LoadConfig(confpath); err != nil {
		mylog.Crit("%s\n", err)
		mylog.Crit("use default config %+v\n", cnf)
	}
	mylog.Notice("loading config: %+v", cnf)

	if err := cnf.LoadPassword(); err != nil {
		mylog.Crit("%s", err)
		os.Exit(1)
	}

	mylog.Debug("config %s settings: %+v\n", confpath, cnf)

	enableTLSClickhouse()
	token, err := cloud.NewCloudToken(cnf.CloudKey)
	if err != nil {
		mylog.Crit("error read CloudKey %s: %s\n", cnf.CloudKey, err)
		os.Exit(1)
	}

	cluster := cloud.NewClusterHosts()
	go cloud.UpdateClusterHosts(token, cluster, cnf.ClusterID, "clickhouse")

	pool := make(cloud.StatusHosts)
	sigs := make(chan os.Signal, 1)
	defer func() { close(sigs) }()

	signal.Notify(sigs, syscall.SIGHUP)

	go UpdateStatusHosts(cluster, pool, cnf)

	go MonitoringReadyHosts(pool, cnf)

	aliveStatus := cloud.NewAliveStatus()
	go RunAliveChecks(pool, cnf, &aliveStatus)

	haproxyRestartStatus := cloud.NewCodeWithMessage(0, "")

	if len(cnf.Certificate) == 0 {
		go StartNativeProxy(cloud.NewCHWriterHosts(pool, "write", cnf.MaxPercentUsedSpace), config.CHHTTPS)
		go StartNativeProxy(cloud.NewCHReaderHosts(pool, "read"), config.ROHTTPS)
	} else {
		go RunMonitoringHaproxy(pool, sigs, &haproxyRestartStatus)
	}

	http.HandleFunc("/ready-write-hosts", func(w http.ResponseWriter, r *http.Request) {
		if err := json.NewEncoder(w).Encode(cloud.ClickhouseReadyWriteHosts(pool, cnf.MaxPercentUsedSpace).Hostnames()); err != nil {
			mylog.Warn("method /ready-write-hosts, error: %s\n", err)
		}
	})

	http.HandleFunc("/ready-read-hosts", func(w http.ResponseWriter, r *http.Request) {
		if err := json.NewEncoder(w).Encode(cloud.ClickhouseReadyReadHosts(pool).Hostnames()); err != nil {
			mylog.Warn("method /ready-read-hosts, error: %s\n", err)
		}
	})

	http.HandleFunc("/alive-pool", func(w http.ResponseWriter, r *http.Request) {
		if err := json.NewEncoder(w).Encode(map[string]bool{"alive": len(pool) > 0}); err != nil {
			mylog.Warn("method /alive-pool, error: %s\n", err)
		}
	})

	http.HandleFunc("/cluster", func(w http.ResponseWriter, r *http.Request) {
		if err := json.NewEncoder(w).Encode(cluster.Hostnames()); err != nil {
			mylog.Warn("method /cluster, error: %s\n", err)
		}
	})

	http.HandleFunc("/haproxy-restart-status", func(w http.ResponseWriter, r *http.Request) {
		if err := json.NewEncoder(w).Encode(haproxyRestartStatus); err != nil {
			mylog.Warn("method /haproxy-restart-status, error: %s\n", err)
		}
	})

	http.HandleFunc("/used-space-statistics", func(w http.ResponseWriter, r *http.Request) {
		if err := json.NewEncoder(w).Encode(MonitoringSpaceDisk(cluster, pool)); err != nil {
			mylog.Warn("method /used-space-statistics, error: %s\n", err)
		}
	})

	http.HandleFunc("/alive", func(w http.ResponseWriter, r *http.Request) {
		if _, err := w.Write(aliveStatus.GetAliveStatus()); err != nil {
			mylog.Warn("method /ping, error: %s\n", err)
		}
	})

	serv := &http.Server{
		Addr:         ":8085",
		WriteTimeout: 30 * time.Second,
		ReadTimeout:  30 * time.Second,
	}

	go StartNativeProxy(cloud.NewCHReaderHosts(pool, "read"), config.CHPORT) //запускаем проксирование 9440

	if err := serv.ListenAndServe(); err != nil {
		mylog.Crit("http error: %s\n", err)
		os.Exit(1)
	}
}
