package main

import (
	"crypto/tls"
	"database/sql"
	"fmt"
	"os"
	"strings"
	"syscall"
	"time"

	clickhouse "github.com/ClickHouse/clickhouse-go"

	cloud "a.yandex-team.ru/direct/infra/dt-haproxy-cloud/internal/cloudapi"
	"a.yandex-team.ru/direct/infra/dt-haproxy-cloud/internal/config"
	"a.yandex-team.ru/direct/infra/dt-haproxy-cloud/internal/mylog"
)

//запрос на подсчет места, занимаемого данными в CH
var request1 string = "SELECT table, sum(bytes_on_disk) AS used FROM system.parts GROUP BY table"

//запрос пока не используется, но через него можно получать списки detached кусочков.
var request2 string = "SELECT table, count(*) AS count FROM system.detached_parts GROUP BY table"

//запрос на получение свободного и общего места, занимаемого данными в CH
var request3 string = "SELECT free_space, total_space FROM system.disks WHERE name='default' "

func NewConnectClickhouse(c config.Config, host string) (conn *sql.DB, err error) {
	address := fmt.Sprintf("tcp://%s:%d?compress=true&debug=%t&username=%s&database=%s&password=%s&tls_config=custom",
		host, config.CHPORT, debug, c.User, c.Database, *c.Password)
	mylog.Debug("conect to %s", address)
	return sql.Open("clickhouse", address)
}

func UpdateStatusHosts(cluster cloud.ClusterHosts, pool cloud.StatusHosts, cnf config.Config) {
	ticker := time.NewTicker(5 * time.Second)
	for range ticker.C {
		for _, host := range *cluster.Hosts {
			//пропускаем zookeeper
			if host.Type == "ZOOKEEPER" {
				continue
			}
			if _, ok := pool[host.Name]; !ok {
				conn, err := NewConnectClickhouse(cnf, host.Name)
				if err != nil {
					*pool[host.Name].Status = "ERRCONNECT"
				}
				pool[host.Name] = cloud.NewStatusHost(host, conn)
			} else {
				*pool[host.Name].ClusterHost = host
			}
		}

		//список имен машин
		hostnames := cluster.Hostnames()
		for host, value := range pool {
			if debug {
				fmt.Println(host, value)
			}
			*pool[host].ModifyTime = time.Now().Unix()
			if !hostnames.HasHost(host) {
				if err := (*pool[host].Conn).Close(); err != nil {
					mylog.Warn("close connect to %s: %s\n", host, err)
				}
				delete(pool, host)
				continue
			}
			//пропускаем мертвые машины
			if !strings.Contains(strings.ToUpper(value.Health), "ALIVE") && len(value.Health) != 0 {
				mylog.Crit("host %s are died, cluster health: %s\n", host, value.Health)
				*pool[host].Status = "DEAD" //очень часто срабатывает DIED, пробую без него
				continue
			}
			//пропускаем плохо работающие машины
			if len(cnf.ExcludeServers) > 0 && strings.Contains(cnf.ExcludeServers, host) {
				mylog.Crit("host %s are skipped, because has skip section in config: %s\n", host, cnf.ExcludeServers)
				*pool[host].Status = "SKIP"
				continue
			}
			//пропускаем ping для неустановленных соединений
			err := ping(value.Conn)
			if err != nil {
				mylog.Crit("error ping() for host %s: %s\n", host, err)
				*pool[host].Status = "ERRPING"
				continue
			}
			*pool[host].Status = "OK"

			statTables, err := RunStatisticTables(value.Conn)
			if err != nil {
				mylog.Crit("%s", err)
			} else {
				*pool[host].StatTables = statTables
			}

			statDisk, err := RunStatisticSpaceDisk(value.Conn)
			if err != nil {
				mylog.Crit("%s", err)
			} else {
				*pool[host].StatSpace = statDisk
			}
		}
	}
}

//включаем поддерку SSL у Clickhouse
func enableTLSClickhouse() {
	rootCertPool, err := cloud.CreateCertPool(cnf.CAcert)
	if err != nil {
		mylog.Crit("error create CA pool: %s\n", err)
	}
	err = clickhouse.RegisterTLSConfig("custom", &tls.Config{
		RootCAs: rootCertPool,
	})
	if err != nil {
		mylog.Crit("error register tsl config: %s\n", err)
	}
}

//Проверяет хосты на наличие новых или удаленных. Если таковые имеются - то отправляется HUP сигнал самому себе.
func MonitoringReadyHosts(pool cloud.StatusHosts, cnf config.Config) {
	var readyHosts []cloud.ActivePool

	readyHosts = append(readyHosts, cloud.NewCHWriterHosts(pool, "write", cnf.MaxPercentUsedSpace))
	readyHosts = append(readyHosts, cloud.NewCHReaderHosts(pool, "read"))

	ticker := time.NewTicker(10 * time.Second)

	for range ticker.C {
		sendHup := false
		for _, readyHost := range readyHosts {
			var removed, added cloud.Hostnames
			currentHosts := readyHost.ActivePool().Hostnames()
			for _, h1 := range currentHosts {
				if !readyHost.LastPool().HasHost(h1) {
					added = append(added, h1)
				}
			}
			lastHosts := readyHost.LastPool().Hostnames()
			for _, h1 := range lastHosts {
				if !currentHosts.HasHost(h1) {
					removed = append(removed, h1)
				}
			}

			mylog.Debug("current %v hosts: %v\n", readyHost.Group(), currentHosts)
			mylog.Debug("last hosts: %v\n", lastHosts)
			if len(removed) > 0 {
				mylog.Notice("deleted %v hosts: %v\n", readyHost.Group(), removed)
			}
			if len(added) > 0 {
				mylog.Notice("added %v hosts: %v\n", readyHost.Group(), added)
			}

			if len(removed) > 0 || len(added) > 0 {
				sendHup = true
			}
			readyHost.SetLastPool(readyHost.ActivePool())
		}

		if sendHup {
			err := syscall.Kill(syscall.Getpid(), syscall.SIGHUP)
			if err != nil {
				mylog.Crit("error send HUP signal: %s\n", err)
			}
		}
	}
}

//Проверяет изменение текущих машин и генерирует новый haproxy конфиг
func RunMonitoringHaproxy(pool cloud.StatusHosts, sigs chan os.Signal, output *cloud.CodeWithMessage) {
	for {
		<-sigs
		mylog.Notice("generate new haproxy config %s\n", cnf.HaproxyPath)
		if err := generateHaproxyConfig(pool, cnf.HaproxyPath); err != nil {
			msg := fmt.Sprintf("error generateHaproxyConfig %s: %s\n", cnf.HaproxyPath, err)
			mylog.Crit(msg)
			*output = cloud.NewCodeWithMessage(1, msg)
			continue
		}
		if err := reloadHaproxy(cnf.HaproxyPid); err != nil {
			msg := fmt.Sprintf("error reloadHaproxy: %s", err)
			mylog.Crit(msg)
			*output = cloud.NewCodeWithMessage(2, msg)
			select {
			case sigs <- syscall.SIGHUP:
				time.Sleep(5 * time.Second)
			default:
			}
			continue
		}
		*output = cloud.NewCodeWithMessage(0, "reload haproxy success")
	}
}

//Проверка живости БД(SELECT 1)
func ping(conn *sql.DB) (err error) {
	err = conn.Ping()
	if exception, ok := err.(*clickhouse.Exception); ok {
		err = fmt.Errorf("[%d] %s %s", exception.Code, exception.Message, exception.StackTrace)
	}
	return
}

//Получение занятого места по таблицам БД
func RunStatisticTables(conn *sql.DB, table ...cloud.Table) (result cloud.StatisticTables, err error) {
	rows, err := conn.Query(request1)
	if err != nil {
		return
	}

	var meta cloud.StatisticTable
	var tables cloud.Tables
	for _, t := range table {
		tables = append(tables, t)
	}
	for rows.Next() {
		err := rows.Scan(&meta.Table, &meta.Used)
		if err != nil {
			mylog.Warn("error scan: %s", err)
			continue
		}
		if len(tables) > 0 && !tables.HasAny(meta.Table) {
			continue
		}
		result = append(result, meta)
	}
	return
}

func RunStatisticSpaceDisk(conn *sql.DB) (result cloud.StatisticSpace, err error) {
	row := conn.QueryRow(request3)

	err = row.Scan(&result.FreeSpace, &result.TotalSpace)
	if err != nil {
		mylog.Warn("error scan: %s", err)
	}
	return
}

func MonitoringSpaceDisk(cluster cloud.ClusterHosts, pool cloud.StatusHosts) (result cloud.SpaceDiskByHosts) {
	for _, host := range *cluster.Hosts {
		if strings.ToUpper(host.Type) != "CLICKHOUSE" {
			continue
		}
		status, ok := pool[host.Name]
		if !status.IsReady() {
			continue
		}
		if !ok {
			result = append(result,
				cloud.NewSpaceDiskByHost(
					host.Name,
					host.ShardName,
					host.GetDiskSize(),
					cloud.Bytes(0),
					cloud.Bytes(0),
				))
			continue
		}
		result = append(result,
			cloud.NewSpaceDiskByHost(
				host.Name,
				host.ShardName,
				host.GetDiskSize(),
				(*status.StatSpace).Used(),
				(*status.StatTables).TotalUsed(),
			))
	}
	return
}
