package main

import (
	"flag"
	"net/http"
	"os"
	"os/signal"
	"syscall"
	"time"

	cloud "a.yandex-team.ru/direct/infra/dt-haproxy-cloud/internal/cloudapi"
	"a.yandex-team.ru/direct/infra/dt-haproxy-cloud/internal/config"
	"a.yandex-team.ru/direct/infra/dt-haproxy-cloud/internal/mylog"
)

var (
	debugMode, help     bool
	configPath, envType string
	err                 error
	cnf                 config.Config
)

const (
	DefaultEnvPath = "/etc/yandex/environment.type"
)

func main() {
	flag.BoolVar(&debugMode, "debug", false, "включить дебаг режим")
	flag.StringVar(&configPath, "config", "", "путь до конфига")
	flag.BoolVar(&help, "help", false, "показать списки команд с примерами")
	flag.StringVar(&envType, "enviroment-type", "", "тип среды: production, testing, dev")
	flag.Parse()

	if debugMode {
		_ = os.Setenv("DEBUG", "true")
	}

	if err = config.LoadEnviromentType(config.ENVFILE); err != nil {
		mylog.Crit("error load enviroment: %s", err)
	}

	mylog.LoadRsyslogConfig()
	mylog.Notice("start daemon at %s\n", time.Now())
	defer func() { mylog.Notice("stop daemon at %s\n", time.Now()) }()

	if len(configPath) == 0 {
		if configPath, err = config.SearchConfigByEnv("mysql", "viewer"); err != nil {
			mylog.Crit("error search config by env: %s", err)
		}
	}

	mylog.Debug("load config %s\n", configPath)

	if cnf, err = config.LoadConfig(configPath); err != nil {
		mylog.Crit("%s\n", err)
		mylog.Crit("use default config %+v\n", cnf)
	}
	mylog.Notice("loading config: %+v", cnf)

	if err := cnf.LoadPassword(); err != nil {
		mylog.Crit("%s", err)
		os.Exit(1)
	}

	mylog.Debug("config %s settings: %+v\n", configPath, cnf)

	token, err := cloud.NewCloudToken(cnf.CloudKey)
	if err != nil {
		mylog.Crit("error read CloudKey %s: %s\n", cnf.CloudKey, err)
		os.Exit(1)
	}

	cluster := cloud.NewClusterHosts()
	go cloud.UpdateClusterHosts(token, cluster, cnf.ClusterID, "mysql")

	pool := make(cloud.StatusHosts)
	sigs := make(chan os.Signal, 1)
	defer func() { close(sigs) }()

	signal.Notify(sigs, syscall.SIGHUP)

	go UpdateStatusHosts(cluster, pool)
	go MonitoringReadyHosts(pool)

	serv := &http.Server{
		Addr:         ":8086",
		WriteTimeout: 30 * time.Second,
		ReadTimeout:  30 * time.Second,
	}

	go StartNativeProxy(cloud.NewMySQLAliveHosts(pool, "write"), config.MYSQLPORT) //запускаем проксирование 9440

	if err := serv.ListenAndServe(); err != nil {
		mylog.Crit("http error: %s\n", err)
		os.Exit(1)
	}
}
