package main

import (
	"context"
	"fmt"
	"io"
	"log"
	"math/rand"
	"net"
	"sync"
	"time"

	"a.yandex-team.ru/direct/infra/dt-haproxy-cloud/internal/cloudapi"
	"a.yandex-team.ru/direct/infra/dt-haproxy-cloud/internal/config"
	"a.yandex-team.ru/direct/infra/dt-haproxy-cloud/internal/mylog"
)

func StartNativeProxy(pool cloudapi.ActivePool, port int) {
	address := fmt.Sprintf(":%d", port)
	incoming, err := net.Listen("tcp", address)
	if err != nil {
		mylog.Warn("could not start server on %s: %s", address, err)
		log.Fatal(err)
	}
	for {
		conn, err := incoming.Accept()
		if err != nil {
			mylog.Warn("error incoming connect: %s", err)
			continue
		}
		go RunProxy(conn, pool)
	}
}

func RunProxy(conn net.Conn, pool cloudapi.ActivePool) {
	hosts := pool.ActivePool().Masters().Hostnames()
	if len(hosts) == 0 {
		mylog.Warn("empty read pool hosts: %s, active %v, all %v", err, pool.ActivePool(), pool.AllPool())
		_ = conn.Close()
		return
	}

	rand.Seed(time.Now().Unix())
	i := rand.Intn(len(hosts))
	masterHost := hosts[i]
	address := fmt.Sprintf("%s:%d", masterHost.ToString(), config.MYSQLPORT)
	mylog.Debug("proxy connect to %s", address)

	proxy, err := net.Dial("tcp", address)
	if err != nil {
		mylog.Warn("error incoming connect: %s", err)
		_ = conn.Close()
		return
	}
	ctx, cancel := context.WithCancel(context.Background())
	var wg sync.WaitGroup
	wg.Add(1)
	go CopyIO(conn, proxy, &wg)
	wg.Add(1)
	go CopyIO(proxy, conn, &wg)
	go func(cancel context.CancelFunc, wg *sync.WaitGroup) {
		wg.Wait()
		cancel()
	}(cancel, &wg)
	for {
		select {
		case <-ctx.Done():
			mylog.Debug("closed ctx")
			_ = proxy.Close()
			_ = conn.Close()
			return
		default:
			newHosts := pool.ActivePool().Masters()
			if !newHosts.HasHost(masterHost) {
				mylog.Warn("master changed: old %v new %v", masterHost, newHosts)
				cancel()
			}
			time.Sleep(5 * time.Second)
		}
	}
}

func CopyIO(src, dest net.Conn, wg *sync.WaitGroup) {
	defer (*wg).Done()
	if _, err = io.Copy(src, dest); err != nil {
		mylog.Warn("error copy %s", err)
	}
}
