package cloudapi

import (
	"bytes"
	"crypto/tls"
	"crypto/x509"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"strings"
	"time"

	"a.yandex-team.ru/direct/infra/dt-haproxy-cloud/internal/mylog"
)

var (
	APICloudFmt       = "https://gw.db.yandex-team.ru/managed-%s/v1/clusters" //clusterId
	APICloudOperation = "https://gw.db.yandex-team.ru/operations"             //operationId
	cacert            *string
)

func APICloud(clusterType string) string {
	return fmt.Sprintf(APICloudFmt, clusterType)
}

type Response struct {
	Data []byte
	Err  error
}

func (r Response) mydata() []byte {
	return r.Data
}

func (r Response) myerr() error {
	return r.Err
}

//get запрос на получение данных из cloud
func Send(url, aimtoken, reqtype string, body []byte) (result Response) {
	client := &http.Client{}
	var caCertPool *x509.CertPool
	var err error
	if strings.Contains(url, "yandex-team.ru") {
		caCertPool, err = CreateCertPool(CAINTERNAL)
	} else {
		caCertPool, err = CreateCertPool(CAEXTERNAL)
	}
	if err != nil {
		result.Err = err
		return
	}
	mytls := &tls.Config{
		RootCAs: caCertPool,
	}
	client.Transport = &http.Transport{
		TLSClientConfig: mytls,
	}

	var req *http.Request
	if body != nil {
		req, err = http.NewRequest(reqtype, url, bytes.NewReader(body))
	} else {
		req, err = http.NewRequest(reqtype, url, nil)
	}
	if err != nil {
		result.Err = err
		return
	}
	req.Header.Set("Authorization", fmt.Sprintf("Bearer %s", aimtoken))
	resp, err := client.Do(req)
	if err != nil {
		result.Err = err
		return
	}
	defer func() { _ = resp.Body.Close() }()
	result.Data, _ = ioutil.ReadAll(resp.Body)
	if resp.StatusCode != 200 {
		result.Err = fmt.Errorf("http code: %s, req %s, data: %s", resp.Status, url, result.Data)
		return
	}
	if len(result.Data) == 0 {
		result.Err = fmt.Errorf("empty body response %s", url)
		return
	}
	return
}

func UpdateClusterHosts(token *CloudToken, chosts ClusterHosts, clusterID, clusterType string) {
	ticker := time.NewTicker(30 * time.Second)
L:
	for range ticker.C {
		cluster, err := token.RequestHostsdByID(clusterID, clusterType)
		if err != nil {
			mylog.Crit("[RequestHostsdByID] error clusterID: %s, error %s\n", clusterID, err)
			continue L
		}

		if len(*cluster.Hosts) > 0 {
			*chosts.Hosts = *cluster.Hosts
			if debug {
				for _, host := range *chosts.Hosts {
					mylog.Debug("host: %+v\n", host)
				}
			}
		}
		time.Sleep(time.Minute)
	}
}

func (ct *CloudToken) RequestShardByName(clusterID, shardName string) (result ShardSettings, err error) {
	iamtoken, errmsg := ct.GetIAMToken()
	if errmsg != nil {
		return result, fmt.Errorf("%s", errmsg)
	}
	url := fmt.Sprintf("%[1]s/%[2]s/shards/%[3]s", APICloud("clickhouse"), clusterID, shardName)

	resp := Send(url, iamtoken, "GET", nil)

	mylog.Debug("RequestShardByName url: %s, token: %s, data: %s\n", url, iamtoken, resp.Data)
	return ShardByName(resp)
}

func (ct *CloudToken) RequestClusterdByID(clusterID, clusterType string) (result ClusterType, err error) {
	iamtoken, errmsg := ct.GetIAMToken()
	if errmsg != nil {
		return result, fmt.Errorf("%s", errmsg)
	}
	url := fmt.Sprintf("%[1]s/%[2]s", APICloud(clusterType), clusterID)

	resp := Send(url, iamtoken, "GET", nil)
	mylog.Debug("RequestHostsdByID url: %s, token: %s, data: %s\n", url, iamtoken, resp.Data)
	return ClusterByID(resp)
}

func (ct *CloudToken) RequestHostsdByID(clusterID, clusterType string) (result ClusterHosts, err error) {
	iamtoken, errmsg := ct.GetIAMToken()
	if errmsg != nil {
		return result, fmt.Errorf("%s", errmsg)
	}
	url := fmt.Sprintf("%[1]s/%[2]s/hosts", APICloud(clusterType), clusterID)

	resp := Send(url, iamtoken, "GET", nil)
	mylog.Debug("RequestHostsdByID url: %s, token: %s, data: %s\n", url, iamtoken, resp.Data)
	return HostsdByID(resp)
}

func (ct *CloudToken) RequestAddShard(clusterID string, shardSpec ShardSettingsSpec) (result CloudTaskStatus, err error) {
	iamtoken, errmsg := ct.GetIAMToken()
	if errmsg != nil {
		return result, fmt.Errorf("%s", errmsg)
	}
	method := fmt.Sprintf("%[1]s/%[2]s/shards", APICloud("clcikhouse"), clusterID)
	body, err := json.Marshal(shardSpec)
	if err != nil {
		return result, fmt.Errorf("error marshal %+v: %s", shardSpec, err)
	}
	resp := Send(method, iamtoken, "POST", body)
	if resp.Err != nil {
		return result, fmt.Errorf("error addShard: data %s, err %s", resp.Data, resp.Err)
	}
	return TaskStatus(resp)
}

func (ct *CloudToken) RequestOperationID(operationID string) (result CloudTaskStatus, err error) {
	iamtoken, errmsg := ct.GetIAMToken()
	if errmsg != nil {
		return result, fmt.Errorf("%s", errmsg)
	}
	method := fmt.Sprintf("%[1]s/%[2]s", APICloudOperation, operationID)
	resp := Send(method, iamtoken, "GET", nil)
	if resp.Err != nil {
		return result, fmt.Errorf("error %s: data %s, err %s", method, resp.Data, resp.Err)
	}
	if debug {
		fmt.Printf("method %s ID %s\n", method, resp.Data)
	}
	return TaskStatus(resp)
}

func (ct *CloudToken) RequestDuplicateShard(clusterID, sourceShardName, destinationShardName string) (result CloudTaskStatus, err error) {
	sourceShard, err := ct.RequestShardByName(clusterID, sourceShardName)
	if err != nil {
		return result, fmt.Errorf("error get shards for %s: %s", clusterID, err)
	}
	if len(destinationShardName) == 0 {
		return result, fmt.Errorf("empty destinationShardName")
	}
	spec := NewShardSettingsSpec(destinationShardName, sourceShard.Config)
	for _, cluster := range []string{"sas", "man", "vla"} {
		host := NewHostSpec(destinationShardName, "CLICKHOUSE", cluster, false)
		spec.HostSpecs = append(spec.HostSpecs, host)
	}
	return ct.RequestAddShard(clusterID, spec)
}

func ShardByName(value InputInterface) (result ShardSettings, err error) {
	if value.myerr() != nil {
		return result, value.myerr()
	}
	result = NewShardSettings()
	if err := json.Unmarshal(value.mydata(), &result); err != nil {
		err = fmt.Errorf("error unmarshal %s: %s", value.mydata(), err)
		return result, err
	}
	return
}

func ClusterByID(value InputInterface) (result ClusterType, err error) {
	if value.myerr() != nil {
		return result, value.myerr()
	}
	result = NewClusterType()
	if err := json.Unmarshal(value.mydata(), &result); err != nil {
		err = fmt.Errorf("error unmarshal %s: %s", value.mydata(), err)
		return result, err
	}
	return
}

func HostsdByID(value InputInterface) (result ClusterHosts, err error) {
	if value.myerr() != nil {
		return result, value.myerr()
	}
	result = NewClusterHosts()
	if err := json.Unmarshal(value.mydata(), &result); err != nil {
		err = fmt.Errorf("error unmarshal %s: %s", value.mydata(), err)
		return result, err
	}
	return
}

func TaskStatus(value InputInterface) (result CloudTaskStatus, err error) {
	if value.myerr() != nil {
		return result, value.myerr()
	}
	result = NewCloudTaskStatus()
	if err := json.Unmarshal(value.mydata(), &result); err != nil {
		err = fmt.Errorf("error unmarshal %s: %s", value.mydata(), err)
		return result, err
	}
	return
}

type InputInterface interface {
	mydata() []byte
	myerr() error
}
