package cloudapi

import (
	"fmt"
	"sort"
	"strconv"
	"strings"
	"time"

	"a.yandex-team.ru/direct/infra/dt-haproxy-cloud/internal/mylog"
)

//Bytes
func (b Bytes) Humanize() string {
	const unit = 1024
	if b < unit {
		return fmt.Sprintf("%dB", b)
	}
	div, exp := int64(unit), 0
	for n := b / unit; n > unit; n /= unit {
		div *= unit
		exp++
	}
	return fmt.Sprintf("%.2f%cB", float64(b)/float64(div), "KMGTPE"[exp])
}

//UsedSpaceByTables
func (stb StatisticTables) TotalUsed() Bytes {
	var used int
	for _, table := range stb {
		used += table.Used
	}
	return Bytes(used)
}

//StatisticSpace
//свободное место на основе таблицы system.disk
func (ss StatisticSpace) Free() int64 {
	v, err := strconv.ParseInt(ss.FreeSpace, 10, 64)
	if err != nil {
		mylog.Crit("error convert %s: %s", ss.FreeSpace, err)
		return 0
	}
	return v
}

//доступное место на основе таблицы system.disk
func (ss StatisticSpace) Total() int64 {
	v, err := strconv.ParseInt(ss.TotalSpace, 10, 64)
	if err != nil {
		mylog.Crit("error convert %s: %s", ss.TotalSpace, err)
		return 0
	}
	return v
}

//занятое место на основе таблицы system.disk
func (ss StatisticSpace) Used() Bytes {
	return Bytes(ss.Total() - ss.Free())
}

//SpaceDiskByHost
func (s SpaceDiskByHost) PercentUsed() float32 {
	if s.DiskSize == 0 {
		return -1
	}
	return (float32(s.UsedSize) / float32(s.DiskSize) * 100.0)
}

func (s SpaceDiskByHost) DetachedSize() Bytes {
	return s.UsedSize - s.TotalPartitionsSize
}

//SpaceDiskByHosts
func (ss SpaceDiskByHosts) Print() {
	sort.Sort(sort.Interface(ss))
	for _, h := range ss {
		fmt.Printf("[%s] %s used: %s, limit: %s, percent used: %0.2f%%, detached partitions size: %s\n",
			h.ShardName, h.Host, h.UsedSize.Humanize(),
			h.DiskSize.Humanize(), h.PercentUsed(), h.DetachedSize().Humanize())
	}
	fmt.Printf("used=sum(parts)+sum(detached)\npercent_used=(used/total)*100\n")
}

func (ss SpaceDiskByHosts) MaxGroupPercent(shard string) (result float32) {
	for _, host := range ss {
		if host.ShardName != shard {
			continue
		}
		if result < host.PercentUsed() {
			result = host.PercentUsed()
		}
	}
	return
}

func (ss SpaceDiskByHosts) Len() int      { return len(ss) }
func (ss SpaceDiskByHosts) Swap(i, j int) { ss[i], ss[j] = ss[j], ss[i] }
func (ss SpaceDiskByHosts) Less(i, j int) bool {
	if ss[i].ShardName != ss[j].ShardName {
		return ss.MaxGroupPercent(ss[i].ShardName) < ss.MaxGroupPercent(ss[j].ShardName)
	}
	return ss[i].ShardName < ss[j].ShardName
}

//Hostnames
//машины готовые для чтения

func ClickhouseReadyWriteHosts(shs StatusHosts, maxperc interface{}) (result StatusHosts) {
	val, ok := maxperc.(int)
	if !ok {
		val = 0
	}
	result = make(StatusHosts)
	for host, status := range shs {
		if status.IsReady() &&
			status.HasFreeSpace(val) &&
			shs.HasFreeSpaceHostsInShard(status.ShardName, val) {
			result[host] = status
		}
	}
	return
}

func ClickhouseReadyReadHosts(shs StatusHosts) (result StatusHosts) {
	result = make(StatusHosts)
	for host, status := range shs {
		if status.IsReady() {
			result[host] = status
		}
	}
	return
}

func MysqlReadyReadHosts(shs StatusHosts) (result StatusHosts) {
	result = make(StatusHosts)
	for host, status := range shs {
		if strings.ToUpper(status.Health) == "ALIVE" {
			result[host] = status
		}
	}
	return
}

//Проверяет наличие свободного места на всех машинах указанного шарда
func (shs StatusHosts) HasFreeSpaceHostsInShard(shard string, maxperc int) bool {
	for _, value := range shs {
		if value.HasShard(shard) && !value.HasFreeSpace(maxperc) {
			return false
		}
	}
	return true
}

func (shs StatusHosts) LastModifyTime() int64 {
	var max int64 = 0
	for _, value := range shs {
		if *value.ModifyTime > max {
			max = *value.ModifyTime
		}
	}
	return max
}

func (shs StatusHosts) Masters() (result StatusHosts) {
	result = make(StatusHosts)
	for host, status := range shs {
		if strings.ToUpper(status.Health) == "ALIVE" && status.IsMaster() {
			result[host] = status
		}
	}
	return
}

func (shs StatusHosts) Hostnames() (result Hostnames) {
	for _, status := range shs {
		result = append(result, status.Hostname())
	}
	return
}

func (sh StatusHost) IsReady() bool {
	//if (strings.ToUpper(sh.Health) == "ALIVE" || len(sh.Health) == 0) &&
	if strings.ToUpper(*sh.Status) == "OK" {
		mylog.Debug("[debug] server: %s, health: %s, status: %s\n", sh.Name, sh.Health, *sh.Status)
		return true
	}
	mylog.Warn("[warn] server: %s, health: %s, status: %s\n", sh.Name, sh.Health, *sh.Status)
	return false
}

func (sh StatusHost) IsMaster() bool {
	//if (strings.ToUpper(sh.Health) == "ALIVE" || len(sh.Health) == 0) &&
	if strings.ToUpper(*sh.Status) == "OK" && strings.ToUpper(sh.Role) == "MASTER" {
		mylog.Debug("[debug] server: %s, health: %s, role: %s\n", sh.Name, sh.Health, sh.Role)
		return true
	}
	mylog.Debug("[warn] server: %s, health: %s, role: %s\n", sh.Name, sh.Health, sh.Role)
	return false
}

func (sh StatusHost) HasFreeSpace(maxperc int) bool {
	if maxperc == 0 {
		return true
	}
	diskSize := sh.GetDiskSize()
	//при нулевом значении total space возвращаем false
	if diskSize == 0 {
		return false
	}
	usedSize := sh.StatSpace.Used()
	return (float64(usedSize)/float64(diskSize))*100.0 < float64(maxperc)
}

func (sh StatusHost) HasShard(name string) bool {
	return sh.ShardName == name
}

func (sh StatusHost) GetStatus() string {
	return *sh.Status
}

//проверяет вхождение fqdn хоста в группу fqdn хостов
func (hs Hostnames) HasHost(name interface{}) bool {
	for _, host := range hs {
		if strings.Contains(host.ToString(), toString(name)) {
			return true
		}
	}
	return false
}

func (shs StatusHosts) HasHost(name interface{}) bool {
	for _, host := range shs {
		if strings.Contains(host.Hostname().ToString(), toString(name)) {
			return true
		}
	}
	return false
}

//конвертирует значение в string
func toString(value interface{}) (result string) {
	switch v := value.(type) {
	case Hostname:
		result = v.ToString()
	case string:
		result = v
	default:
		result = fmt.Sprintf("%s", v)
	}
	return
}

func (h Hostname) ToString() string {
	return fmt.Sprintf("%v", h)
}

func (ch ClusterHost) Hostname() Hostname {
	return Hostname(ch.Name)
}

func (ch ClusterHost) GetDiskSize() Bytes {
	size, err := strconv.ParseInt(ch.Resources.DiskSize, 10, 64)
	if err != nil {
		mylog.Warn("error convert %s to int64\n", ch.Resources.DiskSize)
	}
	return Bytes(size)
}

func (chs ClusterHosts) Hostnames() (result Hostnames) {
	for _, host := range *chs.Hosts {
		result = append(result, Hostname(host.Name))
	}
	return
}

func (cwh CHWriterHosts) ActivePool() StatusHosts {
	return ClickhouseReadyWriteHosts(cwh.current, cwh.unispace)
}

func (cwh CHWriterHosts) AllPool() StatusHosts {
	return cwh.current
}

func (cwh CHWriterHosts) Group() string {
	return cwh.group
}

func (cwh CHWriterHosts) LastPool() StatusHosts {
	return *cwh.last
}

func (cwh CHWriterHosts) SetLastPool(s StatusHosts) {
	*cwh.last = s
}

func (crh CHReaderHosts) ActivePool() StatusHosts {
	return ClickhouseReadyReadHosts(crh.current)
}

func (crh CHReaderHosts) AllPool() StatusHosts {
	return crh.current
}

func (crh CHReaderHosts) LastPool() StatusHosts {
	return *crh.last
}

func (crh CHReaderHosts) SetLastPool(s StatusHosts) {
	*crh.last = s
}

func (crh CHReaderHosts) Group() string {
	return crh.group
}

//active for write
func (mh MySQLAliveHosts) ActivePool() StatusHosts {
	return MysqlReadyReadHosts(mh.current)
}

//all instances
func (mh MySQLAliveHosts) AllPool() StatusHosts {
	return mh.current
}

func (mh MySQLAliveHosts) LastPool() StatusHosts {
	return *mh.last
}

func (mh MySQLAliveHosts) SetLastPool(s StatusHosts) {
	*mh.last = s
}

func (mh MySQLAliveHosts) Group() string {
	return mh.group
}

func (astat AliveStatus) GetAliveStatus() []byte {
	mtime := *astat.ModifyTime
	if time.Since(*astat.ModifyTime) > time.Duration(1*time.Minute) {
		return []byte(fmt.Sprintf("FAILED: alive check died: %s", mtime))
	}
	if *astat.LastErrors != nil {
		return []byte(fmt.Sprintf("FAILED: %s", *astat.LastErrors))
	}
	return []byte("ALIVE")
}

func (t Table) Name() string {
	return fmt.Sprint(t)
}

func (ts Tables) HasAny(table interface{}) bool {
	switch v := table.(type) {
	case Table:
		for _, t := range ts {
			if v.Name() == t.Name() {
				return true
			}
		}
	case Tables:
		for _, t := range v {
			if ts.HasAny(t) {
				return true
			}
		}
	default:
		for _, t := range ts {
			if fmt.Sprint(v) == t.Name() {
				return true
			}
		}
	}
	return false
}
