package cloudapi

import (
	"database/sql"
	"time"
)

//Байты
type Bytes int

//Количество занятого места на хосте
type SpaceDiskByHost struct {
	Host                string `json:"host"`
	ShardName           string `json:"shard-name"`
	DiskSize            Bytes  `json:"disk-size"`
	UsedSize            Bytes  `json:"used-size"`
	TotalPartitionsSize Bytes  `json:"total-partitions-size"`
}

func NewSpaceDiskByHost(host, shard string, ds, us, ps Bytes) SpaceDiskByHost {
	return SpaceDiskByHost{
		Host:                host,
		ShardName:           shard,
		DiskSize:            ds,
		UsedSize:            us,
		TotalPartitionsSize: ps,
	}
}

//Список хостов с занятым местом
type SpaceDiskByHosts []SpaceDiskByHost

//Количество занятого места по таблицам
type StatisticTable struct {
	Table string `db:"table" json:"table"`
	Used  int    `db:"used" json:"used"`
}

func NewStatisticTables(table string, us int) StatisticTable {
	return StatisticTable{
		Table: table,
		Used:  us,
	}
}

//Список таблиц с занятым местом
type StatisticTables []StatisticTable

//Статус выполнения с сообщением
type CodeWithMessage struct {
	Code    int    `json:"code"`
	Message string `json:"message"`
}

func NewCodeWithMessage(code int, msg string) CodeWithMessage {
	return CodeWithMessage{
		Code:    code,
		Message: msg,
	}
}

//Количество чего-то(например битых detached партиций) на таблицу
type CountPerTable struct {
	Table string `json:"table"`
	Count int    `json:"count"`
}

type StatusHosts map[string]StatusHost

type StatusHost struct {
	*ClusterHost
	Status     *string
	StatTables *StatisticTables
	StatSpace  *StatisticSpace
	Conn       *sql.DB
	ModifyTime *int64
}

func NewStatusHost(ch ClusterHost, conn *sql.DB) StatusHost {
	var status = "NEW"
	var now = time.Now().Unix()
	return StatusHost{
		ClusterHost: &ch,
		Status:      &status,
		StatTables:  new(StatisticTables),
		StatSpace:   new(StatisticSpace),
		Conn:        conn,
		ModifyTime:  &now,
	}
}

type AliveStatus struct {
	ModifyTime *time.Time
	LastErrors *error
}

func NewAliveStatus() AliveStatus {
	now := time.Now()
	var err error
	return AliveStatus{
		ModifyTime: &now,
		LastErrors: &err,
	}
}

//fqdn машины с базой
type Hostname string

//список машин с базами кластера
type Hostnames []Hostname

func NewHostnames() Hostnames {
	return Hostnames{}
}

//GET https://mdb.api.cloud.yandex.net/managed-clickhouse/v1/clusters/{clusterId}
type ClusterType struct {
	ClusterID string    `json:"id"`
	Name      string    `json:"name"`
	Health    string    `json:"health"`
	Config    ConfigMap `json:"config"`
}

func NewClusterType() ClusterType {
	config := make(ConfigMap)
	return ClusterType{Config: config}
}

type ConfigMap map[string]interface{} //for clickhouse: ClickhouseSettings

//GET https://mdb.api.cloud.yandex.net/managed-clickhouse/v1/clusters/{clusterId}/shards/{shardName}
type ShardSettings struct {
	ClusterID string    `json:"clusterId"`
	Name      string    `json:"name"`
	Config    ConfigMap `json:"config"` //{config: { clickhouse: { config: ...
}

type ClickhouseSettings struct {
	ClickhouseConfigs   `json:"config"`
	ClickhouseResources `json:"resources"`
	Weight              string `json:"weight"`
}

type ClickhouseResources struct {
	ResourcePresetID string `json:"resourcePresetId"`
	DiskSize         string `json:"diskSize"`
	DiskTypeID       string `json:"diskTypeId"`
}

type ClickhouseConfigs struct {
	EffectiveConfig ClickhouseConfig `json:"effectiveConfig"`
	DefaultConfig   ClickhouseConfig `json:"defaultConfig"`
}

type ClickhouseConfig struct {
	MergeTree                         MergetreeSettings `json:"mergeTree"`
	LogLevel                          string            `json:"logLevel"`
	MaxConnections                    string            `json:"maxConnections"`
	MaxConcurrentQueries              string            `json:"maxConcurrentQueries"`
	KeepAliveTimeout                  string            `json:"keepAliveTimeout"`
	UncompressedCacheSize             string            `json:"uncompressedCacheSize"`
	MarkCacheSize                     string            `json:"markCacheSize"`
	MaxTableSizeToDrop                string            `json:"maxTableSizeToDrop"`
	BuiltinDictionariesReloadInterval string            `json:"builtinDictionariesReloadInterval"`
	MaxPartitionSizeToDrop            string            `json:"maxPartitionSizeToDrop"`
	Timezone                          string            `json:"timezone"`
}

type MergetreeSettings struct {
	ReplicatedDeduplicationWindow        string `json:"replicatedDeduplicationWindow"`
	ReplicatedDeduplicationWindowSeconds string `json:"replicatedDeduplicationWindowSeconds"`
}

func NewShardSettings() ShardSettings {
	config := make(ConfigMap)
	return ShardSettings{Config: config}
}

type Resources struct {
	ResourcePresetID string `json:"resourcePresetId"`
	DiskSize         string `json:"diskSize"`
	DiskTypeID       string `json:"diskTypeId"`
}

type ClusterHost struct {
	AssignPublicIP bool   `json:"assignPublicIp"`
	Name           string `json:"name"`
	ClusterID      string `json:"clusterId"`
	ZoneID         string `json:"zoneId"`
	Type           string `json:"type"`
	Health         string `json:"health"`
	Resources      `json:"resources"`
	ShardName      string `json:"shardName"`
	Role           string `json:"role"`
}

type ClusterHosts struct {
	Hosts *[]ClusterHost `json:"hosts"`
}

func NewClusterHosts() ClusterHosts {
	hosts := []ClusterHost{}
	return ClusterHosts{&hosts}
}

type ShardSettingsSpec struct {
	ShardName  string    `json:"shardName"`
	ConfigSpec ConfigMap `json:"configSpec"`
	HostSpecs  `json:"hostSpecs"`
}

func NewShardSettingsSpec(shardName string, config ConfigMap) ShardSettingsSpec {
	return ShardSettingsSpec{
		ShardName:  shardName,
		ConfigSpec: config,
		HostSpecs:  HostSpecs{},
	}
}

type HostSpecs []HostSpec

type HostSpec struct {
	ZoneID         string `json:"zoneId"`
	Type           string `json:"type"`
	AssignPublicIP bool   `json:"assignPublicIp"`
	ShardName      string `json:"shardName"`
}

func NewHostSpec(shardName, types, zone string, publicIP bool) HostSpec {
	return HostSpec{
		ZoneID:         zone,
		Type:           types,
		ShardName:      shardName,
		AssignPublicIP: publicIP,
	}
}

type CloudTaskStatus struct {
	ID          string    `json:"id"`
	Description string    `json:"description"`
	CreatedAt   string    `json:"createdAt"`
	CreatedBy   string    `json:"createdBy"`
	ModifiedAt  string    `json:"modifiedAt"`
	Done        bool      `json:"done"`
	Error       TaskError `json:"error"`
}

type TaskMetadata struct {
	Type      string `json:"@type"`
	ClusterID string `json:"clusterId"`
	ShardName string `json:"shardName"`
}

type TaskError struct {
	Code    int      `json:"code"`
	Message string   `json:"message"`
	Details []string `json:"details"`
}

type MetaStatus struct {
	current StatusHosts
	last    *StatusHosts
	group   string
}

func NewCloudTaskStatus() CloudTaskStatus {
	return CloudTaskStatus{}
}

type CHWriterHosts struct {
	MetaStatus
	unispace int
}

func NewCHWriterHosts(pool StatusHosts, group string, maxperc int) CHWriterHosts {
	l := make(StatusHosts)
	return CHWriterHosts{
		MetaStatus: MetaStatus{
			current: pool,
			last:    &l,
			group:   group,
		},
		unispace: maxperc,
	}
}

type CHReaderHosts struct {
	MetaStatus
}

func NewCHReaderHosts(pool StatusHosts, group string) CHReaderHosts {
	l := make(StatusHosts)
	return CHReaderHosts{
		MetaStatus: MetaStatus{
			current: pool,
			last:    &l,
			group:   group,
		},
	}
}

type MySQLAliveHosts struct {
	MetaStatus
}

func NewMySQLAliveHosts(pool StatusHosts, group string) MySQLAliveHosts {
	l := make(StatusHosts)
	return MySQLAliveHosts{
		MetaStatus: MetaStatus{
			current: pool,
			last:    &l,
			group:   group,
		},
	}
}

type StatisticSpace struct {
	FreeSpace  string `db:"free_space" json:"free_space"`
	TotalSpace string `db:"total_space" json:"total_space"`
}

type Table string
type Tables []Table

type ActivePool interface {
	AllPool() StatusHosts
	ActivePool() StatusHosts
	Group() string
	LastPool() StatusHosts
	SetLastPool(StatusHosts)
}
