package config

import (
	"bytes"
	"fmt"
	"io/ioutil"
	"os"
	"path/filepath"
	"strings"

	"gopkg.in/yaml.v2"
)

const (
	CONFDIR               = "/etc/yandex/dt-haproxy-cloud/"
	CLOUDKEY              = "/tmp/key.json"
	CACERT                = "/usr/share/yandex-internal-root-ca/YandexInternalRootCA.crt"
	CLUSTERID             = "mdbns84iukrpbke13rul"
	CHUSER                = "direct_writer"
	PASSWDFILE            = "/tmp/pass"
	CHDB                  = "directdb"
	HAPROXYPATH           = "/tmp/haproxy.cnf"
	HAPROXYPID            = "/tmp/haproxy.pid"
	MAXPERCENTUSEDSPACE   = 0
	CHPORT                = 9440
	CHHTTPS               = 8443
	CHHTTP                = 8440
	ROHTTPS               = 7443
	ROHTTP                = 7440
	MYSQLPORT             = 3306
	ENVFILE               = "/etc/yandex/environment.type"
	DefaultExcludeServers = ""
)

func NewConfig() Config {
	newpass := ""
	return Config{
		User:                CHUSER,
		Password:            &newpass,
		PasswordFile:        PASSWDFILE,
		Database:            CHDB,
		ClusterID:           CLUSTERID,
		CAcert:              CACERT,
		HaproxyPath:         HAPROXYPATH,
		HaproxyPid:          HAPROXYPID,
		CloudKey:            CLOUDKEY,
		MaxPercentUsedSpace: MAXPERCENTUSEDSPACE,
		Certificate:         "",
		ExcludeServers:      DefaultExcludeServers,
	}
}

type Config struct {
	User                string  `yaml:"user"`                   //имя пользователя для подключения к БД
	Password            *string `yaml:"-"`                      //пароль для подключения к БД(запослняется на основе содержимого PasswordFile)
	PasswordFile        string  `yaml:"password_file"`          //путь до файла с паролем к БД clickhouse
	Database            string  `yaml:"database"`               //имя БД в clickhouse
	ClusterID           string  `yaml:"cluster_id"`             //id кластера clickhouse(id дирректории, в котором размещается база)
	CAcert              string  `yaml:"ca_cert"`                //путь до сертификата CA(нужен для генерации токена IAM и подключения к clickhouse)
	HaproxyPath         string  `yaml:"haproxy_path"`           //путь до конфига haproxy(в него записывается новый конфиг)
	HaproxyPid          string  `yaml:"haproxy_pid"`            //путь до pid файла haproxy(по нему отправляется HUP сигнал)
	CloudKey            string  `yaml:"cloud_key"`              //ключ робота(ssh+id в вормате json) для работы с mds
	MaxPercentUsedSpace int     `yaml:"max_percent_used_space"` //процент занятого места, после которого машина убирается из списка
	Certificate         string  `yaml:"certificate"`            //указывает какой сертификат должен использоваться
	ExcludeServers      string  `yaml:"exclude_servers"`        //указывает какие сервера требуется исключить из списка активных
}

//На основе переменных окружения определяем какой конфиг использовать: CONFDIR/server.<ENVTYPE>.yaml
func SearchConfigByEnv(cluster, suffix string) (path string, err error) {
	env, ok := os.LookupEnv("ENVTYPE")
	if !ok {
		err = fmt.Errorf("enviroment ENVTYPE not found")
		return
	}
	sf := fmt.Sprintf("%[3]s.%[1]s.%[2]s.yaml", env, suffix, cluster)
	configs, err := ioutil.ReadDir(CONFDIR)
	if err != nil {
		return
	}
	for _, file := range configs {
		if strings.Contains(file.Name(), sf) {
			return filepath.Join(CONFDIR, file.Name()), nil
		}
	}
	return
}

//Загружаем ENVTYPE из файла на сервере
func LoadEnviromentType(path string) (err error) {
	if len(path) == 0 {
		path = ENVFILE
	}
	envtype, err := ioutil.ReadFile(path)
	if err != nil {
		return fmt.Errorf("error read %s: %s", path, err)
	}
	envtype = bytes.TrimRight(envtype, "\n")
	return os.Setenv("ENVTYPE", strings.ToLower(string(envtype)))
}

//Загружает конфиг. На вход подается путь до конфига, на выходе структура Config и ошибка error.
func LoadConfig(path string) (config Config, err error) {
	config = NewConfig()
	data, err := ioutil.ReadFile(path)
	if err != nil {
		err = fmt.Errorf("error read config %s: %s", path, err)
		return
	}
	if err = yaml.Unmarshal(data, &config); err != nil {
		return
	}
	return
}

//Читает файл PasswordFile и загружает его значение в Password. На выходе ошибка error.
func (c Config) LoadPassword() (err error) {
	pass, err := ioutil.ReadFile(c.PasswordFile)
	if err != nil {
		return fmt.Errorf("error read password file %s: %s", c.PasswordFile, err)
	}
	pass = bytes.TrimRight(pass, "\n")
	*c.Password = string(pass)
	return
}
