import collections
import inspect
import yaml

from .feature_dumper import FeatureDumper, FeatureLoader, quoted


class Feature:
    def __init__(self, name, description, type, deprecated, enabled_in_e2e):
        self.name = name
        self.description = description
        self.type = type
        self.deprecated = deprecated
        self.enabled_in_e2e = enabled_in_e2e

    def upper_name(self):
        return self.name.upper()

    def camel_name(self):
        words = self.name.lower().split('_')
        words = words[:1] + [word.capitalize() for word in words[1:]]
        return ''.join(words)


def read_feature_config(config_path):
    with open(config_path, 'r') as f:
        feature_map = yaml.load(f, Loader=FeatureLoader)

    features = [
        Feature(
            name=feature_name,
            description=data['text'],
            type=data.get('type', 'TEMP').upper(),
            deprecated=data.get('status') == 'deprecated',
            enabled_in_e2e=data.get('enabled_in_e2e') == 'true',
        )
        for feature_name, data in feature_map.items()
    ]

    return features


def generate_feature_config(features):
    source = inspect.cleandoc('''
        # please add new features to the end of the file
    ''')
    source += '\n\n'

    feature_map = collections.OrderedDict()
    for feature in features:
        feature_object = collections.OrderedDict()

        feature_object['text'] = feature.description
        if feature.type != 'TEMP':
            feature_object['type'] = feature.type.lower()
        if feature.deprecated:
            feature_object['status'] = 'deprecated'
        if feature.enabled_in_e2e:
            feature_object['enabled_in_e2e'] = quoted('true')

        feature_map[feature.name] = feature_object

    source += yaml.dump(
        feature_map,
        Dumper=FeatureDumper,
        allow_unicode=True,
        width=2**31-1,
    )

    return source
