package main

import (
	logger "a.yandex-team.ru/direct/infra/go-libs/pkg/logformat"
	"a.yandex-team.ru/direct/infra/go-libs/pkg/mdb"
	"encoding/json"
	"flag"
	"fmt"
	"log"
	"os"
	"sort"
	"strings"
)

const (
	DefaultFolderID = "fooa07bcrr7souccreru"
	DefaultMDBKey   = ""
	DefaultFilter   = ""
)

var (
	folderID       string
	debug          bool
	mdbKey, filter string
	jsonFormat     bool
)

func main() {
	flag.StringVar(&folderID, "folder-id", DefaultFolderID, "id директории проекта в MDB")
	flag.StringVar(&mdbKey, "mdb-key", DefaultMDBKey, "ключ для работы с MDB")
	flag.StringVar(&filter, "filter", DefaultFilter, "фильтр по которому отфильтровать имена кластеров")
	flag.BoolVar(&debug, "debug", false, "debug mode")
	flag.BoolVar(&jsonFormat, "json", false, "json format")
	flag.Parse()

	newLogger := log.New(os.Stdout, "", 0)

	//enable debug mode
	if debug {
		logger.NewLoggerFormat(
			logger.WithLogLevel(logger.DebugLvl),
			logger.WithLogger(newLogger),
		)
	} else {
		logger.NewLoggerFormat(
			logger.WithLogger(newLogger),
		)
	}

	conn, err := mdb.ConnectMysqlCluster("", folderID, mdbKey)
	if err != nil {
		logger.Crit("error connect mdb: %s", err)
		os.Exit(2)
	}

	clusters, err := conn.AllClusters()
	if err != nil {
		logger.Crit("error get clusters: %s", err)
	}

	ids := make(map[string]string)
	var hosts []string
	var mscl MySQLClusters
	for _, cluster := range clusters {
		conn.SetClusterID(cluster.ID)
		h, _ := conn.ListHosts()
		dnsMaster := fmt.Sprintf("c-%s.rw.db.yandex.net", cluster.ID)
		if len(filter) > 0 && strings.Contains(cluster.Name, filter) {
			ids[cluster.Name] = fmt.Sprintf("%s %+v hosts: %s", dnsMaster, h.GetMaster(), strings.Join(h.GetHosts(), ","))
			mscl = append(mscl, NewMySQLCluster(dnsMaster, cluster.Name))
			hosts = append(hosts, cluster.Name)
		} else if len(filter) == 0 {
			ids[cluster.Name] = fmt.Sprintf("%s %+v hosts: %s", dnsMaster, h.GetMaster(), strings.Join(h.GetHosts(), ","))
			mscl = append(mscl, NewMySQLCluster(dnsMaster, cluster.Name))
			hosts = append(hosts, cluster.Name)
		}
	}
	if !jsonFormat {
		sort.Strings(hosts)
		for _, host := range hosts {
			fmt.Printf("%s: %s\n", host, ids[host])
		}
	} else {
		jdata, _ := json.Marshal(mscl)
		fmt.Printf("%s\n", jdata)
	}
}

type MySQLCluster struct {
	Host        string `json:"host"`
	Instance    string `json:"instance"`
	Port        int    `json:"port"`
	GroupName   string `json:"group-name"`
	ReplicaName string `json:"replica-name"`
}

type MySQLClusters []MySQLCluster

func NewMySQLCluster(host, clusterName string) MySQLCluster {
	name := strings.Split(clusterName, "-")
	var groupName, replName string
	if len(name) > 1 {
		groupName = name[1]
	}
	if len(name) > 2 {
		replName = name[2]
	}
	return MySQLCluster{
		Host:        host,
		Port:        3306,
		Instance:    name[0],
		GroupName:   groupName,
		ReplicaName: replName,
	}
}
