package main

import (
	"a.yandex-team.ru/direct/infra/go-libs/pkg/mdb"
	"context"
	"flag"
	"fmt"
	"log"
	"strings"
	"time"
)

const (
	KeyFile        = "/etc/direct-tokens/mdb_robot-direct-admin-manager"
	DirectFolderID = "fooa07bcrr7souccreru"
)

var (
	listBackups, restoreBackup, debugMode, jsonOutput, waiting bool
	clusterID, folderID, token, clusterName, newClusterName    string
	clusterZones                                               string
	backupName, opid                                           string
)

func main() {
	flag.BoolVar(&debugMode, "debug", false, "enable debug mode")
	flag.BoolVar(&listBackups, "list-backups", false, "listing enabled backups")
	flag.StringVar(&clusterID, "cluster-id", "", "MDB cluster ID")
	flag.StringVar(&folderID, "folder-id", DirectFolderID, "MDB folder ID")
	flag.StringVar(&token, "token", KeyFile, "token key file")
	flag.StringVar(&newClusterName, "new-cluster-name", "", "new name cluster")
	flag.StringVar(&clusterZones, "cluster-zones", "sas,vla", "cluster zones(sas,vla, man)")
	flag.StringVar(&backupName, "backup-name", "", "backup name")
	flag.StringVar(&opid, "operation-id", "", "check operation id")
	flag.StringVar(&clusterName, "cluster-name", "", "cluster name")
	flag.BoolVar(&restoreBackup, "restore-backup", false, "resotore mdb backup")
	flag.BoolVar(&jsonOutput, "json", false, "json output")
	flag.BoolVar(&waiting, "wait", false, "waiting status")
	flag.Parse()

	if debugMode {
		mdb.SetLogLevel(1)
	} else {
		mdb.SetLogLevel(2)
	}

	mdbconn, err := mdb.ConnectMysqlCluster(clusterID, folderID, token)
	if err != nil {
		log.Printf("error connect to MDB cluster: %s", err)
	}

	if len(clusterID) > 0 {
		mdbconn.SetClusterID(clusterID)
	} else if len(clusterName) > 0 && len(folderID) > 0 {
		clusterID, err = mdbconn.FindClusterIDByName(clusterName)
		if err != nil {
			log.Fatalf("error find cluster: %s", err)
		}
		mdbconn.SetClusterID(clusterID)
	}

	switch {
	case listBackups:
		if list, err := mdbconn.ListBackups(); err != nil {
			log.Fatal(err)
		} else {
			fmt.Println(strings.Join(list.ToList(), "\n"))
		}
	case restoreBackup:
		if len(newClusterName) == 0 {
			log.Fatalf("empty -new-cluster-name for restore")
		}
		if len(clusterID) == 0 {
			log.Fatalf("empty -cluster-id or -cluster-name for download backup")
		}
		var backupMDB mdb.MysqlBackup

		backups, err := mdbconn.ListBackups()
		if err != nil {
			log.Fatalf("error list backups for restore: %s", err)
		}
		if len(backups.Backups) == 0 {
			log.Fatalf("empty list backups in %s", mdbconn.GetClusterID())
		}

		if len(backupName) > 0 {
			if bmdb, ok := backups.FindBackupID(backupName); ok {
				backupMDB = bmdb
			} else {
				log.Fatalf("not found backup %s in cluster backups%s", backupName, mdbconn.GetClusterID())
			}
		} else {
			backupMDB = backups.Backups[0]
			log.Printf("dont set -backup-name, used last backup %s", backupMDB.ID)
		}

		operationID, err := mdbconn.RestoreBackup(newClusterName, backupMDB, strings.Split(clusterZones, ","), false)
		if err != nil {
			log.Fatalf("error restore backup %s to %s: %s", backupName, newClusterName, err)
		}
		if status, err := mdb.JSONStatus(operationID); err != nil {
			log.Printf("error marshal operationID in restore: %s", err)
			fmt.Printf("%+v\n", status)
		} else {
			fmt.Printf("%s\n", status)
		}
		if waiting {
			cntx, cancel := context.WithTimeout(context.Background(), time.Hour*6)
			defer cancel()
			if ok := mdb.WaitingOperationDone(cntx, operationID.ID, mdbconn); ok {
				log.Printf("DONE")
			} else {
				log.Fatal("FAILED")
			}
		}
	case len(opid) > 0:
		if waiting {
			cntx, cancel := context.WithTimeout(context.Background(), time.Hour*6)
			defer cancel()
			if ok := mdb.WaitingOperationDone(cntx, opid, mdbconn); ok {
				log.Printf("DONE")
			} else {
				log.Fatal("FAILED")
			}
		} else {
			operationID, err := mdb.OperationStatus(opid, mdbconn)
			if err != nil {
				log.Fatalf("error restore backup %s to %s: %s", backupName, newClusterName, err)
			}
			if status, err := mdb.JSONStatus(operationID); err != nil {
				log.Printf("error marshal operationID in status: %s", err)
				fmt.Printf("%+v\n", status)
			} else {
				fmt.Printf("%s\n", status)
			}
		}
	}
}
