package main

import (
	"context"
	"encoding/json"

	"flag"
	"fmt"
	"io/ioutil"
	"log"
	"os"
	"path/filepath"
	"strings"
	"time"

	"gopkg.in/yaml.v2"

	logger "a.yandex-team.ru/direct/infra/go-libs/pkg/logformat"
	"a.yandex-team.ru/direct/infra/go-libs/pkg/mdb"
	"a.yandex-team.ru/direct/infra/go-libs/pkg/oauthlib"
	"a.yandex-team.ru/library/go/yandex/yav/httpyav"
)

const (
	DefaultFolderID     = "fooa07bcrr7souccreru"
	DefaultMySQLID      = ""
	DefaultMDBKey       = ""
	DefaultGrantsFile   = ""
	DefaultYavVersion   = "ver-01crge87f1cgfhj4xr2wxc1hgr"
	DefaultTokenFile    = ".yav.token"
	DefaultDBNames      = "ppc"
	DefaultZoneSpecs    = "sas,vla"
	DefaultEnviroment   = "production"
	DefaultMysqlVersion = "5.7"
	DefaultGroupName    = "devtest"
	DefaultMysqlPort    = 3306
	DefaultMysqlUser    = "direct-test"
)

var (
	//common
	folderID, mdbKey, login, command, mysqlUser string
	mysqlPort                                   int
	debug                                       bool
	grantsFile                                  string
	exitCode                                    int
	yavTokenFile, YavVersion                    string

	yavClient *httpyav.Client

	IgnoredRoles = mdb.Roles([]string{"SUPER", "SHOW_DATABASES"})

	//update users
	mysqlIDs string

	//create cluster
	publicIP                                                   bool
	argClusterName, argDBnames, argZones, argEnv, argGroupName string
	mysqlVersion                                               string

	ds                   = mdb.NewBytes("10G")
	DefaultMysqlResource = mdb.MysqlResources{
		ResourcePresetID: "s2.small",
		DiskSize:         &ds, //10G
		DiskTypeID:       "local-ssd",
	}
)

type Definitions struct {
	Grants    map[string][]string `yaml:"grant"`
	Instances map[string][]string `yaml:"instance"`
	DBHosts   map[string][]string `yaml:"db_host"`
}

func GroupingRoles(roles mdb.Roles, dbavail interface{}) (mdb.DatabasesRoles, mdb.Roles) {
	var dbnames mdb.MysqlDatabases
	if len(fmt.Sprint(dbavail)) > 0 {
		switch v := dbavail.(type) {
		case mdb.MysqlDatabase:
			dbnames = append(dbnames, v)
		case mdb.MysqlDatabases:
			dbnames = append(dbnames, v...)
		case []string:
			for _, i := range v {
				dbnames = append(dbnames, mdb.MysqlDatabase{Name: i})
			}
		default:
			dbnames = append(dbnames, mdb.MysqlDatabase{Name: fmt.Sprint(v)})
		}
	}

	var glperm mdb.Roles
	var dbperm mdb.DatabasesRoles
	for _, dbname := range dbnames {
		var tmp mdb.DatabaseRole
		for _, role := range roles {
			role = strings.ReplaceAll(role, " ", "_")
			if IgnoredRoles.HasRole(role) {
				logger.Info("role %s dont support, ignore it", role)
				continue
			}
			if mdb.IsGlobalPermition(role) {
				glperm = append(glperm, role)
			} else {
				tmp.Database = dbname.Name
				tmp.Roles = append(tmp.Roles, role)
			}
		}
		if tmp.Roles != nil && len(tmp.Roles) > 0 {
			dbperm = append(dbperm, tmp)
		}
	}
	return dbperm, glperm
}

func (gc GrantsConfig) ResourcesByGroup(groupName string) mdb.MysqlResources {
	mr := DefaultMysqlResource
	if config, ok := gc.Resources["common"]; ok {
		mr = config.Normalize()
	}

	subGroup := strings.Split(groupName, "-")
	if len(subGroup) > 1 {
		if resource, ok := gc.Resources[subGroup[0]]; ok {
			if md, err := mdb.MergeStructs(mr.Normalize(), resource.Normalize()); err == nil {
				var tmp mdb.MysqlResources
				if err := json.Unmarshal(md, &tmp); err == nil {
					mr = tmp
				}
			}
		}
	}
	if resource, ok := gc.Resources[groupName]; ok {
		if md, err := mdb.MergeStructs(mr.Normalize(), resource.Normalize()); err == nil {
			var tmp mdb.MysqlResources
			if err := json.Unmarshal(md, &tmp); err == nil {
				mr = tmp
			}
		}
	}
	return mr
}

func (gc GrantsConfig) PrepareBlocksByGroup(groupName string) []PrepareBlock {
	logger.Debug("%+v", gc.PrepareBlockGroups)
	pb, ok := gc.PrepareBlockGroups[groupName]
	if !ok {
		logger.Info("not found prepare block for %s", groupName)
		return []PrepareBlock{}
	}
	return pb
}

func (gc GrantsConfig) MysqlSettings57ByGroup(groupName string) mdb.MysqlConfig {
	mc := make(mdb.MysqlConfig)
	if config, ok := gc.Settings57["common"]; ok {
		mc = config
	}
	subGroup := strings.Split(groupName, "-")
	if len(subGroup) > 1 {
		if config, ok := gc.Settings57[subGroup[0]]; ok {
			for k, v := range config {
				mc[k] = v
			}
		}
	}
	if config, ok := gc.Settings57[groupName]; ok {
		for k, v := range config {
			mc[k] = v
		}
	}
	if len(mc) == 0 {
		logger.Crit("not found mysql config for group name: %s, %s-*, common. Use default settings", groupName, groupName)
	}
	return mc
}

func (gc GrantsConfig) MysqlSettings80ByGroup(groupName string) mdb.MysqlConfig {
	mc := make(mdb.MysqlConfig)
	if config, ok := gc.Settings80["common"]; ok {
		mc = config
	}
	subGroup := strings.Split(groupName, "-")
	if len(subGroup) > 1 {
		if config, ok := gc.Settings80[subGroup[0]]; ok {
			for k, v := range config {
				mc[k] = v
			}
		}
	}
	if config, ok := gc.Settings80[groupName]; ok {
		for k, v := range config {
			mc[k] = v
		}
	}
	if len(mc) == 0 {
		logger.Crit("not found mysql config for group name: %s, %s-*, common. Use default settings", groupName, groupName)
	}
	return mc
}

func (gc GrantsConfig) AllUserPermitions(availdb interface{}) mdb.UsersPermissions {
	var perms mdb.UsersPermissions
	for _, user := range gc.Usernames() {
		perm, err := gc.UsersPermissionByUser(user, availdb)
		if err != nil {
			logger.Crit("error user %s: %s", user, err)
			continue
		}
		perms = append(perms, perm)
	}
	return perms
}

func (gc GrantsConfig) UsersPermissionByUser(value string, availdb interface{}) (mdb.UserPermission, error) {
	var user mdb.UserPermission
	for _, rule := range gc.Rules {
		if !strings.EqualFold(rule.User, value) {
			continue
		}
		user.Username = rule.User
		switch v := rule.Grant.(type) {
		case []string:
			dbperm, glperm := GroupingRoles(mdb.Roles(v), rule.Databases(availdb)) //преобразуем звёздочку в имена баз
			user.DatabasePermissions = append(user.DatabasePermissions, dbperm...)
			user.GlobalPermissions = append(user.GlobalPermissions, glperm...)
			user.Password = rule.Password()
		case string:
			role := strings.Split(v, "/")
			if len(role) > 1 {
				name := role[1]
				roles, ok := gc.Definitions.Grants[name]
				if !ok {
					continue
				}
				dbperm, glperm := GroupingRoles(roles, rule.Databases(availdb))
				user.DatabasePermissions = append(user.DatabasePermissions, dbperm...)
				user.GlobalPermissions = append(user.GlobalPermissions, glperm...)
				user.Password = rule.Password()
			} else {
				dbperm, glperm := GroupingRoles(mdb.Roles(role), rule.Databases(availdb))
				user.DatabasePermissions = append(user.DatabasePermissions, dbperm...)
				user.GlobalPermissions = append(user.GlobalPermissions, glperm...)
				user.Password = rule.Password()
			}
		case []interface{}:
			buff := make([]string, len(v))
			for n, i := range v {
				buff[n] = fmt.Sprintf("%s", i)
			}
			dbperm, glperm := GroupingRoles(mdb.Roles(buff), rule.Databases(availdb))
			user.DatabasePermissions = append(user.DatabasePermissions, dbperm...)
			user.GlobalPermissions = append(user.GlobalPermissions, glperm...)
			user.Password = rule.Password()
		default:
			return user, fmt.Errorf("not found type %s: %s", user, v)
		}
	}
	return user, nil
}

func (gc GrantsConfig) Usernames() []string {
	var usernames []string
	for _, rule := range gc.Rules {
		usernames = append(usernames, rule.User)
	}
	return usernames
}

func (gc GrantsConfig) Databases() []string {
	var databases []string
	tmp := make(map[string]int)
	for _, resource := range gc.Resources {
		for _, db := range strings.Split(resource.Databases, ",") {
			tmp[db] += 1
		}
	}
	for k := range tmp {
		databases = append(databases, k)
	}
	return databases
}

type Rule struct {
	Name       string      `yaml:"name"`
	DBHost     interface{} `yaml:"db_host"`
	Instance   interface{} `yaml:"instance"`
	Grant      interface{} `yaml:"grant"`
	ON         string      `yaml:"on"`
	User       string      `yaml:"user"`
	ClientHost string      `yaml:"client_host"`
	Secret     string      `yaml:"password"`
}

func (r Rule) Databases(dbavail interface{}) mdb.MysqlDatabases {
	val := strings.Split(r.ON, ".")
	if val[0] == "*" && len(fmt.Sprint(dbavail)) > 0 {
		switch v := dbavail.(type) {
		case mdb.MysqlDatabases:
			return v
		case []string:
			var i mdb.MysqlDatabases
			for _, name := range v {
				i = append(i, mdb.MysqlDatabase{Name: name})
			}
			return i
		case string:
			var i mdb.MysqlDatabases
			for _, name := range strings.Split(fmt.Sprint(v), ",") {
				i = append(i, mdb.MysqlDatabase{Name: name})
			}
			return i
		default:
		}
	}
	return mdb.MysqlDatabases{
		mdb.MysqlDatabase{Name: val[0]},
	}
}

func (r Rule) Tables() []string {
	val := strings.Split(r.ON, ".")
	return []string{val[1]}
}

func (r Rule) Password() string {
	if strings.Contains(r.Secret, "ver") {
		cntx, cancel := context.WithTimeout(context.Background(), 60*time.Second)
		defer cancel()

		secret, err := yavClient.GetVersion(cntx, r.Secret)
		if err != nil {
			logger.Crit("error get secret %s: %s", r.Secret, err)
			return ""
		}
		if secret.Status == "error" {
			logger.Crit("error read secret %s: %s", r.Secret, secret.Err())
			return ""
		}
		for _, v := range secret.Version.Values {
			if strings.Contains(v.Key, r.User) {
				return v.Value
			}
		}
	}
	return r.Secret
}

type Rules []Rule

type PrepareBlock struct {
	Database string
	Script   string
}

type Resources map[string]mdb.MysqlResources
type Settings57 map[string]mdb.MysqlConfig
type Settings80 map[string]mdb.MysqlConfig
type PrepareBlockGroups map[string][]PrepareBlock //group + ppcdata1:/tmp/runnable_script.sh

type GrantsConfig struct {
	Definitions        `yaml:"definitions"`
	Rules              `yaml:"rules"`
	Resources          `yaml:"resources"`
	Settings57         `yaml:"settings_5_7"`
	Settings80         `yaml:"settings_8_0"`
	PrepareBlockGroups `yaml:"prepare"`
}

func (d Definitions) FindGrant(name string) ([]string, error) {
	if grants, ok := d.Grants[name]; ok {
		return grants, nil
	}
	return []string{}, fmt.Errorf("not found %s", name)
}

func LoadGrantsFile(name string) (GrantsConfig, error) {
	var cnf GrantsConfig
	data, err := ioutil.ReadFile(name)
	if err != nil {
		return cnf, fmt.Errorf("error read %s: %s", name, err)
	}
	err = yaml.Unmarshal(data, &cnf)
	return cnf, err
}

func LoadToken(path string) (string, error) {
	if v := os.Getenv("OAUTH"); len(v) > 0 {
		return v, nil
	}
	if !filepath.IsAbs(path) {
		path = filepath.Join(os.Getenv("HOME"), filepath.Base(path))
	}
	if _, err := os.Stat(path); err == nil {
		out, err := ioutil.ReadFile(path)
		return strings.Trim(fmt.Sprint(out), "\n"), err
	} else if os.IsNotExist(err) {
		return "", fmt.Errorf("token file %s dosnt exist", path)
	}
	return "", fmt.Errorf("error get token")
}

func HasKey(key string, keys []string) bool {
	for _, k := range keys {
		if k == key {
			return true
		}
	}
	return false
}

func main() {
	flag.StringVar(&folderID, "folder-id", DefaultFolderID, "id директории проекта в MDB")
	flag.StringVar(&mysqlIDs, "mysql-ids", DefaultMySQLID, "список id mysql в MDB")
	flag.StringVar(&mdbKey, "mdb-key", DefaultMDBKey, "ключ для работы с MDB")
	flag.BoolVar(&debug, "debug", false, "debug mode")
	flag.StringVar(&grantsFile, "grants-file", DefaultGrantsFile, "файл с грантами в формате yaml")
	flag.StringVar(&yavTokenFile, "yav-token-path", "./.yav.token", "файл где записан токе для секретницы")
	flag.StringVar(&YavVersion, "yav-version", DefaultYavVersion, "версия ключа в YAV")
	flag.StringVar(&login, "login", "", "логин со staff(если текущий USER отличается)")
	flag.StringVar(&command, "command", "", "команда для выполнения: create-cluster/update-users")

	flag.StringVar(&argClusterName, "cluster-name", "", "имя нового кластера")
	flag.StringVar(&argDBnames, "dbnames", "", "имя баз данных для файла с грантами(парсин звездочки)")
	flag.StringVar(&argZones, "zones", "", "список через запятую зон")
	flag.BoolVar(&publicIP, "public-ip", false, "использовать публичный IP адрес")
	flag.StringVar(&argEnv, "enviroment", DefaultEnviroment, "окружение по-умолчанию")
	flag.StringVar(&mysqlVersion, "5.7", DefaultMysqlVersion, "версия mysql")
	flag.StringVar(&argGroupName, "group-name", DefaultGroupName, "имя группы для поиска ресурсов в конфиге")
	flag.IntVar(&mysqlPort, "mysql-port", DefaultMysqlPort, "порт для подключения к БД mysql")
	flag.StringVar(&mysqlUser, "mysql-user", DefaultMysqlUser, "пользователь для подключения к БД")
	flag.Parse()

	logDirectFormat := logger.DirectMessageFormat("direct.back", "mysql-binlog-age")

	newlogger := log.New(os.Stdout, "", 0)
	logger.NewLoggerFormat(
		logger.WithLogLevel(logger.InfoLvl),
		logger.WithLogger(newlogger),
		logger.WithLogFormat(logDirectFormat),
	)

	//enable debug mode
	if debug {
		logger.NewLoggerFormat(
			logger.WithLogLevel(logger.DebugLvl),
			logger.WithLogger(newlogger),
			logger.WithLogFormat(logDirectFormat),
		)
	}

	yavToken, err := LoadToken(yavTokenFile)
	if err != nil {
		logger.Crit("error load yav token file %s: %s", yavTokenFile, err)
	}
	if !os.IsExist(err) {
		oauthData := oauthlib.NewOAuthToken(
			"",
			"",
			DefaultTokenFile,
			login,
		)
		yavToken, err = oauthData.LoadOAuthToken()
		if err != nil {
			logger.Crit("error load token %s: %s", oauthData, err)
			os.Exit(2)
		}
	}

	logger.Debug("success load token: %s", yavToken)

	cnf, err := LoadGrantsFile(grantsFile)
	if err != nil {
		logger.Crit("error load config %s: %s", grantsFile, err)
		os.Exit(1)
	}

	logger.Debug("config %+v", cnf)

	yavOpt := httpyav.WithOAuthToken(yavToken)
	yavClient, err = httpyav.NewClient(yavOpt)
	if err != nil {
		logger.Crit("error connect YAV: %s", err)
	}

	cntx, cancel := context.WithTimeout(context.Background(), 60*time.Second)
	defer cancel()

	res, err := yavClient.GetVersion(cntx, YavVersion)
	logger.Debug("read yav version %s: %+v, err %s", YavVersion, res, err)

	//
	configSpec := make(mdb.MysqlConfig)
	configSpec["version"] = mysqlVersion
	currentResource := cnf.ResourcesByGroup(argGroupName)
	configSpec["resources"] = currentResource
	logger.Debug("resources %+v", currentResource)

	//получаем список зон, в которых размещаем базы
	zoneSpecs := strings.Split(DefaultZoneSpecs, ",")
	if len(currentResource.Zones) > 0 {
		zoneSpecs = strings.Split(currentResource.Zones, ",")
	}

	if len(argZones) > 0 {
		zoneSpecs = strings.Split(argZones, ",")
	}

	//получаем список имен баз данных
	dbSpec := strings.Split(DefaultDBNames, ",")
	if len(currentResource.Databases) > 0 {
		dbSpec = strings.Split(currentResource.Databases, ",")
	}
	if len(argDBnames) > 0 {
		dbSpec = strings.Split(argDBnames, ",")
	}

	//получаем всех пользователей из конфига
	usersSpec := cnf.AllUserPermitions(dbSpec)

	if mysqlVersion == "5.7" {
		configSpec["mysqlConfig57"] = cnf.MysqlSettings57ByGroup(argGroupName)
	} else {
		configSpec["mysqlConfig80"] = cnf.MysqlSettings80ByGroup(argGroupName)
	}

	switch command {
	case "update-users":
		if done := MysqlUpdateUsers(argClusterName, cnf); done {
			os.Exit(0)
		}
		os.Exit(1)
	case "create-cluster":
		if done := MysqlCreateCluster(argClusterName, zoneSpecs, dbSpec, publicIP, argEnv,
			usersSpec, configSpec); !done {
			exitCode = 1
		}
		os.Exit(exitCode)
	case "update-cluster":
		if done := MysqlUpdateConfig(argClusterName, configSpec); !done {
			exitCode = 1
		}
		os.Exit(exitCode)
	case "update-databases":
		if done := MysqlUpdateDatabases(argClusterName, currentResource); !done {
			exitCode = 1
		}
		os.Exit(exitCode)
	case "prepare":
		//if done := MysqlPrepareDatabase(clusterName, cnf); !done {
		//	exitCode = 1
		//}
	default:
		logger.Crit("не указана команда(create-cluster/update-users/update-cluster/update-databases")
		os.Exit(1)
	}
}

func MysqlUpdateConfig(clusterName string, configSpec mdb.MysqlConfig) bool {
	cntx, cancel := context.WithTimeout(context.Background(), 1200*time.Second) //кластер может долго создаваться
	defer cancel()

	logger.Info("start update config cluster %s", clusterName)
	cluster, err := mdb.ConnectMysqlCluster("", folderID, mdbKey)
	if err != nil {
		logger.Crit("failed connect cluster, err connect mdb: %s", err)
		return false
	}

	id, err := cluster.FindClusterIDByName(clusterName)
	if err != nil {
		logger.Crit("not found clusterID with name %s: %s", clusterName, err)
		return false
	}
	cluster.SetClusterID(id)
	used, err := GetUsedSpaceByCluster(id)
	if err != nil {
		logger.Warn("%s", err)
	}

	if err := CheckFreeSpaceFromCluster(configSpec, used); err != nil {
		logger.Crit("%s", err)
		return false
	}

	if opid, err := cluster.UpdateClusterSettings(clusterName, configSpec); err == nil {
		if ok := mdb.WaitingOperationDone(cntx, opid.ID, cluster, 30*time.Second); !ok {
			logger.Crit("failed update config cluster %s", clusterName)
		} else {
			logger.Info("success update config cluster %s", clusterName)
			return true
		}
	} else {
		logger.Crit("failed update clusterID %s, error %s", clusterName, err)
	}

	return false

}

func MysqlCreateCluster(clusterName string, zoneSpecs, dbSpec []string, publicIP bool,
	env string, usersSpec mdb.UsersPermissions, configSpec mdb.MysqlConfig) bool {
	cntx, cancel := context.WithTimeout(context.Background(), 1200*time.Second) //кластер может долго создаваться
	defer cancel()

	logger.Info("start create cluster %s", clusterName)
	cluster, err := mdb.ConnectMysqlCluster("", folderID, mdbKey)
	if err != nil {
		logger.Crit("failed create cluster, err connect mdb: %s", err)
		return false
	}

	if opid, err := cluster.CreateCluster(clusterName, zoneSpecs, dbSpec, publicIP, env,
		usersSpec, configSpec); err == nil {
		if ok := mdb.WaitingOperationDone(cntx, opid.ID, cluster, 30*time.Second); !ok {
			logger.Crit("failed create cluster %s", clusterName)
		} else {
			logger.Info("success create cluster %s", clusterName)
			return true
		}
	} else {
		logger.Crit("failed create cluster %s, error %s", clusterName, err)
	}
	return false
}

func CheckFreeSpaceFromCluster(configSpec mdb.MysqlConfig, used int64) error {
	if resources, ok := configSpec["resources"]; ok {
		if v, ok := resources.(mdb.MysqlResources); ok {
			currentSpace := mdb.NewBytes(used)
			newSpace := *v.DiskSize
			if ok, err := (newSpace).Less(currentSpace); err == nil && ok {
				return fmt.Errorf("config space %s smaller than current %s", newSpace, currentSpace)
			} else if err == nil && !ok {
				return nil
			} else {
				return err
			}
		} else {
			return fmt.Errorf("error convert %s to MysqlResources", resources)
		}
	}
	return fmt.Errorf("not found section 'resources'")
}

func MysqlUpdateUsers(clusterName string, cnf GrantsConfig) bool {
	var exitCode int
	var ids []string //ClusterID

	cluster, err := mdb.ConnectMysqlCluster("", folderID, mdbKey)
	if err != nil {
		logger.Crit("failed update users, error connect mdb: %s", err)
		return false
	}

	for _, name := range strings.Split(clusterName, ",") {
		if len(name) == 0 {
			continue
		}
		if id, err := cluster.FindClusterIDByName(clusterName); err != nil {
			logger.Crit("not found clusterID with name %s: %s", clusterName, err)
		} else {
			ids = append(ids, id)
		}
	}

	if len(mysqlIDs) > 0 {
		ids = append(ids, strings.Split(mysqlIDs, ",")...)
	}

	for _, id := range ids {
		//create new connection with MDB
		cluster.SetClusterID(id)

		hosts, err := cluster.ListHosts()
		logger.Debug("list hosts for cluster %s: %s, error: %s", *cluster.ClusterID, hosts, err)
		dbnames, err := cluster.AllDatabases()
		if err != nil {
			logger.Crit("error get databases: %s", err)
			continue
		}

		//Получаем список пользователей в БД для сравнения с грантами в файле.
		currentMysqlUsers, err := cluster.AllDatabaseUsers()
		if err != nil {
			logger.Crit("error get users by cluster %s: %s", *cluster.ClusterID, err)
			os.Exit(1)
		}
		logger.Debug("database users: %s, error %s", currentMysqlUsers, err)

		for _, username := range cnf.Usernames() {
			//Ищем пользователей из конфига в базе MySQL
			perm, err := cnf.UsersPermissionByUser(username, dbnames) //dbnames нужна для преобразования звездочки
			if err != nil {
				logger.Warn("error UsersPermissionByUser %s", err)
			}
			if _, ok := currentMysqlUsers.FindUser(username); ok {
				//Если пользователь найден в БД, то обновляем гранты у него
				logger.Info("user %s found. Updating grants", username)
				if opid, err := cluster.UpdateMysqlUser(perm); err != nil {
					logger.Crit("error update user %s: %s", username, err)
					exitCode = 2
				} else {
					cntx, cancel := context.WithTimeout(context.Background(), 120*time.Second)
					if ok := mdb.WaitingOperationDone(cntx, opid.ID, cluster, 30*time.Second); !ok {
						exitCode = 2
						cancel()
						break //вероятнее всего кластер в updating, останавливаем обновление
					} else {
						cancel()
						logger.Info("success update user %s", username)
					}
				}
			} else {
				//Если пользователя нет в БД, то создаем
				logger.Info("user %s not found. Creating grants", username)
				if opid, err := cluster.CreateMysqlUser(perm); err != nil {
					logger.Crit("error create user %s: %s", username, err)
					exitCode = 2
				} else {
					cntx, cancel := context.WithTimeout(context.Background(), 120*time.Second)
					if ok := mdb.WaitingOperationDone(cntx, opid.ID, cluster, 30*time.Second); !ok {
						exitCode = 2
						cancel()
						break //вероятнее всего кластер в updating, останавливаем обновление
					} else {
						cancel()
						logger.Info("success create user %s", username)
					}
				}
			}
		}
		var deletedUsers []mdb.UserPermission
		for _, user := range currentMysqlUsers {
			if !HasKey(user.Username, cnf.Usernames()) {
				deletedUsers = append(deletedUsers, user)
			}
		}

		for _, duser := range deletedUsers {
			if opid, err := cluster.RemoveMysqlUser(duser); err != nil {
				logger.Crit("error remove user %s: %s", duser.Username, err)
				exitCode = 2
			} else {
				cntx, cancel := context.WithTimeout(context.Background(), 120*time.Second)
				if ok := mdb.WaitingOperationDone(cntx, opid.ID, cluster, 30*time.Second); !ok {
					exitCode = 2
					cancel()
					break //вероятнее всего кластер в updating, останавливаем обновление
				} else {
					cancel()
					logger.Info("success delete user %s", duser.Username)
				}
			}
		}
	}

	return exitCode == 0
}

func MysqlUpdateDatabases(clusterName string, resources mdb.MysqlResources) bool {
	var exitCode int
	var ids []string //ClusterID

	cluster, err := mdb.ConnectMysqlCluster("", folderID, mdbKey)
	if err != nil {
		logger.Crit("failed update users, error connect mdb: %s", err)
		return false
	}

	for _, name := range strings.Split(clusterName, ",") {
		if len(name) == 0 {
			continue
		}
		if id, err := cluster.FindClusterIDByName(clusterName); err != nil {
			logger.Crit("not found clusterID with name %s: %s", clusterName, err)
		} else {
			ids = append(ids, id)
		}
	}

	if len(mysqlIDs) > 0 {
		ids = append(ids, strings.Split(mysqlIDs, ",")...)
	}

	for _, id := range ids {
		//create new connection with MDB
		cluster.SetClusterID(id)

		currentDatabases, err := cluster.AllDatabases()
		if err != nil {
			logger.Crit("error get databases: %s", err)
			continue
		}

		logger.Debug("current databases: %s, error %s", currentDatabases, err)

		for _, dbname := range resources.DatabaseNames() {
			//Ищем пользователей из конфига в базе MySQL
			if !HasKey(dbname, currentDatabases.Names()) {

				//Если пользователь найден в БД, то обновляем гранты у него
				logger.Info("databases %s not found. Create database", dbname)
				if opid, err := cluster.CreateDatabase(dbname); err != nil {
					logger.Crit("error create database %s: %s", dbname, err)
					exitCode = 2
				} else {
					cntx, cancel := context.WithTimeout(context.Background(), 120*time.Second)
					if ok := mdb.WaitingOperationDone(cntx, opid.ID, cluster, 30*time.Second); !ok {
						exitCode = 2
						cancel()
						break //вероятнее всего кластер в updating, останавливаем обновление
					} else {
						cancel()
						logger.Info("success create database %s", dbname)
					}
				}
			}
		}
		for _, dbname := range currentDatabases.Names() {
			if !HasKey(dbname, resources.DatabaseNames()) {
				logger.Crit("not found database '%s' in config. "+
					"Please manual remove database for %s(%s)", dbname,
					argClusterName, cluster.GetClusterID())
			}
		}
	}

	return exitCode == 0
}

/*
func MysqlPrepareDatabase(clusterName string, cnf GrantsConfig) bool {
	var ids []string

	cluster, err := mdb.ConnectMysqlCluster("", folderID, mdbKey)
	if err != nil {
		logger.Crit("failed update users, error connect mdb: %s", err)
		return false
	}

	for _, name := range strings.Split(clusterName, ",") {
		if len(name) == 0 {
			continue
		}
		if id, err := cluster.FindClusterIDByName(clusterName); err != nil {
			logger.Crit("not found clusterID with name %s: %s", clusterName, err)
		} else {
			ids = append(ids, id)
		}
	}

	if len(mysqlIDs) > 0 {
		ids = append(ids, strings.Split(mysqlIDs, ",")...)
	}

	mysqlMasters := make(map[string]string) //map[ClusterName]MasterHost
	clustersConfings, _ := cluster.AllClusters()

	for _, id := range ids {
		name, ok := clustersConfings.ClusterNameByID(id)
		if !ok {
			logger.Crit("not found ClusterID %s", id)
			continue
		}
		cluster.SetClusterID(id)
		hosts, err := cluster.ListHosts()
		if err != nil {
			logger.Crit("error get hosts for %s", id)
			continue
		}
		mysqlMasters[name.Name] = hosts.GetMaster()
	}

	prepareBlocks := cnf.PrepareBlocksByGroup(groupName)
	if len(prepareBlocks) == 0 {
		logger.Crit("empty prepare blocks")
		return false
	}

	user, err := cnf.UsersPermissionByUser(mysqlUser, "")
	if err != nil {
		logger.Crit("dont found user %s for connect to mysql", mysqlUser)
		return false
	}

	var consoles []Console

	for clusterName, masterHost := range mysqlMasters {
		dbname := "ppc"
		if strings.Contains(clusterName, "ppcdict") {
			dbname = "ppcdict"
		}
		myconfig := NewMySQLConfig(masterHost, mysqlPort, user.Username, user.Password, dbname)
		consoles = append(consoles, NewConsole(myconfig))
	}

	var wg sync.WaitGroup
	for _, console := range consoles {
		wg.Add(1)
		go func(console Console) {
			wg.Done()
			console.StartConsole()
			defer console.Close()
		}(console)
	}

	time.Sleep(10 * time.Second)

	for _, pb := range prepareBlocks {
		for _, console := range consoles {
			fmt.Println("CONS", console.Config.db, pb.Database)
			if strings.Contains(console.Config.db, pb.Database) {
				logger.Info("start script %s console %s", pb.Script, console.SocketPath())
				cmd := exec.Command(pb.Script, console.SocketPath())
				stderr, _ := cmd.StderrPipe()
				if err := cmd.Start(); err != nil {
					logger.Crit("error start command %s: %s", pb.Script, err)
					continue
				}

				msg, _ := ioutil.ReadAll(stderr)
				if err := cmd.Wait(); err != nil {
					logger.Crit("fail script %s console %s: %s strerr: %s", pb.Script, console.SocketPath(), err, msg)
				} else {
					logger.Info("success script %s console %s", pb.Script, console.SocketPath())
				}
			}
		}
	}

	for _, console := range consoles {
		console.Close()
	}

	wg.Wait()
	return true
}

type MySQLConfig struct {
	host   string
	port   int
	db     string
	user   string
	passwd string
}

func NewMySQLConfig(host string, port int, user, passwd string, db string) MySQLConfig {
	return MySQLConfig{
		host:   host,
		passwd: passwd,
		user:   user,
		port:   port,
		db:     db,
	}
}


func CreateConnect(c MySQLConfig) (*myclient.Conn, error) {
	addressRow := fmt.Sprintf("%s:%d", c.host, c.port)
	logger.Debug("connect to %s", addressRow)
	conn, err := myclient.Connect(addressRow, c.user, c.passwd, c.db)
	if err != nil {
		return nil, err
	}
	if err := conn.Ping(); err != nil {
		return nil, err
	}
	return conn, nil
}

type Console struct {
	Config  MySQLConfig
	InChan  chan []byte
	OutChan chan []byte
	Lock    *sync.WaitGroup
	Cancel  context.CancelFunc
	Socket  *net.Listener
	Running *bool
	Cntx    context.Context
}

func NewConsole(c MySQLConfig) Console {
	var wg sync.WaitGroup
	cntx, cancel := context.WithCancel(context.Background())
	var mysock net.Listener //создаем пустой объект сокета, который потом заполним
	var running = false
	in := make(chan []byte)
	out := make(chan []byte)
	return Console{
		Config:  c,
		InChan:  in,
		OutChan: out,
		Lock:    &wg,
		Cancel:  cancel,
		Socket:  &mysock,
		Running: &running,
		Cntx:    cntx,
	}
}

func (c Console) SocketPath() string {
	return fmt.Sprintf("/tmp/%s.sock", c.Config.host)
}

func (c Console) Close() {
	logger.Debug("remove socket file %s", c.SocketPath())
	if err := os.Remove(c.SocketPath()); err != nil {
		logger.Crit("error %s", c.SocketPath(), err)
	}
	c.Cancel()
}

func (c Console) StartConsole() {
	/*
	defer func() {
		*c.Running = false
	}()
	defer c.Cancel()
	c.Lock.Add(1)
	defer c.Lock.Done()
	sq, err := CreateConnect(c.Config)
	if err != nil {
		logger.Crit("error connect %+v: %s", c.Config, err)
	}

	*c.Socket, err = net.Listen("unix", c.SocketPath())
	if err != nil {
		logger.Crit("error create socket")
		return
	}

	go func(mysock net.Listener) {
		*c.Running = true
		for {
			fd, err := mysock.Accept()
			if err != nil {
				logger.Crit("error create connect")
			}

			cntx, cancel := context.WithCancel(context.Background())
			go func() {
				defer func() { _ = fd.Close() }()
				ch := make(chan []byte, 1000)

				go func() {
					for {
						select {
						case out, ok := <-c.OutChan:
							if !ok {
								logger.Debug("closed outgoing chan")
								return
							}
							logger.Debug("receive data to socket: %s", out)
							if _, err := fd.Write(out); err != nil {
								logger.Crit("error write socket")
							}
						case <-cntx.Done():
							logger.Info("close cntx")
							return
						}
					}
				}()

				var buffer []byte
			L:
				for {
					buf := make([]byte, 512)
					i, err := fd.Read(buf)
					buf = buf[:i]

					if len(buf) > 0 {
						buffer = append(buffer, buf...)
						if bytes.HasSuffix(buf, []byte(";")) {
							logger.Debug("send data to socket: %s")
							c.InChan <- buffer
							buffer = []byte{}
						}
					}

					switch err {
					case io.EOF:
						break L
					default:
						break L
					}
				}
				logger.Debug("close chan")
				close(ch)
				cancel()
			}()
		}
	}(*c.Socket)

	defer func() { _ = sq.Close() }()
L:
	for {
		select {
		case inch := <-c.InChan:
			go func(inch []byte) {
				myResponse := MysqlResponse{}
				command := fmt.Sprintf("%s", inch)
				logger.Debug("run execute command %s", command)
				result, err := sq.Execute(command)
				if err != nil {
					logger.Crit("error exec command %s: %s", command, err)
					myResponse.Error = err
				} else {
					defer result.Close()
					if result.Resultset == nil {
						t := make(map[string]interface{})
						t["status"] = "success"
						myResponse.Out = t
					} else {
						mask := make(map[int]string)
						for name, pos := range result.FieldNames {
							mask[pos] = name
						}

						var data []map[string]interface{}
						for _, line := range result.Values {
							tmp := make(map[string]interface{})
							for i, d1 := range line {
								switch v := d1.Value().(type) {
								case []byte:
									tmp[mask[i]] = fmt.Sprintf("%s", v)
								default:
									tmp[mask[i]] = v
								}
							}
							data = append(data, tmp)
						}
						myResponse.Out = data
					}
				}
				jdata, err := json.Marshal(myResponse)
				if err != nil {
					logger.Crit("error marshal %s: %s", c.OutChan, err)
				}
				select {
				case c.OutChan <- jdata:
				default:
				}
				//result.Close()
				logger.Debug("finish execute command %s", command)
			}(inch)
		case <-c.Cntx.Done():
			logger.Info("closed context, exit")
			break L
		default:
			time.Sleep(5 * time.Second)
		}
	}
	logger.Debug("exit mysql connect %+v", c.Config)
	return


}*/

type MysqlResponse struct {
	Out   interface{}
	Error interface{}
}
