package main

import (
	"a.yandex-team.ru/direct/infra/go-libs/pkg/mdb"
	"bytes"
	"encoding/json"
	"fmt"
	"strings"
	"testing"
)

var (
	settingsConfig = map[string]mdb.MysqlConfig{
		"common": map[string]interface{}{
			"transactionIsolation":   "READ_COMMITTED",
			"autoIncrementIncrement": 1,
			"autoIncrementOffset":    1,
			"sqlMode":                []string{"NO_ENGINE_SUBSTITUTION"},
		},
		"dev7": map[string]interface{}{
			"autoIncrementOffset": 2,
		},
		"dev7-ppcdict": map[string]interface{}{
			"autoIncrementIncrement": 5,
		},
	}

	size1 = mdb.NewBytes("1G")
	size2 = mdb.NewBytes("1048576")
	size5 = mdb.NewBytes("100G")

	resourcesConfig = map[string]mdb.MysqlResources{
		"common": mdb.MysqlResources{
			ResourcePresetID: "s2.small",
			DiskSize:         &size1,
			DiskTypeID:       "local-ssd",
			Zones:            "sas,vla",
			Databases:        "ppc",
		},
		"dev7": mdb.MysqlResources{
			ResourcePresetID: "s2.medium",
		},
		"dev7-ppcdict": mdb.MysqlResources{
			DiskTypeID: "local-hdd",
		},
		"dev7-ppcdata": mdb.MysqlResources{
			DiskTypeID: "local-hdd",
			DiskSize:   &size5,
		},
	}

	grantsConfig = GrantsConfig{
		Settings57: settingsConfig,
		Settings80: settingsConfig,
		Resources:  resourcesConfig,
	}

	goodMarshal = []byte("{\"resourcePresetId\":\"s2.small\",\"diskSize\":\"1073741824\",\"diskTypeId\":\"local-ssd\"," +
		"\"zones\":\"sas,vla\",\"databases\":\"ppc\"}")

	size3 = mdb.NewBytes("1.5G")
	size4 = mdb.NewBytes("1572864")

	resourcesConfig2 = map[string]mdb.MysqlResources{
		"common": mdb.MysqlResources{
			ResourcePresetID: "s2.small",
			DiskSize:         &size3,
			DiskTypeID:       "local-ssd",
			Zones:            "sas,vla",
			Databases:        "ppc",
		},
		"dev7": mdb.MysqlResources{
			ResourcePresetID: "s2.medium",
		},
		"dev7-ppcdict": mdb.MysqlResources{
			DiskTypeID: "local-hdd",
		},
		"dev7-ppcdata": mdb.MysqlResources{
			DiskTypeID: "local-hdd",
			DiskSize:   &size5,
		},
	}

	solomonSensorResponse = []byte("{\"vector\":[{\"timeseries\":{\"alias\":\"\",\"kind\":\"DGAUGE\",\"type\":\"DGAUGE\"," +
		"\"labels\":{\"cluster\":\"mdb_mdb2ktk8f8nilm6c8b8i\",\"node\":\"by_host\",\"service\":\"mdb\", " +
		"\"host\":\"sas-rem5ke8n5yhl0js4.db.yandex.net\",\"name\":\"disk-used_bytes_mysql_data\", " +
		"\"project\":\"internal-mdb\",\"sensor\":\"disk-used_bytes_mysql_data\",\"shard\":\"fooa07bcrr7souccreru\"," +
		"\"dc\":\"by_host\"},\"timestamps\":[1607510510000,1607510703000,1607510896000,1607511089000,1607511282000," +
		"1607511475000,1607511668000,1607511861000,1607512054000],\"values\":[-1.0442733845384616E9,-1.0442728467692307E9," +
		"-1.0442754213333334E9,-1.0442724099230769E9,-1.0442717964615384E9,-1.0442712518461539E9,-1.0442717940769231E9," +
		"-1.0442707618461539E9,-1.0442725436666666E9]}},{\"timeseries\":{\"alias\":\"\",\"kind\":\"DGAUGE\",\"type\":\"DGAUGE\"," +
		"\"labels\":{\"cluster\":\"mdb_mdb2ktk8f8nilm6c8b8i\",\"node\":\"by_host\",\"service\":\"mdb\"," +
		"\"host\":\"myt-xtuoxth3r96hst5n.db.yandex.net\",\"name\":\"disk-used_bytes_mysql_data\",\"project\":\"internal-mdb\"," +
		"\"sensor\":\"disk-used_bytes_mysql_data\",\"shard\":\"fooa07bcrr7souccreru\",\"dc\":\"by_host\"}," +
		"\"timestamps\":[1607510510000,1607510703000,1607510896000,1607511089000,1607511282000,1607511475000,1607511668000," +
		"1607511861000,1607512054000],\"values\":[5.153725696369231E10,5.153725614530769E10,5.1537252015916664E10," +
		"5.153725541261539E10,5.153725656561539E10,5.1537256537E10,5.153725645946154E10,5.153725605546154E10," +
		"5.1537257044333336E10]}},{\"timeseries\":{\"alias\":\"\",\"kind\":\"DGAUGE\",\"type\":\"DGAUGE\"," +
		"\"labels\":{\"cluster\":\"mdb_mdb2ktk8f8nilm6c8b8i\",\"node\":\"by_host\",\"service\":\"mdb\"," +
		"\"host\":\"man-9uld86zjwpre3gwh.db.yandex.net\",\"name\":\"disk-used_bytes_mysql_data\",\"project\":\"internal-mdb\"," +
		"\"sensor\":\"disk-used_bytes_mysql_data\",\"shard\":\"fooa07bcrr7souccreru\",\"dc\":\"by_host\"},\"values\":[]}}," +
		"{\"timeseries\":{\"alias\":\"\",\"kind\":\"DGAUGE\",\"type\":\"DGAUGE\",\"labels\":{\"cluster\":\"mdb_mdb2ktk8f8nilm6c8b8i\"," +
		"\"node\":\"by_host\",\"service\":\"mdb\",\"host\":\"vla-90pa7kkfuoh0az6y.db.yandex.net\"," +
		"\"name\":\"disk-used_bytes_mysql_data\",\"project\":\"internal-mdb\",\"sensor\":\"disk-used_bytes_mysql_data\"," +
		"\"shard\":\"fooa07bcrr7souccreru\",\"dc\":\"by_host\"},\"timestamps\":[1607510510000,1607510703000,1607510896000," +
		"1607511089000,1607511282000,1607511475000,1607511668000,1607511861000,1607512054000],\"values\":[5.130229122823077E10," +
		"5.130229210515385E10,5.1302294811166664E10,5.1302290387E10,5.130229105630769E10,5.130229131584615E10," +
		"5.130229096153846E10,5.130229055376923E10,5.1302293316666664E10]}}]}")
)

func TestMysqlSettings57ByGroup(t *testing.T) {
	mycnf := grantsConfig.MysqlSettings57ByGroup("dev7-ppcdict")
	good1 := 2
	if v, ok := mycnf["autoIncrementOffset"]; ok && v != 2 {
		t.Errorf("error autoIncrementOffset, need=%d, current=%d", good1, v)
	}
	good2 := 5
	if v, ok := mycnf["autoIncrementIncrement"]; ok && v != 5 {
		t.Errorf("error autoIncrementIncrement, need=%d, current=%d", good2, v)
	}
	good3 := "READ_COMMITTED"
	if v, ok := mycnf["transactionIsolation"]; ok && !strings.EqualFold(v.(string), good3) {
		t.Errorf("error transactionIsolation, need=%s, cureent=%s", good3, v)
	}
}

func TestMysqlSettings80ByGroup(t *testing.T) {
	mycnf := grantsConfig.MysqlSettings80ByGroup("dev7-ppcdict")
	good1 := 2
	if v, ok := mycnf["autoIncrementOffset"]; ok && v != good1 {
		t.Errorf("error autoIncrementOffset, need=%d, current=%d", good1, v)
	}
	good2 := 5
	if v, ok := mycnf["autoIncrementIncrement"]; ok && v != good2 {
		t.Errorf("error autoIncrementIncrement, need=%d, current=%d", good2, v)
	}
	good3 := "READ_COMMITTED"
	if v, ok := mycnf["transactionIsolation"]; ok && !strings.EqualFold(v.(string), good3) {
		t.Errorf("error transactionIsolation, need=%s, cureent=%s", good3, v)
	}
}

func TestResourcesByGroup(t *testing.T) {
	resource := grantsConfig.ResourcesByGroup("dev7-ppcdict")
	good1 := "local-hdd"
	if !strings.EqualFold(resource.DiskTypeID, good1) {
		t.Errorf("error DiskTypeID, need=%s, current=%s", good1, resource.DiskTypeID)
	}
	good2 := "s2.medium"
	if !strings.EqualFold(resource.ResourcePresetID, good2) {
		t.Errorf("error ResourcePresetID, need=%s, current=%s", good2, resource.ResourcePresetID)
	}
	good3 := "ppc"
	if !strings.EqualFold(resource.Databases, good3) {
		t.Errorf("error Databases, need=%s, current=%s", good3, resource.Databases)
	}
	good4 := "1073741824"
	if !strings.EqualFold(string(*resource.DiskSize), good4) {
		t.Errorf("error Databases, need=%s, current=%s", good4, resource.DiskSize)
	}
}

func TestConvertToBytes1(t *testing.T) {
	sizeBytes := size1.String()
	good := "1073741824"
	if !strings.EqualFold(sizeBytes, good) {
		t.Errorf("error ConvertToBytes, need=%s, current=%s", good, sizeBytes)
	}
}

func TestConvertToBytes2(t *testing.T) {
	sizeBytes := size2.String()
	good := "1048576"
	if !strings.EqualFold(sizeBytes, good) {
		t.Errorf("error ConvertToBytes, need=%s, current=%s", good, sizeBytes)
	}
}

func TestConvertToBytes3(t *testing.T) {
	sizeBytes := size3.String()
	good := "1610612736"
	if !strings.EqualFold(sizeBytes, good) {
		t.Errorf("error ConvertToBytes, need=%s, current=%s", good, sizeBytes)
	}
}

func TestHumanize1(t *testing.T) {
	sizeBytes := size1.String()
	humSize := mdb.Bytes(sizeBytes).Humanize()
	good := "1.00GB"
	if !strings.EqualFold(humSize, good) {
		t.Errorf("error ConvertToBytes, need=%s, current=%s", good, humSize)
	}
}

func TestHumanize2(t *testing.T) {
	sizeBytes := size2.String()
	humSize := mdb.Bytes(sizeBytes).Humanize()
	good := "1.00MB"
	if !strings.EqualFold(humSize, good) {
		t.Errorf("error ConvertToBytes, need=%s, current=%s", good, humSize)
	}
}

func TestMarshalJSONResources(t *testing.T) {
	resources, err := json.Marshal(resourcesConfig["common"])
	if err != nil {
		t.Errorf("error MarshalJSONResource, error %s", err)
	}
	if !bytes.EqualFold(resources, goodMarshal) {
		t.Errorf("error MarshalJSONResources equal %s != %s", resources, goodMarshal)
	}
}

/* Проверяем, что значение места в конфиге не больше доступного в кластере.
 */
func TestSolomonMaxValuesGood(t *testing.T) {
	var vals SolomonDataBlock
	if err := json.Unmarshal(solomonSensorResponse, &vals); err != nil {
		t.Errorf("error TestSolomonMaxValuesGood unmarshal: %s", err)
	}
	good := int64(51537257044)
	if used := int64(vals.MaxValueAllVectors()); used != good {
		t.Errorf("error TestSolomonMaxValuesGood, need=%d, current=%d", used, good)
	} else {
		configSpec := make(mdb.MysqlConfig)
		r := grantsConfig.ResourcesByGroup("dev7-ppcdata")
		configSpec["resources"] = r
		fmt.Println(r)
		err := CheckFreeSpaceFromCluster(configSpec, used)
		if err != nil {
			t.Errorf("error TestSolomonMaxValuesGood/CheckFreeSpaceFromCluster, %s", err)
		}
	}
}

/* Проверяем, что значение места в конфиге меньше доступного в кластере.
 */
func TestSolomonMaxValuesFailed(t *testing.T) {
	var vals SolomonDataBlock
	if err := json.Unmarshal(solomonSensorResponse, &vals); err != nil {
		t.Errorf("error TestSolomonMaxValuesFailed unmarshal: %s", err)
	}
	good := int64(51537257044)
	if used := int64(vals.MaxValueAllVectors()); used != good {
		t.Errorf("error TestSolomonMaxValuesFailed, need=%d, current=%d", used, good)
	} else {
		configSpec := make(mdb.MysqlConfig)
		r := grantsConfig.ResourcesByGroup("dev7")
		configSpec["resources"] = r
		err := CheckFreeSpaceFromCluster(configSpec, used)
		if err == nil {
			t.Errorf("error TestSolomonMaxValuesFailed/CheckFreeSpaceFromCluster, need not empty error")
		}
	}
}
