package main

import (
	"a.yandex-team.ru/direct/infra/go-libs/pkg/oauthlib"
	"a.yandex-team.ru/direct/infra/go-libs/pkg/sslmode"
	"bytes"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"math"
	"net/http"
	"os"
	"strings"
	"time"

	logger "a.yandex-team.ru/direct/infra/go-libs/pkg/logformat"
)

const (
	SolomonAPI = "https://solomon.yandex.net/api/v2"
)

/*
curl https://solomon.yandex.net/api/v2/projects/internal-mdb/sensors/data
 -X POST -H "Authorization: OAuth AQAD-q" -d '{"program":"{project=\"internal-mdb\",
 cluster=\"mdb_mdb2ktk8f8nilm6c8b8i\",service=\"mdb\",host=\"*\",dc=\"by_host\",
 sensor=\"disk-used_bytes_mysql_data\"}","downsampling":{"aggregation":"AVG","points":100},
 "from":"2019-06-06T10:39:28.864Z","to":"2019-06-06T11:08:28.864Z"}' -H 'Content-type: application/json'  -v
*/

type SolomonDataBlock struct {
	Vector SolomonVectorsBlock `json:"vector"`
}

func (sd SolomonDataBlock) MaxValueAllVectors() float64 {
	var maxv float64
	for _, i := range sd.Vector {
		maxv = math.Max(maxv, i.Timeseries.MaxValue())
	}
	return maxv
}

type SolomonSensorsData struct {
	Program      interface{}              `json:"program"` //SolomonProgramBlock -> string
	Downsampling SolomonDownsamplingBlock `json:"downsampling"`
	From         string                   `json:"from"`
	To           string                   `json:"to"`
}

type SolomonProgramBlock map[string]string

func (sp SolomonProgramBlock) String() string {
	var result []string
	for k, v := range sp {
		result = append(result, strings.ToLower(fmt.Sprintf("%s=\"%s\"", k, v)))
	}
	return "{" + strings.Join(result, ",") + "}"
}

type SolomonDownsamplingBlock struct {
	Aggregation string
	Points      int
}

/*
{"vector":[{"timeseries":{"alias":"","kind":"DGAUGE","type":"DGAUGE","labels":{"cluster":"mdb_mdb2ktk8f8nilm6c8b8i",
 "node":"by_host","service":"mdb","host":"sas-rem5ke8n5yhl0js4.db.yandex.net","name":"disk-used_bytes_mysql_data",
 "project":"internal-mdb","sensor":"disk-used_bytes_mysql_data","shard":"fooa07bcrr7souccreru","dc":"by_host"},
 "values":[]}},{"timeseries":{"alias":"","kind":"DGAUGE","type":"DGAUGE" ... }]}
*/
type SolomonVectorsBlock []SolomonVectorBlock

type SolomonVectorBlock struct {
	Timeseries SolomonTimeseriesBlock
}

type SolomonTimeseriesBlock struct {
	Values []float64     `json:"values"`
	Labels SolomonLabels `json:"labels"`
}

func (stb SolomonTimeseriesBlock) MaxValue() float64 {
	var maxv float64
	for _, i := range stb.Values {
		maxv = math.Max(maxv, i)
	}
	return maxv
}

type SolomonLabels struct {
	Host string `json:"host"`
	Name string `json:"name"`
	SolomonProgramBlock
}

func GetUsedSpaceByCluster(clusterID string) (int64, error) {
	oauthData := oauthlib.NewOAuthToken(
		"",
		"",
		"",
		os.Getenv("USER"),
	)
	oauthToken, err := oauthData.LoadOAuthToken()
	if err != nil {
		return 0, fmt.Errorf("error load token %v: %s", oauthData, err)
	}
	logger.Debug("new OAuth token: %s\n", oauthToken)

	curtime := time.Now().UTC()
	toTime := fmt.Sprint(curtime.Format("2006-01-02T15:04:05.999Z"))
	curtime = curtime.Add(-1 * time.Hour)
	fromTime := fmt.Sprint(curtime.Format("2006-01-02T15:04:05.999Z"))
	logger.Info("time from %s to %s", fromTime, toTime)
	s := SolomonSensorsData{
		Program: SolomonProgramBlock{
			"project": "internal-mdb",
			"cluster": fmt.Sprintf("mdb_%s", clusterID),
			"service": "mdb",
			"host":    "*",
			"dc":      "by_host",
			"sensor":  "disk-used_bytes_mysql_data",
		}.String(),
		Downsampling: SolomonDownsamplingBlock{
			Aggregation: "CURRENT",
			Points:      10,
		},
		From: fromTime,
		To:   toTime,
	}

	url := fmt.Sprintf("%s/%s", SolomonAPI, "projects/internal-mdb/sensors/data")
	data, _ := json.Marshal(s)
	out, err := SolomonRequest("POST", url, oauthToken, data)
	fmt.Printf("OUT %s, ERR %s\n", out, err)

	var vals SolomonDataBlock
	if err = json.Unmarshal(out, &vals); err != nil {
		return 0, err
	}
	return int64(vals.MaxValueAllVectors()), nil
}

func SolomonRequest(method, url, token string, data []byte) ([]byte, error) {
	client, err := sslmode.NewHTTPSClient()
	if err != nil {
		return nil, err
	}

	req, err := http.NewRequest(method, url, bytes.NewReader(data))
	if err != nil {
		return nil, err
	}

	req.Header.Set("Authorization", fmt.Sprintf("OAuth %s", token))
	req.Header.Set("Content-type", "application/json")
	resp, err := client.Do(req)
	if err != nil {
		return nil, err
	}
	defer func() { _ = resp.Body.Close() }()

	out, _ := ioutil.ReadAll(resp.Body)
	if resp.StatusCode != 200 {
		return nil, fmt.Errorf("http code: %s, req %s, data %s, out %s", resp.Status, url, data, out)
	}
	if len(out) == 0 {
		return nil, fmt.Errorf("empty body response %s", url)
	}

	return out, nil
}
