package main

import (
	"bytes"
	"context"
	//l "github.com/siddontang/go-log/log"
	//"github.com/siddontang/go-mysql/client"

	"database/sql"
	"flag"
	"fmt"
	"io"
	"io/ioutil"
	"log"
	"log/syslog"
	"os"
	"strings"
	"sync"
	"time"

	"encoding/json"
	"net/http"

	_ "github.com/go-sql-driver/mysql"
	"github.com/siddontang/go-mysql/mysql"
	r "github.com/siddontang/go-mysql/replication"

	"a.yandex-team.ru/direct/infra/go-libs/pkg/dbconfig"
	"a.yandex-team.ru/direct/infra/go-libs/pkg/juggler"
	logger "a.yandex-team.ru/direct/infra/go-libs/pkg/logformat"
	ct "a.yandex-team.ru/direct/infra/go-libs/pkg/mythreads"
	"a.yandex-team.ru/direct/infra/go-libs/pkg/ytlib"
	"a.yandex-team.ru/direct/infra/go-libs/pkg/zklib"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/yt/go/ytlock"
)

const (
	DefaultZookeeperServers      = "ppc-zk-1.da.yandex.ru:2181"
	DefaultZookeeperPathDBConfig = "/direct/np/db-config/db-config.dev7.json"
	DefaultYTLockPath            = "//home/direct-np/testing/apps/binlogage/lock"
	DefaultYTLockCluster         = "locke"
	DefaultYTLockToken           = "/tmp/ytToken"
	DefaultZookeperPassFile      = ""
	DefaultZookeperTokenFile     = ""
	DefaultZookeeperUser         = ""

	DefaultSyslogSocket = "/dev/log"
	DefaultEnviron      = "dev"

	StopWriteZookeperNode = "/direct_binlogage_%s_stopflag"

	SolomonAgentURL = "http://localhost:19090/direct/mysql-binlog-age-monitoring"
)

var (
	zkPath, zkServers, zkTokenFile, zkPassFile, zkUser string
	debugMode                                          bool

	zknode zklib.ZkNode

	jugglerAddress string

	syslogEnable bool
	syslogSocket string
	environ      string

	lockYTPath, lockYTToken, lockYTCluster string

	loglevel = logger.InfoLvl
)

type SolomonMetric struct {
	Labels map[string]string `json:"labels"`
	TS     int64             `json:"ts"`
	Value  float64           `json:"value"`
}

type SolomonMessage struct {
	Metrics []SolomonMetric `json:"metrics"`
}

func main() {
	flag.StringVar(&zkPath, "zkPath", DefaultZookeeperPathDBConfig, "db-config zookeeper path")
	flag.StringVar(&zkServers, "zkServers", DefaultZookeeperServers, "zookeeper servers")
	flag.StringVar(&zkTokenFile, "zkToken", DefaultZookeperTokenFile, "zookeeper token")
	flag.StringVar(&zkPassFile, "zkPassFile", DefaultZookeperPassFile, "zookeeper password file")
	flag.StringVar(&zkUser, "zkUser", DefaultZookeeperUser, "zookeeper user")
	flag.StringVar(&environ, "env", DefaultEnviron, "enviroment")
	flag.StringVar(&jugglerAddress, "jugglerAddress", "", "juggler client api address")
	flag.BoolVar(&debugMode, "debug", false, "enable debug mode")
	flag.StringVar(&syslogSocket, "syslogSocket", DefaultSyslogSocket, "syslog socket path")
	flag.BoolVar(&syslogEnable, "syslog", false, "enable syslog logger")
	flag.StringVar(&lockYTPath, "ytLockPath", DefaultYTLockPath, "YT path for lock")
	flag.StringVar(&lockYTCluster, "ytLockCluster", DefaultYTLockCluster, "YT cluster for lock")
	flag.StringVar(&lockYTToken, "ytLockToken", DefaultYTLockToken, "YT token for connect")
	flag.Parse()

	if debugMode {
		loglevel = logger.DebugLvl
	}

	maxprocs.AdjustYP()

	zkStopPath := fmt.Sprintf(StopWriteZookeperNode, environ)
	zkStopNode := zklib.NewZkNode(zkStopPath)

	dbcnf := dbconfig.NewDBConfig()

	logDirectFormat := logger.DirectMessageFormat("direct.back", "mysql-binlog-age")

	if syslogEnable {
		//для логировнаия используем syslog. Какой уровень логирования пиштеся, решается в обертке  logformat
		syslogger, err := syslog.Dial("unixgram", syslogSocket, syslog.LOG_DEBUG|syslog.LOG_DAEMON, os.Args[0])
		if err != nil {
			log.Fatal(err) //вывод в stderr и exit 1
		}
		logger.NewLoggerFormat(
			logger.WithLogger(syslogger),
			logger.WithLogLevel(loglevel),
			logger.WithLogFormat(logDirectFormat),
		)
	} else {
		newlogger := log.New(os.Stdout, "", 0)
		logger.NewLoggerFormat(
			logger.WithLogLevel(loglevel),
			logger.WithLogger(newlogger),
			logger.WithLogFormat(logDirectFormat),
		)
	}
	//ct.SetLogger(logger)

	tokenFile, err := os.Open(lockYTToken)
	if err != nil {
		logger.Crit("error open %s, %s", lockYTToken, err)
		return
	}
	defer func() { _ = tokenFile.Close() }()

	var ytclients []yt.Client
	for _, ytCluster := range strings.Split(lockYTCluster, ",") {
		ytclient, err := ytlib.NewYtClient(
			ytlib.WithLockClusterName(ytCluster),
			ytlib.WithLockPath(lockYTPath),
			ytlib.WithLockToken(tokenFile),
		)
		if err != nil {
			logger.Crit("error func(NewYtClient): %s", err)
		} else {
			ytclients = append(ytclients, ytclient)
		}
	}
	if len(ytclients) == 0 {
		logger.Crit("empty yt clients")
		return
	}

	/*
		if lock, err := ytlib.MustLockTransaction(ytclient); err != nil {
			logger.Crit("error func(MustLockTransaction): %s", err)
			return
		} else {
			defer func() { _ = ytlib.UnlockTransaction(lock) }()
		}*/
	//Берем блокировку из нескольких YT источников
	if locks, err := ytlib.MustMultiLockTransaction(ytclients); err != nil {
		logger.Crit("error func(MustMultiLockTransaction): %s", err)
		return
	} else {
		defer func(locks []*ytlock.Lock) {
			for _, l := range locks {
				_ = ytlib.UnlockTransaction(l)
			}
		}(locks)
		logger.Info("count yt locks %d", len(locks))
	}

	zkServers := strings.Split(zkServers, ",")
	zkAddress := zklib.NewZkAddress(zkUser, "", zkServers...)

	//вычитываем токен для zookeeper, если он присутствует в системе
	if len(zkTokenFile) > 0 && len(zkPassFile) > 0 {
		logger.Crit("error arguments: dont use '-zktoken-file' with '-zkpass-file'")
		log.Fatal()
	}

	if len(zkPassFile) > 0 && len(zkUser) == 0 {
		logger.Crit("error arguments: not found zkuser. Use '-zkuser' with '-zkpass-file'")
		log.Fatal()
	}

	if len(zkPassFile) > 0 && len(zkUser) > 0 {
		if mypass, err := ioutil.ReadFile(zkPassFile); err != nil {
			logger.Crit("error read %s: %s", zkPassFile, err)
		} else {
			zkPass := string(bytes.TrimSuffix(mypass, []byte("\n")))
			zkAddress = zklib.NewZkAddress(zkUser, zkPass, zkServers...)
		}
	}

	if len(zkTokenFile) > 0 {
		if mytoken, err := ioutil.ReadFile(zkTokenFile); err != nil {
			logger.Crit("error read %s: %s", zkTokenFile, err)
		} else {
			zkToken := string(bytes.TrimSuffix(mytoken, []byte("\n")))
			zkAddress = zklib.NewZkAddressWithToken(zkToken, zkServers...)
		}

	}

	zkconn, err := zklib.NewZkConnect(zkAddress)
	if err != nil {
		logger.Crit("error connect to %s, error %s", zkServers, err)
		os.Exit(1)
	}

	if ok, err := zkconn.ExistNode(zkStopNode); err == nil && ok {
		logger.Warn("found stop flag, exit monitoring")
		os.Exit(0)
	}

	zkconn, err = zklib.NewZkConnect(zkAddress)
	if err != nil {
		logger.Crit("error connect to %s, error %s", zkServers, err)
		os.Exit(1)
	}
	zknode = zklib.NewZkNode(zkPath)

	var wg sync.WaitGroup
	wg.Add(4)
	go ct.RunReadZookeeperThread(&zkconn, &zknode, &wg) //вычитываем конфиг из зукипера
	jugch := make(juggler.JugglerMessages, 1000)
	job := NewMySQLBinlogAgeJob()
	go ct.RunDatabaseWorkerThread(&dbcnf, &job, jugch, &wg) //запускаем процессы для работы с БД
	go ct.RunUpdateDBConfigThread(&dbcnf, &zknode, &wg)     //парсим конфиг dbconfig
	go ct.RunJugglerSender(jugch, &wg)                      //отправка событий в juggler
	wg.Wait()
}

type MySQLBinlogAgeJob struct {
	Settings   *ct.DatabaseWorker
	HTTPClient *http.Client
}

func NewMySQLBinlogAgeJob() MySQLBinlogAgeJob {
	return MySQLBinlogAgeJob{&ct.DatabaseWorker{}, &http.Client{}}
}

func (bj MySQLBinlogAgeJob) New(args interface{}) ct.Job {
	worker, ok := args.(ct.DatabaseWorker)
	if !ok {
		logger.Crit("unsupport meta data %t need *DatabaseWorker", worker)
		return MySQLBinlogAgeJob{}
	}
	return MySQLBinlogAgeJob{
		Settings:   &worker,
		HTTPClient: &http.Client{},
	}
}

type VariablesStruct struct {
	VariableName string
	Values       string
}

type SlaveHostsStruct struct {
	ServerID  int
	Host      string
	Port      int
	MasterID  int
	SlaveUUID string
}

func (bj MySQLBinlogAgeJob) Do() {
	jchan := bj.Settings.JugglerChan
	sqlCmdPurged := "SHOW VARIABLES LIKE '%gtid_purged%'"
	sqlCmdSlaveHosts := "SHOW SLAVE HOSTS"
	var myGTID string
	logger.Info("start new worker %s", bj.Settings.DatabaseMeta)
	t := time.NewTicker(30 * time.Second)

	cfg := r.BinlogSyncerConfig{
		ServerID: 6767,
		Flavor:   "mysql",
		Host:     bj.Settings.GetHostname(),
		Port:     uint16(bj.Settings.GetPort()),
		User:     bj.Settings.GetUser(),
		Password: bj.Settings.GetPassword(),
	}

	dns := fmt.Sprintf("%s:%s@tcp(%s:%d)/mysql",
		bj.Settings.GetUser(),
		bj.Settings.GetPassword(),
		bj.Settings.GetHostname(),
		bj.Settings.GetPort())
	conn, err := sql.Open("mysql", dns)
	if err != nil {
		logger.Crit("error connect %s: %s", bj.Settings, err)
		return
	}
	defer func() { _ = conn.Close() }()

	var attempt int
L:
	for range t.C {
		select {
		case <-bj.Settings.Cntx.Done():
			logger.Warn("cntx recieve %s\n", bj.Settings.DatabaseMeta)
			break L
		default:
		}
		serverIds := make(map[int]int)

		if err := conn.Ping(); err == nil {
			cntx, cancel := context.WithTimeout(context.Background(), 60*time.Second)
			row := conn.QueryRowContext(cntx, sqlCmdPurged)
			var purged VariablesStruct
			err := row.Scan(&purged.VariableName, &purged.Values)
			cancel()
			if err != nil {
				logger.Crit("error scan %s, attempt %d, %s", sqlCmdPurged, attempt, err)
				attempt++
				continue
			} else {
				myGTID = purged.Values
			}
			attempt = 0
			func() {
				cntx, cancel := context.WithTimeout(context.Background(), 60*time.Second)
				defer cancel()
				row2, err2 := conn.QueryContext(cntx, sqlCmdSlaveHosts)
				if err2 != nil {
					logger.Warn("error execute %s at %s:%d, attempt %d, %s", sqlCmdSlaveHosts, cfg.Host, cfg.Port, attempt, err2)
					attempt++
					return
				} else {
					defer func() { _ = row2.Close() }()
					var slave SlaveHostsStruct
					for row2.Next() {
						if err := row2.Scan(&slave.ServerID, &slave.Host, &slave.Port, &slave.MasterID, &slave.SlaveUUID); err == nil {
							serverIds[slave.ServerID] = 0
						} else {
							logger.Warn("error scan row at %s:%d: %s", cfg.Host, cfg.Port, err)
							time.Sleep(1 * time.Second)
						}
					}
				}
			}()
		} else {
			logger.Warn("error ping() database %s: %s", bj.Settings.DatabaseMeta, err)
			attempt++
		}
		if attempt > 10 {
			break
		}

		var usedIds []string
		for id := range serverIds {
			usedIds = append(usedIds, fmt.Sprint(id))
		}
		logger.Debug("busy server ids: %s", strings.Join(usedIds, ","))
		firstID := cfg.ServerID
		for {
			if cfg.ServerID-firstID > 100 {
				logger.Crit("used 100 attempts to change the server_id. Exit")
				break L
			}
			if _, ok := serverIds[int(cfg.ServerID)]; ok {
				logger.Warn("server_id=%d already busy, generate new server_id", cfg.ServerID)
				cfg.ServerID++
			} else {
				break
			}
		}

		func() {
			syncer := r.NewBinlogSyncer(cfg)
			defer syncer.Close()

			gtid, err := mysql.ParseGTIDSet(mysql.MySQLFlavor, myGTID)
			if err != nil {
				logger.Crit("error ParseGTIDSet, %s", err)
				return
			}

			streamer, err := syncer.StartSyncGTID(gtid)
			if err != nil {
				logger.Warn("error start SyncGTID %s, %s", bj.Settings.DatabaseMeta, err)
				return
			}

			cntx, cancel := context.WithTimeout(context.Background(), 30*time.Second)
			defer cancel()
			for range [10]int{} {
				binevent, err := streamer.GetEvent(cntx)
				if err != nil {
					logger.Crit("error streamer GetEvent, %s", err)
					continue
				}
				if binevent.Header == nil {
					logger.Crit("error NullPoint binevent.Header")
					continue
				}
				headerTime := (*binevent.Header).Timestamp
				if headerTime == 0 {
					continue
				} else {
					mtime := time.Unix(int64(headerTime), 0)

					bj.SendMetricToSolomon(mtime)

					event := bj.GenerateJugglerEvent(mtime)
					select {
					case jchan <- event:
						logger.Debug("send event %s to juggler chan", event)
					default:
					}
					return
				}
			}
		}()
	}
	logger.Info("stop job %s", bj.Settings.DatabaseMeta)
}

func (bj MySQLBinlogAgeJob) GenerateJugglerEvent(mtime time.Time) juggler.JugglerMessage {
	worker := bj.Settings
	instance := worker.Instance.Print()
	jclient := juggler.NewJugglerClient(
		fmt.Sprintf("mysql-binlog-age-%s", worker.Instance.Print()),
		juggler.WithJugglerAddress(jugglerAddress),
		juggler.WithEnviron(environ))
	//если binlog больше 30 дней - CRIT, не работает ротировнаие
	if time.Since(mtime) > 24*30*time.Hour {
		msg := fmt.Sprintf("[%s] the age of the binlog server %s is more than 30 days: %s", instance, worker.Hostname, mtime)
		logger.Info(msg)
		return jclient.NewJugglerRequest(2, msg)
	}
	// DIRECT-171008: не хочется хранить слишком много бинлогов, так как живут на одном разделе с данными
	if time.Since(mtime) > 10*30*time.Hour {
		msg := fmt.Sprintf("[%s] the age of the binlog server %s is more than 10 days: %s, set up less space for binlogs", instance, worker.Hostname, mtime)
		// пока непонятно, насколько сильно моргала бы проверка, только пишем в лог
		logger.Info(msg)
	}
	//если binlog больше 5, но меньше 7 дней - WARN
	if time.Since(mtime) < 24*7*time.Hour && time.Since(mtime) > 24*5*time.Hour {
		msg := fmt.Sprintf("[%s] the binlog server %s is less than 7 days old: %s", instance, worker.Hostname, mtime)
		logger.Info(msg)
		return jclient.NewJugglerRequest(1, msg)
	}
	//если бинлог меньше 5 дней - CRIT
	if time.Since(mtime) <= 24*5*time.Hour {
		msg := fmt.Sprintf("[%s] the binlog server %s is less than 5 days old: %s", instance, worker.Hostname, mtime)
		logger.Info(msg)
		return jclient.NewJugglerRequest(2, msg)
	}
	msg := fmt.Sprintf("[%s] binlog age for %s good: %s", instance, worker.Hostname, mtime)
	logger.Info(msg)
	return jclient.NewJugglerRequest(0, msg)
}

func (bj MySQLBinlogAgeJob) SendMetricToSolomon(mtime time.Time) {
	labels := make(map[string]string)
	labels["sensor"] = "mysql_binlog_age"
	labels["instance"] = bj.Settings.Instance.Print()
	labels["environment"] = environ
	binlogAge := time.Since(mtime).Seconds()
	ts := time.Now().Unix()
	metric := SolomonMetric{labels, ts, binlogAge}
	metrics := make([]SolomonMetric, 1)
	metrics[0] = metric
	var msg SolomonMessage
	msg.Metrics = metrics
	msgJSON, err := json.Marshal(msg)
	if err != nil {
		logger.Warn("could not marshal structure for sending to solomon to json: %v", metrics)
	} else {
		resp, err := bj.HTTPClient.Post(SolomonAgentURL, "application/json", bytes.NewReader(msgJSON))
		if err != nil {
			logger.Warn("HTTP POST request to %s failed: %s, data: %s", SolomonAgentURL, err, msgJSON)
		} else {
			var respBody []byte
			if resp.ContentLength < 0 {
				// Content-Length нет как минимум при успешном ответе, в этом случае он скорее всего пустой и неинтересен
				_, err := io.Copy(ioutil.Discard, resp.Body)
				if err != nil {
					// с Discard не должно случиться, но тесты требуют, чтобы ошибка обрабатывалась
					logger.Warn("unexpected error while discarding response body: %s", err)
				}
			} else {
				respBody = make([]byte, resp.ContentLength)
				_, err := resp.Body.Read(respBody)
				if err != nil {
					logger.Warn("error while reading HTTP response from solomon agent: %s", err)
				}
			}
			if resp.StatusCode > 299 || resp.StatusCode < 200 {
				// solomon-agent не должен отвечать редиректом, так что приравниваем такой ответ к ошибке
				logger.Warn("HTTP POST request to %s returned a non-2xx code: %s, response body: %s", SolomonAgentURL, resp.StatusCode, respBody)
			}
			err = resp.Body.Close()
			if err != nil {
				logger.Warn("error while closing HTTP response reader: %s", err)
			}
		}
	}
}
