package main

import (
	"bytes"
	"context"
	//"encoding/json"
	"flag"
	"fmt"
	"io/ioutil"
	"log"
	"os"
	"path/filepath"
	"reflect"
	"sort"
	"strconv"
	"strings"
	"sync"
	"time"

	_ "github.com/go-sql-driver/mysql"
	"github.com/jmoiron/sqlx"

	logger "a.yandex-team.ru/direct/infra/go-libs/pkg/logformat"
	//"github.com/peterbourgon/diskv"
	//"github.com/tidwall/buntdb"
)

const (
	DefaultMySQLHost   = "ppcdiscord.yandex.ru"
	DefaultMySQLPort   = 3306
	DefaultMySQLDB     = "ppc"
	DefaultMySQLUser   = "loaddb"
	DefaultMysqlPasswd = ""
	DefaultMaxConnect  = 15
)

type MySQLConfig struct {
	host   string
	port   int
	db     string
	user   string
	passwd string
}

func NewMySQLConfig(host string, port int, user, passwd string, db string) MySQLConfig {
	return MySQLConfig{
		host:   host,
		passwd: passwd,
		user:   user,
		port:   port,
		db:     db,
	}
}

func CreateConnect(c MySQLConfig) (*sqlx.DB, error) {
	addressRow := fmt.Sprintf("%s:%s@tcp(%s:%d)/%s", c.user, c.passwd, c.host, c.port, c.db)
	logger.Debug("connect to %s", addressRow)
	conn, err := sqlx.Open("mysql", addressRow)
	if err != nil {
		return nil, err
	}
	if err := conn.Ping(); err != nil {
		return nil, err
	}
	conn.SetMaxIdleConns(DefaultMaxConnect)
	return conn, nil
}

type MySQLTableStruct struct {
	DB      string        `db:"table_schema"`
	Table   string        `db:"table_name"`
	Keys    *TableKeys    `db:"-"`
	Columns *TableColumns `db:"-"`
}

type MySQLTableStucts []MySQLTableStruct

func (mts MySQLTableStucts) Find(table MySQLTableStruct) (MySQLTableStruct, error) {
	for _, t := range mts {
		if t.Table == table.Table && t.DB == table.DB {
			return t, nil
		}
	}
	return MySQLTableStruct{}, fmt.Errorf("not found")
}

type TableKey struct {
	Table        string `db:"Table"`
	NonUnique    int    `db:"Non_unique"`
	KeyName      string `db:"Key_name"`
	SeqInIndex   string `db:"Seq_in_index"`
	ColumnName   string `db:"Column_name"`
	Collation    string `db:"Collation"`
	SubPart      []byte `db:"Sub_part"`
	Packed       []byte `db:"Packed"`
	Null         string `db:"Null"`
	IndexType    string `db:"Index_type"`
	Comment      string `db:"Comment"`
	IndexComment string `db:"Index_comment"`
	Cardinality  string `db:"Cardinality"`
}

type TableKeys []TableKey

type TableColumn struct {
	Field   string `db:"Field"`
	Type    string `db:"Type"`
	Null    string `db:"Null"`
	Key     string `db:"Key"`
	Default []byte `db:"Default"`
	Extra   []byte `db:"Extra"`
}

type TableColumns []TableColumn

func (tks TableKeys) AllKeys() []string {
	var keys, seqs []string
	k1 := make(map[string]string)
	for _, key := range tks {
		k1[key.SeqInIndex] = key.ColumnName
		seqs = append(seqs, key.SeqInIndex)
	}
	sort.Strings(seqs)
	for _, seq := range seqs {
		keys = append(keys, k1[seq])
	}
	return keys
}

func (tcs TableColumns) AllKeys() []string {
	var seqs []string
	for _, key := range tcs {
		seqs = append(seqs, key.Field)
	}
	sort.Strings(seqs)
	return seqs
}

type PrimaryValues map[string][]byte

func ToString(pv interface{}) []string {
	var r []string

	logger.Debug("%v %v", pv, reflect.TypeOf(pv))
	switch reflect.TypeOf(pv).Kind() {
	case reflect.Array:
		items := reflect.ValueOf(pv)
		for i := 0; i < items.Len(); i++ {
			v := reflect.Indirect(items.Index(i))
			if v.Kind() == reflect.Uint8 {
				return append(r, strconv.Quote(fmt.Sprintf("%s", items)))
			}
			r = append(r, ToString(v.Interface())...)
		}
		if items.Len() == 0 {
			return append(r, "")
		}
	case reflect.Slice:
		items := reflect.ValueOf(pv)
		for i := 0; i < items.Len(); i++ {
			v := reflect.Indirect(items.Index(i))
			if v.Kind() == reflect.Uint8 {
				return append(r, strconv.Quote(fmt.Sprintf("%s", items)))
			}
			r = append(r, ToString(v.Interface())...)
		}
		if items.Len() == 0 {
			return append(r, "")
		}
	case reflect.Int64:
		r = append(r, strings.ReplaceAll(strconv.Quote(fmt.Sprintf("%d", pv)), "'", "\\'"))
	case reflect.Int:
		r = append(r, strings.ReplaceAll(strconv.Quote(fmt.Sprintf("%d", pv)), "'", "\\'"))
	case reflect.Uint8:
		r = append(r, strings.ReplaceAll(strconv.Quote(fmt.Sprintf("%d", pv)), "'", "\\'"))
	case reflect.String:
		r = append(r, strings.ReplaceAll(strconv.Quote(fmt.Sprintf("%s", pv)), "'", "\\'"))
	default:
		r = append(r, strings.ReplaceAll(strconv.Quote(fmt.Sprintf("%v", pv)), "'", "\\'"))
	}
	//logger.Debug("%v", r)
	return r
}

type MySQLTablePrimaryValues struct {
	MySQLTableStruct
	//PrimaryValues *buntdb.DB//*diskv.Diskv
}

type MySQLTableStructList []MySQLTableStruct

func (mtpl MySQLTableStructList) Find(table MySQLTableStruct) (MySQLTableStruct, error) {
	for _, m := range mtpl {
		if m.Table == table.Table && m.DB == table.DB {
			return m, nil
		}
	}
	return MySQLTableStruct{}, fmt.Errorf("not found table %s.%s", table.DB, table.Table)
}

func (pv PrimaryValues) HasKeyByHash(hash string) bool {
	if _, ok := pv[hash]; ok {
		return true
	}
	return false
}

func (pv PrimaryValues) Truncate() {
	for _, key := range pv.AllKeys() {
		delete(pv, key)
	}
}

func (pv PrimaryValues) AllKeys() []string {
	var keys []string
	for key := range pv {
		keys = append(keys, key)
	}
	return keys
}

//Список таблиц по определенной БД
func ListTablesByDatabase(conn *sqlx.DB, db string) (MySQLTableStucts, error) {
	var myTables MySQLTableStucts
	cntx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()
	query := "SELECT TABLE_SCHEMA AS table_schema, TABLE_NAME AS table_name FROM information_schema.TABLES WHERE TABLE_SCHEMA = ?"

	stmt, err := conn.PreparexContext(cntx, query)
	if err != nil {
		return myTables, err
	}

	rows, err := stmt.Queryx(db)
	if err != nil {
		return myTables, err
	}

	var errmsg []error
	for rows.Next() {
		var t1 MySQLTableStruct
		var keys TableKeys
		var columns TableColumns
		if err := rows.StructScan(&t1); err != nil {
			errmsg = append(errmsg, err)
			continue
		}
		t1.Keys = &keys
		t1.Columns = &columns
		myTables = append(myTables, t1)
	}
	if len(errmsg) > 0 {
		return myTables, fmt.Errorf("%s", errmsg)
	}

	return myTables, nil
}

//Список ключей по определенной таблице.Обновляет структуру MySQLTableStruct.Table.Keys
func SelectKeysForTable(conn *sqlx.DB, table MySQLTableStruct) (MySQLTableStruct, error) {
	cntx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()
	query := fmt.Sprintf("SHOW KEYS FROM `%s`.`%s` WHERE Key_name = 'PRIMARY'", table.DB, table.Table)
	logger.Debug("run query %s", query)
	stmt, err := conn.PreparexContext(cntx, query)
	if err != nil {
		return table, err
	}
	defer func() { _ = stmt.Close() }()

	rows, err := stmt.Queryx()
	if err != nil {
		return table, err
	}

	var errmsg []error
	for rows.Next() {
		var t2 TableKey
		if err := rows.StructScan(&t2); err != nil {
			errmsg = append(errmsg, err)
			continue
		}
		*table.Keys = append(*table.Keys, t2)
	}
	if len(errmsg) > 0 {
		return table, fmt.Errorf("%s", errmsg)
	}

	return table, nil
}

//Список ключей по определенной таблице.Обновляет структуру MySQLTableStruct.Table.Keys
func SelectColumnsForTable(conn *sqlx.DB, table MySQLTableStruct) (MySQLTableStruct, error) {
	cntx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()
	query := fmt.Sprintf("SHOW COLUMNS FROM `%s`.`%s`", table.DB, table.Table)
	logger.Debug("run query %s", query)
	stmt, err := conn.PreparexContext(cntx, query)
	if err != nil {
		return table, err
	}
	defer func() { _ = stmt.Close() }()

	rows, err := stmt.Queryx()
	if err != nil {
		return table, err
	}

	var errmsg []error
	for rows.Next() {
		var t3 TableColumn
		if err := rows.StructScan(&t3); err != nil {
			errmsg = append(errmsg, err)
			continue
		}
		*table.Columns = append(*table.Columns, t3)
	}
	if len(errmsg) > 0 {
		return table, fmt.Errorf("%s", errmsg)
	}

	return table, nil
}

//type MySQLTableStatus struct {
//	MySQLTableStruct
//	Result HashRows
//}

type MySQLTablesStatus map[MySQLTableStruct]HashRows

func Zip(vs ...[]interface{}) [][]interface{} {
	result := make([][]interface{}, len(vs[0]))
	for _, b := range vs {
		for i, v := range b {
			result[i] = append(result[i], v)
		}
	}
	return result
}

func GenerateSelectPrimaryKeysRequest(table MySQLTableStruct, queuePK [][]interface{}) string {
	//группируем по primary key: ClientID: [12, 33, 44]
	q := make(map[string][]string)
	for i, key := range table.Keys.AllKeys() {
		for _, j := range queuePK {
			if v := ToString(j); i < len(v) {
				if len(v[i]) > 0 {
					q[key] = append(q[key], v[i])
				}
			} else {
				logger.Crit("error index %d in %v %v", i, v, j)
				continue
			}
		}
	}

	//Составляем запрос WHERE IN для набора ключей
	var p []string
	for k, v := range q {
		p = append(p, fmt.Sprintf("`%s` IN (%s)", k, strings.Join(v, ",")))
	}
	query := fmt.Sprintf("SELECT `%s` FROM %s.%s WHERE %s", strings.Join(table.Keys.AllKeys(), "`,`"),
		table.DB, table.Table, strings.Join(p, " AND "))
	logger.Debug("%s", query)
	return query
}

type HashRows map[string][]interface{}

//принимает список ключей, которые надо проверить в первой базе и возвращает список найденных во второй
func RunReplicaPrimaryKeysByTable(conn2 *sqlx.DB, table MySQLTableStruct, queuePK [][]interface{}) (HashRows, error) {
	result := make(HashRows)
	//result[table] = [][]interface{}{}

	if len(queuePK) == 0 {
		return nil, fmt.Errorf("empty primary keys for %s", table.Table)
	}

	cntx, cancel := context.WithTimeout(context.Background(), 30*time.Minute)
	defer cancel()

	query := GenerateSelectPrimaryKeysRequest(table, queuePK)

	if _, err := conn2.Exec("SET net_read_timeout=3600"); err != nil {
		logger.Warn("error set net_read_timeout: %s", err)
	}
	if _, err := conn2.Exec("SET net_write_timeout=3600"); err != nil {
		logger.Warn("error set net_write_timeout: %s", err)
	}
	stmt1, err := conn2.PreparexContext(cntx, query)
	if err != nil {
		logger.Crit("error build stmt %s, error %s", query, err)
		return nil, err
	}
	defer func() { _ = stmt1.Close() }()
	rows1, err := stmt1.Queryx()
	if err != nil {
		logger.Crit("error run queryx %s, error %s", query, err)
		return nil, err
	}
	var errmsg []error
	defer func() { _ = rows1.Close() }()
	for rows1.Next() {
		var t4 []interface{}
		if t4, err = rows1.SliceScan(); err != nil {
			errmsg = append(errmsg, err)
			continue
		}
		key := strings.Join(ToString(t4), "_")
		result[key] = t4
	}
	if len(errmsg) > 0 {
		return result, fmt.Errorf("error select rows %s", errmsg)
	}
	return result, nil
}

func HasKey(key string, keys []string) bool {
	for _, k := range keys {
		if k == key {
			return true
		}
	}
	return false
}

func HasHashKey(key string, keys HashRows) bool {
	if _, ok := keys[key]; ok {
		return true
	}
	return false
}

//принимает коннект к мастер и слейв базе, а так же таблицы, которую надо проверить
func RunDifferentPrimaryKeysByTable(conn1, conn2 *sqlx.DB, table MySQLTableStruct) (HashRows, error) {
	result := make(HashRows)
	cntx, cancel := context.WithTimeout(context.Background(), 30*time.Minute)
	defer cancel()

	query := fmt.Sprintf("SELECT `%s` FROM %s.%s", strings.Join(table.Keys.AllKeys(), "`,`"), table.DB, table.Table)
	logger.Debug("run query %s", query)

	if _, err := conn1.Exec("SET net_read_timeout=3600"); err != nil {
		logger.Warn("error conn set net_read_timeout: %s", err)
	}
	if _, err := conn1.Exec("SET net_write_timeout=3600"); err != nil {
		logger.Warn("error conn1 set net_write_timeout: %s", err)
	}

	if _, err := conn1.Exec("SET max_allowed_packet=104217728"); err != nil {
		logger.Debug("error conn set max_allowed_packet: %s", err)
	}
	if _, err := conn1.Exec("SET net_write_timeout=104217728"); err != nil {
		logger.Debug("error conn1 set max_allowed_packet: %s", err)
	}

	stmt1, err := conn1.PreparexContext(cntx, query)
	if err != nil {
		return nil, err
	}
	defer func() { _ = stmt1.Close() }()

	rows1, err := stmt1.Queryx()
	if err != nil {
		return nil, err
	}
	defer func() { _ = rows1.Close() }()

	var errmsg []error
	queuePK := make(HashRows)
	var size int
	//var queuePK [][]interface{}
	//собираем знчения primary keys в пачки по 100 значений

	for rows1.Next() {
		var t1 []interface{}
		if t1, err = rows1.SliceScan(); err != nil {
			errmsg = append(errmsg, err)
			continue
		}
		key1 := strings.Join(ToString(t1), "_")
		queuePK[key1] = t1
		size += len(key1)
		//queuePK = append(queuePK, t1)

		if len(queuePK) > 100 || size > 10*1024*1024 {
			var replicaPK HashRows
			for i := 0; i < 3; i++ {
				//fmt.Printf("LEN %d\n", len(queuePK))
				replicaPK, err = RunReplicaPrimaryKeysByTable(conn2, table, queuePK.GetValues())
				if err != nil {
					logger.Crit("%s", err)
					time.Sleep(10 * time.Second)
					continue
				}
				break
			}
			for pk, out := range queuePK {
				if HasHashKey(pk, replicaPK) {
					continue
				} else {
					result[pk] = out
					logger.Debug("%s not found key %s (%s) / %v", table.Table, pk, table.Keys.AllKeys(), out)
				}
			}
			queuePK = make(HashRows)
			size = 0
			//queuePK = [][]interface{}{}
		}
	}
	if len(queuePK) > 0 {
		var replicaPK HashRows
		for i := 0; i < 3; i++ {
			replicaPK, err = RunReplicaPrimaryKeysByTable(conn2, table, queuePK.GetValues())
			if err != nil {
				logger.Crit("%s", err)
				time.Sleep(10 * time.Second)
				continue
			}
			break
		}
		for pk, out := range queuePK {
			if HasHashKey(pk, replicaPK) {
				continue
			} else {
				result[pk] = out
				logger.Debug("%s not found key %s (%s) /%v", table.Table, pk, table.Keys.AllKeys(), out)
			}
		}
	}

	if len(errmsg) > 0 {
		return result, fmt.Errorf("[RunDifferentPrimaryKeysByTable] error rows %s", errmsg)
	}
	return result, nil
}

func (to MySQLTablesStatus) AllTables() []string {
	var result []string
	for tableName := range to {
		result = append(result, tableName.Table)
	}
	return result
}

//[]interface{} - list output primary keys(queuePK)
//type TableOutputs map[MySQLTableStruct][][]interface{} //queuePK

//func (mts MySQLTablesStatus) AllOutputByTables() (TableOutputs) {
//	result := make(TableOutputs)
//	for _, m := range mts {
//		if _, ok := result[m.MySQLTableStruct]; !ok {
//			result[m.MySQLTableStruct] = [][]interface{}{}
//		}
//		result[m.MySQLTableStruct] = append(result[m.MySQLTableStruct], m.Result)
//	}
//	return result
//}

func (to MySQLTablesStatus) PrintByButches() {
	for key, val := range to {
		max := len(val)
		pkeys := val.GetValues()
		for i := 0; i < max/100; i++ {
			logger.Crit("[%s] %v: %s", key.Table, key.Keys.AllKeys(), ToString(pkeys[i*100:(i+1)*100]))
			logger.Crit("[%s] %s", key.Table, GenerateSelectPrimaryKeysRequest(key, pkeys[i*100:(i+1)*100]))
		}
		if i := (max / 100) * 100; max > i {
			logger.Crit("[%s] %v: %s", key.Table, key.Keys.AllKeys(), ToString(pkeys[i:max]))
			logger.Crit("[%s] %s", key.Table, GenerateSelectPrimaryKeysRequest(key, pkeys[i:max]))
		}
	}
}

func (hs HashRows) GetHashes() []string {
	var result []string
	for key := range hs {
		result = append(result, key)
	}
	return result
}

func (hs HashRows) GetValues() [][]interface{} {
	var result [][]interface{}
	for _, val := range hs {
		result = append(result, val)
	}
	return result
}

/*
func ChecksumByPrimarySegment(conn *sqlx.DB, pvl MySQLTablePrimaryValues) (string, error) {
	cntx, cancel := context.WithTimeout(context.Background(), 30*time.Minute)
	defer cancel()

	var primarys []string
	for _, t := range pvl.PrimaryValues {
		values := ToString(t)
		var i []string
		for num, v := range pvl.MySQLTableStruct.Keys.AllKeys() {
			q := fmt.Sprintf("`%s`=%s", v, values[num])
			i = append(i, q)
		}
		p1 := fmt.Sprintf("(%s)", strings.Join(i, " AND "))
		primarys = append(primarys, p1)
	}
	p2 := fmt.Sprintf("%s", strings.Join(primarys, " OR "))
	fields := fmt.Sprintf("`%s`", strings.Join(pvl.Columns.AllKeys(), "`,`"))
	query := fmt.Sprintf("SELECT COALESCE(LOWER(CONCAT(LPAD(CONV(BIT_XOR(CAST(CONV(SUBSTRING"+
		"(@crc, 1, 16), 16, 10) AS UNSIGNED)), 10, 16), 16, '0'), LPAD(CONV(BIT_XOR(CAST(CONV(SUBSTRING(@crc := md5(CONCAT_WS('#', %s))"+
		", 17, 16), 16, 10) AS UNSIGNED)), 10, 16), 16, '0'))), 0) AS crc FROM `%s`.`%s` WHERE %s", fields, pvl.DB, pvl.Table, p2)
	logger.Debug("run query %s", query)
	stmt, err := conn.PreparexContext(cntx, query)
	if err != nil {
		return "", err
	}
	defer func() { _ = stmt.Close() }()

	row1 := stmt.QueryRowx()
	if err != nil {
		return "", err
	}

	var t4 TableCRC
	err = row1.StructScan(&t4)
	if err != nil {
		return "", err
	}
	//fmt.Printf("%s, %s\n", ToString(t3), pvl.PrimaryValues)
	//logger.Debug("%s %+v", key3, t3)
	//}
	return t4.CRC, nil
}*/

/*func Check(){
	prefix := fmt.Sprintf("[%s.%s]", val1.DB, val1.Table)
		logger.Info("%s START", prefix)
		val2, err := t2.Find(val1)
		if err != nil {
			logger.Crit("%s %s", prefix)
		}
		for _, key1 := range val1.PrimaryValues {
			if ok := val2.HasKey(key1); !ok {
				logger.Crit("%s not found key %s %s", prefix, ToString(key1), val1.Keys.AllKeys())
			} else {
				logger.Debug("%s founded %s %s", prefix, ToString(key1), val1.Keys.AllKeys())
			}
		}
		logger.Info("%s FINISH", prefix)
}*/

func main() {
	srcHost := flag.String("source-host", DefaultMySQLHost, "source mysql host")
	dstHost := flag.String("destination-host", "", "destination mysql host")
	srcPort := flag.Int("source-port", DefaultMySQLPort, "source mysql port")
	dstPort := flag.Int("destination-port", DefaultMySQLPort, "destination mysql port")
	debug := flag.Bool("debug", false, "debug mode")
	maxconn := flag.Int("maxconn", DefaultMaxConnect, "максимальное количество конектов и потоков на select")
	allowedTables := flag.String("allowed-tables", "", "какие таблицы проверять(по умолчанию все)")
	passwordFile := flag.String("password-file", "", "токен с паролем для подключения к БД")
	password := flag.String("password", DefaultMysqlPasswd, "пароль для подключения к БД")
	user := flag.String("user", DefaultMySQLUser, "пользователь для подключения к БД")
	db := flag.String("db", DefaultMySQLDB, "база для подключения к БД")
	flag.Parse()

	if *debug {
		logger.NewLoggerFormat(
			logger.WithLogLevel(logger.DebugLvl),
			logger.WithLogger(log.New(os.Stdout, "", 0)))
	} else {
		logger.NewLoggerFormat(
			logger.WithLogLevel(logger.WarningLvl),
			logger.WithLogger(log.New(os.Stdout, "", 0)))
	}

	if len(*passwordFile) > 0 && len(*password) > 0 {
		logger.Crit("cannot use together --password-file and --password")
		os.Exit(1)
	}

	if len(*passwordFile) > 0 {
		if !filepath.IsAbs(*passwordFile) {
			prefix := filepath.Dir(*passwordFile)
			if strings.HasPrefix(prefix, "~") {
				*passwordFile = filepath.Join(os.Getenv("HOME"), filepath.Base(*passwordFile))
			}
			*passwordFile = filepath.Join(os.Getenv("PWD"), filepath.Base(*passwordFile))
		}
		if data, err := ioutil.ReadFile(*passwordFile); err != nil {
			logger.Crit("error read password file %s: %s", *passwordFile, err)
		} else {
			data = bytes.Trim(data, "\n")
			*password = string(data)
		}
	}

	mycnf1 := NewMySQLConfig(*srcHost, *srcPort, *user, *password, *db)
	mycnf2 := NewMySQLConfig(*dstHost, *dstPort, *user, *password, *db)

	r1 := make(map[MySQLConfig]MySQLTableStructList)
	for _, mycnf := range []MySQLConfig{mycnf1, mycnf2} {
		conn, err := CreateConnect(mycnf)
		defer func() { _ = conn.Close() }()
		if err != nil {
			logger.Crit("error connect %+v: %s", mycnf, err)
			os.Exit(1)
		}

		tables, err := ListTablesByDatabase(conn, *db)
		logger.Debug("%v %+v %s", mycnf, tables, err)

		for _, table := range tables {
			_, err := SelectKeysForTable(conn, table)
			if err != nil {
				logger.Crit("%s", err)
			}
			_, err = SelectColumnsForTable(conn, table)
			if err != nil {
				logger.Crit("%s", err)
			}
			r1[mycnf] = append(r1[mycnf], table)
			logger.Debug("%s.%s: %+v len: %d", table.DB, table.Table, *table.Keys, len(*table.Keys))
		}
	}

	conn1, err := CreateConnect(mycnf1)
	if err != nil {
		logger.Crit("error connect %v: %s", mycnf1, err)
		os.Exit(0)
	}
	defer func() { _ = conn1.Close() }()

	conn2, err := CreateConnect(mycnf2)
	if err != nil {
		logger.Crit("error connect %v: %s", mycnf2, err)
		os.Exit(0)
	}
	defer func() { _ = conn2.Close() }()

	var wg sync.WaitGroup

	maxWorkers := make(chan int, *maxconn)
	for _, val := range r1[mycnf1] {
		if len(*allowedTables) > 0 {
			if tableNames := strings.Split(*allowedTables, ","); !HasKey(val.Table, tableNames) {
				//example val.Table != "account_score"
				continue
			}
		}
		val1 := val
		if err != nil {
			logger.Crit("[%s.%s] %s", val1.DB, val1.Table, err)
			continue
		}
		wg.Add(1)
		go Worker(&wg, val1, conn1, conn2, maxWorkers)
	}
	wg.Wait()
	close(maxWorkers)
}

func Worker(wg *sync.WaitGroup, table MySQLTableStruct, conn1, conn2 *sqlx.DB, m chan int) {
	defer (*wg).Done()
	defer func() { <-m }()
	m <- 1

	prefix := fmt.Sprintf("[%s.%s]", table.DB, table.Table)

	logger.Info("%s worker start", prefix)

	pkeys, err := RunDifferentPrimaryKeysByTable(conn1, conn2, table)
	if err != nil {
		logger.Crit("%s source connect error: %s", prefix, err)
	}
	for _, i := range pkeys {
		logger.Info("table %s, result %s", table.Table, i)
	}

	ticker := time.NewTicker(1 * time.Minute)
	var count int

	for range ticker.C {
		count++
		//oldPkeys := results1[table]
		if count > 5 || len(pkeys) == 0 {
			break
		}

		batch := pkeys.GetValues()
		for i := 0; i > len(batch)/100; i++ {
			rkeys, err := RunReplicaPrimaryKeysByTable(conn2, table, batch[i:100*(i+1)])
			if err != nil {
				logger.Warn("error replica check for %s: %s", table.Table, err)
				continue
			}
			pkeys, err = RunReplicaPrimaryKeysByTable(conn1, table, batch[i:100*(i+1)])
			if err != nil {
				logger.Warn("error replica check for %s: %s", table.Table, err)
				continue
			}
			nkeys := make(HashRows)
			for key2, val2 := range pkeys {
				if ok := HasHashKey(key2, rkeys); ok {
					logger.Warn("founded primary key %s as done", ToString(val2))
				} else {
					nkeys[key2] = val2
				}
			}
			pkeys = nkeys
		}
	}

	result := make(MySQLTablesStatus)
	result[table] = pkeys
	result.PrintByButches()

	/*
			if len(batches) < 10 {
				batches[hash] = key1
				continue
			} else {
				crc1, err1 := ChecksumByPrimarySegment(conn1, MySQLTablePrimaryValues{
					pks1.MySQLTableStruct,
					batches,
				})
				if err1 != nil {
					logger.Warn("%s error CRC connect1 %s: %s", prefix, batches, err1)
					continue
				}
				crc2, err2 := ChecksumByPrimarySegment(conn2, MySQLTablePrimaryValues{
					pks1.MySQLTableStruct,
					batches,
				})

				if err2 != nil {
					logger.Warn("%s error CRC connect2 %s: %s", prefix, batches, err2)
					continue
				}
				if crc1 != crc2 {
					logger.Crit("%s different crc1 %s != crc2 %s for batches %+v", prefix, crc1, crc2, batches.AllKeys())
				}
				logger.Debug("CRC1 %s, CRC2 %s, error %s", crc1, crc2, err)
				batches.Truncate()
			}
		}

		if len(batches) > 0 {
			crc1, err1 := ChecksumByPrimarySegment(conn1, MySQLTablePrimaryValues{
				pks1.MySQLTableStruct,
				batches,
			})
			if err1 != nil {
				logger.Warn("%s error CRC connect1 %s: %s", prefix, batches, err1)
				return
			}
			crc2, err2 := ChecksumByPrimarySegment(conn2, MySQLTablePrimaryValues{
				pks1.MySQLTableStruct,
				batches,
			})
			if err2 != nil {
				logger.Warn("%s error CRC connect2 %s: %s", prefix, batches, err2)
				return
			}
			if crc1 != crc2 {
				logger.Crit("%s different crc1 %s != crc2 %s for batches %+v", prefix, crc1, crc2, batches.AllKeys())
			}
			logger.Debug("%s CRC1 %s, CRC2 %s, error %s", prefix, crc1, crc2, err)
			batches.Truncate()
		}*/
	logger.Info("%s worker finis", prefix)
}
