package main

import (
	"fmt"
	"strings"
	"sync"
	"time"

	"a.yandex-team.ru/direct/infra/go-libs/pkg/juggler"
	logger "a.yandex-team.ru/direct/infra/go-libs/pkg/logformat"
	"a.yandex-team.ru/direct/infra/go-libs/pkg/mysqls"
	"a.yandex-team.ru/direct/infra/go-libs/pkg/zklib"
)

const (
	MonitoringJugglerDuration = 5 * time.Second
	MaxReadDurationZookeeper  = 3 * ReadDurationZookeeper
	MaxWriteDurationZookeeper = 3 * ReadDurationZookeeper
	JugglerService            = "direct-mysql-master-monitor"
)

func RunJugglerMonitorThread(zkconn ZkConnInterface, zknode *zklib.ZkNode, stopNode *zklib.ZkNode,
	mygroup *mysqls.MysqlMonitoringGroup, jugglerAddress, environ string, wg *sync.WaitGroup) {
	defer wg.Done()
	jclient := juggler.NewJugglerClient(
		JugglerService,
		juggler.WithEnviron(environ),
		juggler.WithJugglerAddress(jugglerAddress))
	t := time.NewTicker(MonitoringJugglerDuration)
	startTime := time.Now()

	montime := time.Now()
	for range t.C {
		var errmsg []string
		if ok, err := zkconn.ExistNode(stopNode.GetPath()); ok && err == nil {
			//если найден флаг остановки, то записываем сообщение в juggler
			msg := fmt.Sprintf("found stop flag %s, skip write zknode %s", stopNode.GetPath(), zknode.GetPath())
			errmsg = append(errmsg, msg)
		} else if err != nil {
			//если были ошибки в работе с zookeeper - просто логируем
			logger.Warn("error check exist node %s: %s. Skip save %s", stopNode.GetPath(), err, zknode.GetPath())
		} else if !ok {
			//если stop флага нет, то проверяем лаг записи
			if ok, err := zknode.CheckZookeeperNode(MaxReadDurationZookeeper, MaxWriteDurationZookeeper); !ok {
				switch err.(type) {
				//для исключения возможности флапа, когда нода не успела обновится в zookeeper'e
				case zklib.WriteLagError:
					if time.Since(montime) > MaxReadDurationZookeeper {
						errmsg = append(errmsg, fmt.Sprint(err))
					}
				default:
					montime = time.Now()
					errmsg = append(errmsg, fmt.Sprint(err))
				}
			}
		}
		if ok, err := mygroup.CheckMasterActiveGroups(); !ok && err != nil {
			errmsg = append(errmsg, fmt.Sprint(err))
		}
		var jmessage juggler.JugglerMessage
		if len(errmsg) == 0 {
			jmessage = jclient.NewJugglerRequest(0, "OK")
		} else if time.Since(startTime) < 1*time.Minute {
			jmessage = jclient.NewJugglerRequest(0, "starting program, wait data")
		} else {
			jmessage = jclient.NewJugglerRequest(2, strings.Join(errmsg, ","))
		}
		if err := jmessage.Send(); err != nil {
			logger.Warn("error send juggler event, %s", err)
		}
	}
}
