package main

import (
	"bytes"
	"crypto/md5"
	"encoding/json"
	"flag"
	"fmt"
	"io/ioutil"
	"log"
	"log/syslog"
	"net/http"
	"os"
	"strings"
	"sync"
	"time"

	"a.yandex-team.ru/direct/infra/go-libs/pkg/dbconfig"
	logger "a.yandex-team.ru/direct/infra/go-libs/pkg/logformat"
	"a.yandex-team.ru/direct/infra/go-libs/pkg/mysqls"
	"a.yandex-team.ru/direct/infra/go-libs/pkg/zklib"
	"a.yandex-team.ru/infra/rsm/nvgpumanager/pkg/juggler"
	"a.yandex-team.ru/library/go/maxprocs"
)

const (
	DefaultZookeeperServers      = "ppcdiscord.yandex.ru:2181"
	DefaultZookeeperPathDBConfig = "/test/db-config"
	DefaultZookeperPassFile      = ""
	DefaultZookeperTokenFile     = ""
	DefaultZookeeperUser         = ""
	DefaultMySQLInstances        = InstnaceNames("ppcdata1")
	APIAddress                   = "localhost:80"

	DefaultSyslogSocket = "/dev/log"
	DefaultEnviron      = "dev"

	StopWriteZookeperNode = "/direct_mmm_%s_stopflag"
)

var (
	zkPath, zkServers, zkTokenFile, zkPassFile, zkUser string
	debugMode                                          bool

	err           error
	wg            sync.WaitGroup
	zknode        zklib.ZkNode
	instanceNames InstnaceNames

	mygroup mysqls.MysqlMonitoringGroup

	jugglerAddress string

	syslogEnable bool
	syslogSocket string
	environ      string

	logLevel = logger.InfoLvl
)

func init() {
	instanceNames = DefaultMySQLInstances
	mygroup = mysqls.NewMysqlMonitoringGroup()
}

type InstnaceNames string

func (in InstnaceNames) ToList() []string {
	return strings.Split(string(in), ",")
}

func (in InstnaceNames) String() string {
	return string(in)
}

func (in *InstnaceNames) Set(name string) error {
	*in = InstnaceNames(name)
	return nil
}

//http сервер для получения статуса работы приложения
func RunHTTPServerThread(wg *sync.WaitGroup) {
	defer wg.Done()
	http.HandleFunc("/masters", func(w http.ResponseWriter, r *http.Request) {
		masters := make(map[string][]string)
		for _, name := range mygroup.GetActiveInstanceNames() {
			fqdns, _ := mygroup.GetMastersByInstanceName(name)
			masters[name] = fqdns
		}
		data, err := json.Marshal(masters)
		if err != nil {
			http.Error(w, fmt.Sprint(err), http.StatusInternalServerError)
			return
		}

		_, _ = fmt.Fprintf(w, "%s", data)
	})

	http.HandleFunc("/alive", func(w http.ResponseWriter, r *http.Request) {
		var errmsg []string
		//проверяем, что есть хотябы один мастер из списка инстансов(работает подключение к MySQL)
		if ok, err := mygroup.CheckAnyAliveMaster(); !ok || err != nil {
			errmsg = append(errmsg, fmt.Sprint(err))
		}
		//проверяем, что работает подключение к zookeeper
		if ok, err := zknode.CheckZookeeperNode(MaxReadDurationZookeeper, MaxWriteDurationZookeeper); !ok || err != nil {
			errmsg = append(errmsg, fmt.Sprint(err))
		}
		if len(errmsg) != 0 {
			http.Error(w, strings.Join(errmsg, ","), http.StatusInternalServerError)
			return
		}
		_, _ = fmt.Fprint(w, "OK")
	})

	http.HandleFunc("/status", func(w http.ResponseWriter, r *http.Request) {
		data, err := json.Marshal(mygroup)
		if err != nil {
			http.Error(w, fmt.Sprint(err), http.StatusInternalServerError)
			return
		}

		_, _ = fmt.Fprintf(w, "%s", data)
	})

	t := time.NewTicker(5 * time.Second)
	for range t.C {
		err := http.ListenAndServe(APIAddress, nil)
		if err != nil {
			logger.Crit("%s", err)
		}
	}
}

func LoggingCurrentState(mygroup *mysqls.MysqlMonitoringGroup, instanceNames []string, wg *sync.WaitGroup) {
	defer wg.Done()
	t := time.NewTicker(5 * time.Minute)
	for range t.C {
		for _, name := range instanceNames {
			masters, err := mygroup.GetMastersByInstanceName(name)
			if err != nil {
				logger.Crit("%s", err)
				continue
			}
			logger.Info("current master for %s is %s", name, masters[0])
		}
	}
}

//обновляет в локальной копии db-config текущие мастера на основе данных по активным мастерам из MysqlMonitoringGroup
func RunUpdateMySQLMasterThread(zknode *zklib.ZkNode, mygroup *mysqls.MysqlMonitoringGroup, instanceNames []string, wg *sync.WaitGroup) {
	defer wg.Done()
	t := time.NewTicker(1 * time.Second)
	for range t.C {
		var updater bool
		version := *zknode.LastReadVersion //вычитываем версию, чтобы быть уверенными, что данные не устареют
		zkdata := *zknode.Data
		zkconf := dbconfig.NewDBConfig() //очищаем старый dbconfig
		if err := zkconf.LoadDBConfig(zkdata); err != nil {
			logger.Warn("error dbconfig, zkdata %s, error %s", *zknode.Data, err)
			continue
		}
		for _, name := range instanceNames {
			masters, err := mygroup.GetMastersByInstanceName(name)
			if err != nil {
				logger.Crit("%s", err)
				continue
			}
			master := masters[0] //вызов безопасен
			zkmaster, _ := zkconf.Master(name)
			if !strings.EqualFold(zkmaster, master) {
				logger.Warn("change master %s -> %s", zkmaster, master)
				zkconf.SetParamForInstance(name, "host", master)
				updater = true
			}
		}

		if updater {
			oldsum := md5.Sum(zkdata)
			data, err := json.MarshalIndent(zkconf.GetDBConfig(), "", "\t")
			if err != nil {
				logger.Crit("error pretty marshal json data %+v, error %s", data, err)
				continue
			}
			newsum := md5.Sum(data)

			if strings.EqualFold(fmt.Sprintf("%x", oldsum), fmt.Sprintf("%x", newsum)) {
				logger.Warn("not found changes for local db-config. Skip save node")
				continue
			}

			//Передаем версию, которую вычитали. Если при сохранении версии будут отличаться,
			// то данные не сохранятся в zookeeper.
			if err := zknode.UpdateNode(data, version); err != nil {
				logger.Crit("%s", err)
			}
		}
	}

}

//проверка наличия объекта в списке
func Has(val string, vals []string) bool {
	for _, v := range vals {
		if strings.EqualFold(v, val) {
			return true
		}
	}
	return false
}

//читает из db-config по указанным группам(instanceNames) имена серверов из параметра cluster_hosts
//1. если хост отсутствует в мониторинге - добавляет
//2. если в мониторинге присутствует хост, которого нет в конфиге - удаляет и останавливает его моняторящий поток
func RunUpdateMySQLHostsThread(zknode *zklib.ZkNode, instanceNames []string, wg *sync.WaitGroup) {
	defer wg.Done()
	t := time.NewTicker(5 * time.Second)
	for range t.C {
		if zknode == nil {
			logger.Warn("empty zknode. Waiting")
			continue
		}
		dbcnf := dbconfig.NewDBConfig()
		err := dbcnf.LoadDBConfig(*zknode.Data)
		if err != nil {
			logger.Warn("error dbconfig, error %s", err)
			continue
		}

		for _, name := range instanceNames {
			//fmt.Printf("%+v\n", dbconf)
			clusterHosts, err := dbcnf.ClusterHosts(name)
			if err != nil {
				logger.Warn("%s", err)
				continue
			}
			var configMySQLHosts []string
			for _, host := range clusterHosts {
				configMySQLHosts = append(configMySQLHosts, host)
				if !mygroup.Has(name, host) {
					logger.Info("add host %s to group %s", host, name)
					if stat, err := dbcnf.CreateMysqlInstanceStatus(host, name); err != nil {
						logger.Crit("%s", err)
						continue
					} else {
						mygroup.Append(stat)
					}
				}
			}
			if currentMySQLHosts, err := mygroup.GetHostsByInstanceName(name); err != nil {
				logger.Crit("%s", err)
			} else {
				for _, host := range currentMySQLHosts {
					if !Has(host, configMySQLHosts) {
						i := mygroup.GetGroupByInstanceName(name)
						for _, h := range *i.GetGroupByHost(host).InstancesStatus {
							logger.Warn("stop monitoring host %s in group %s", host, name)
							h.Cancel()
							logger.Warn("delete host %s in group %s", host, name)
							mygroup.Delete(host, name)
						}
					}
				}
			}
		}
	}
}

func main() {
	flag.StringVar(&zkPath, "zkpath", DefaultZookeeperPathDBConfig, "db-config zookeeper path")
	flag.StringVar(&zkServers, "zkservers", DefaultZookeeperServers, "zookeeper servers")
	flag.StringVar(&zkTokenFile, "zktoken-file", DefaultZookeperTokenFile, "zookeeper token")
	flag.StringVar(&zkPassFile, "zkpass-file", DefaultZookeperPassFile, "zookeeper password file")
	flag.StringVar(&zkUser, "zkuser", DefaultZookeeperUser, "zookeeper user")
	flag.StringVar(&environ, "env", DefaultEnviron, "enviroment")
	flag.StringVar(&jugglerAddress, "juggler-address", juggler.DefaultLocalURL, "juggler client api address")
	flag.BoolVar(&debugMode, "debug", false, "enable debug mode")
	flag.Var(&instanceNames, "instances", "mysql instances")
	flag.StringVar(&syslogSocket, "syslog-socket", DefaultSyslogSocket, "syslog socket path")
	flag.BoolVar(&syslogEnable, "syslog", false, "enable syslog logger")
	flag.Parse()

	if debugMode {
		logLevel = logger.DebugLvl
	}

	maxprocs.AdjustYP()

	zkStopPath := fmt.Sprintf(StopWriteZookeperNode, environ)
	zkStopNode := zklib.NewZkNode(zkStopPath)

	logDirectFormat := logger.DirectMessageFormat("direct.back", "mysql-master-monitor")

	if syslogEnable {
		//для логировнаия используем syslog. Какой уровень логирования пиштеся, решается в обертке  logformat
		syslogger, err := syslog.Dial("unixgram", syslogSocket, syslog.LOG_DEBUG|syslog.LOG_DAEMON, os.Args[0])
		if err != nil {
			log.Fatal(err) //вывод в stderr и exit 1
		}
		logger.NewLoggerFormat(
			logger.WithLogger(syslogger),
			logger.WithLogLevel(logLevel),
			logger.WithLogFormat(logDirectFormat),
		)
	} else {
		newlogger := log.New(os.Stdout, "", 0)
		logger.NewLoggerFormat(
			logger.WithLogger(newlogger),
			logger.WithLogLevel(logLevel),
			logger.WithLogFormat(logDirectFormat),
		)
	}

	zkServers := strings.Split(zkServers, ",")
	zkAddress := zklib.NewZkAddress(zkUser, "", zkServers...)

	//вычитываем токен для zookeeper, если он присутствует в системе
	if len(zkTokenFile) > 0 && len(zkPassFile) > 0 {
		logger.Crit("error arguments: dont use '-zktoken-file' with '-zkpass-file'")
		log.Fatal()
	}

	if len(zkPassFile) > 0 && len(zkUser) == 0 {
		logger.Crit("error arguments: not found zkuser. Use '-zkuser' with '-zkpass-file'")
		log.Fatal()
	}

	if len(zkPassFile) > 0 && len(zkUser) > 0 {
		if mypass, err := ioutil.ReadFile(zkPassFile); err != nil {
			logger.Crit("error read %s: %s", zkPassFile, err)
		} else {
			zkPass := string(bytes.TrimSuffix(mypass, []byte("\n")))
			zkAddress = zklib.NewZkAddress(zkUser, zkPass, zkServers...)
		}
	}

	if len(zkTokenFile) > 0 {
		if mytoken, err := ioutil.ReadFile(zkTokenFile); err != nil {
			logger.Crit("error read %s: %s", zkTokenFile, err)
		} else {
			zkToken := string(bytes.TrimSuffix(mytoken, []byte("\n")))
			zkAddress = zklib.NewZkAddressWithToken(zkToken, zkServers...)
		}

	}

	zkconn, err := zklib.NewZkConnect(zkAddress)
	if err != nil {
		logger.Crit("error connect to %s, error %s", zkServers, err)
		os.Exit(1)
	}
	zknode = zklib.NewZkNode(zkPath)

	wg.Add(8)
	go RunReadZookeeperThread(&zkconn, &zknode, &wg)               //тред на чтение данных из zookeeper
	go RunWriteZookeeperThread(&zkconn, &zknode, &zkStopNode, &wg) //тред на запись данных в zookeeper

	go RunUpdateMySQLHostsThread(&zknode, instanceNames.ToList(), &wg)            //добавление/удаление хостов из мониторинга состояния
	go mysqls.RunCheckRolesMySQLThread(&mygroup, &wg)                             //запуск мониторингов состояния для проверки роли у mysql
	go RunUpdateMySQLMasterThread(&zknode, &mygroup, instanceNames.ToList(), &wg) //обновление мониторинга состояния в zknode

	go RunJugglerMonitorThread(&zkconn, &zknode, &zkStopNode, &mygroup, jugglerAddress, environ, &wg) //для отправки значений в juggler
	go RunHTTPServerThread(&wg)                                                                       //для возможности руками дернуть ручку статуса
	go LoggingCurrentState(&mygroup, instanceNames.ToList(), &wg)                                     //раз в N минут выводим в лог текущее состояние мастеров

	wg.Wait()
}
