package main

import (
	"fmt"
	"strings"
	"sync"
	"testing"
	"time"

	"a.yandex-team.ru/direct/infra/go-libs/pkg/zklib"
	"github.com/go-zookeeper/zk"

	"a.yandex-team.ru/library/go/core/resource"
)

type ZkNodeStruct struct {
	Version   *int32
	Data      *[]byte
	Exists    *bool
	SavedData *zklib.ZkNode
}

func NewZkNodeStruct(data []byte, version int32) ZkNodeStruct {
	saved := zklib.NewZkNode("")
	return ZkNodeStruct{
		Version:   &version,
		Data:      &[]byte{},
		SavedData: &saved,
	}
}

func (z ZkNodeStruct) CheckConnect() bool {
	return true
}

func (z ZkNodeStruct) LoadNode(node interface{}) (zklib.ZkNode, error) {
	var newZkNode = zklib.NewZkNode("")
	*newZkNode.Data = *z.Data
	*newZkNode.Stat = zk.Stat{
		Version: *z.Version,
	}
	newZkNode.Path = fmt.Sprint(node)
	newZkNode.LastReadVersion = z.Version
	*newZkNode.LastReadTime = time.Now()
	return newZkNode, nil
}

func (z ZkNodeStruct) SaveNode(node zklib.ZkNode) (zklib.ZkNode, error) {
	*z.SavedData = node
	*z.Version = node.GetNodeVersion()
	return node, nil
}

func (z ZkNodeStruct) ExistNode(node interface{}) (bool, error) {
	if strings.Contains(fmt.Sprint(node), "stop_flag") {
		return false, nil
	}
	return true, nil
}

//Проверяем, что происходит перечитывание при обновлении данных из zookeeper
func TestRunReadZookeeperThread(t *testing.T) {
	zknode := zklib.NewZkNode("/data1.json")
	startData := resource.Get("/data1.json")
	var wg sync.WaitGroup
	zkconn := NewZkNodeStruct(startData, 5)
	wg.Add(1)
	go RunReadZookeeperThread(zkconn, &zknode, &wg)
	go func(wg *sync.WaitGroup) {
		time.Sleep(20 * time.Second)
		wg.Done()
	}(&wg)
	time.Sleep(5 * time.Second)
	if zknode.GetNodeVersion() != 5 {
		t.Errorf("wrong zknode version")
	}
	*zkconn.Version = 6
	time.Sleep(5 * time.Second)
	if zknode.GetNodeVersion() != 6 {
		t.Errorf("wrong zknode version")
	}
	wg.Wait()
}

func TestRunWriteZookeeperThread(t *testing.T) {
	zknode := zklib.NewZkNode("/data1.json")
	zkstop := zklib.NewZkNode("./gotest/stop_flag")
	startData := resource.Get("/data1.json")
	var wg sync.WaitGroup
	zkconn := NewZkNodeStruct(startData, 5)
	wg.Add(2)
	go RunReadZookeeperThread(zkconn, &zknode, &wg)
	go RunWriteZookeeperThread(zkconn, &zknode, &zkstop, &wg)
	go func(wg *sync.WaitGroup) {
		time.Sleep(20 * time.Second)
		wg.Done()
		wg.Done()
	}(&wg)
	time.Sleep(5 * time.Second)
	zknode.Stat.Version = 6
	time.Sleep(5 * time.Second)
	if *zkconn.Version != zknode.GetNodeVersion() {
		t.Errorf("error update zknode version")
	}
	if len(*zkconn.Data) != len(*zknode.Data) {
		t.Errorf("error update data")
	}
	wg.Wait()
}
