package main

import (
	"os"
	"os/signal"
	"sync"
	"syscall"
	"time"

	logger "a.yandex-team.ru/direct/infra/go-libs/pkg/logformat"
	"a.yandex-team.ru/direct/infra/go-libs/pkg/zklib"
)

const (
	ReadDurationZookeeper  = 3 * time.Second
	WriteDurationZookeeper = 1 * time.Second
)

type ZkConnInterface interface {
	CheckConnect() bool
	LoadNode(interface{}) (zklib.ZkNode, error)
	SaveNode(zklib.ZkNode) (zklib.ZkNode, error)
	ExistNode(interface{}) (bool, error)
}

func RunReadZookeeperThread(zkconn ZkConnInterface, zknode *zklib.ZkNode, wg *sync.WaitGroup) {
	defer wg.Done()
	sig := make(chan os.Signal, 1)
	signal.Notify(sig, syscall.SIGHUP)
	t := time.NewTicker(ReadDurationZookeeper)
	update := func() {
		if ok := zkconn.CheckConnect(); !ok {
			logger.Warn("reconnect to zookeeper")
			return
		}
		tmp, err := zkconn.LoadNode(zknode.GetPath())
		if err != nil {
			logger.Crit("error get config %s, error %s", zknode.GetPath(), err)
			return
		}
		if zknode.GetLastReadVersion() < tmp.GetNodeVersion() {
			zknode.Locker().Lock()
			*zknode.Data = *tmp.Data
			*zknode.Stat = *tmp.Stat
			*zknode.LastReadVersion = *tmp.LastReadVersion
			zknode.Locker().Unlock()
		}
		*zknode.LastReadTime = *tmp.LastReadTime
	}
	t2 := time.NewTicker(1 * time.Second)
	for range t2.C {
		select {
		case <-t.C:
			logger.Debug("cron load dbconfig")
			update()
		case s := <-sig:
			logger.Debug("signal %s load dbconfig", s)
			update()
		}
	}
}

func RunWriteZookeeperThread(zkconn ZkConnInterface, zknode *zklib.ZkNode, stopNode *zklib.ZkNode, wg *sync.WaitGroup) {
	defer wg.Done()
	t := time.NewTicker(WriteDurationZookeeper)

	for range t.C {
		if zknode.GetLastReadVersion() == -1 || zknode.GetLastReadVersion() >= zknode.GetNodeVersion() {
			continue
		}

		if diff := time.Since(zknode.GetLastReadTime()); diff > ReadDurationZookeeper*3 {
			logger.Crit("current zknode older than ReadDurationZookeeper: %s more %s(%s). Skip write",
				diff, ReadDurationZookeeper, zknode.GetLastReadTime())
			continue
		}

		if ok, err := zkconn.ExistNode(stopNode.GetPath()); ok {
			logger.Crit("found stop flag %s, skip write zknode %s", stopNode.GetPath(), zknode.GetPath())
			continue
		} else if err != nil {
			logger.Warn("error check exist node %s: %s. Skip save %s", stopNode.GetPath(), err, zknode.GetPath())
			continue
		}
		zknode.Locker().Lock()
		n, err := zkconn.SaveNode(*zknode)
		if err != nil {
			logger.Warn("error save %s, error %s", zknode.GetPath(), err)
		} else {
			logger.Info("success save %s", zknode.GetPath())
			*zknode.LastWriteTime = *n.LastWriteTime
			_ = syscall.Kill(os.Getpid(), syscall.SIGHUP) //перечитываем конфиг из zookeeper
		}
		zknode.Locker().Unlock()
	}
}
