package main

import (
	"flag"
	"fmt"
	"log"
	"sort"
	"strings"

	"a.yandex-team.ru/direct/infra/go-libs/pkg/flags"
	logger "a.yandex-team.ru/direct/infra/go-libs/pkg/logformat"
	"a.yandex-team.ru/direct/infra/go-libs/pkg/oauthlib"
	tm "a.yandex-team.ru/direct/infra/go-libs/pkg/tmlib"
)

const (
	DefaultFolderID      = "mdb-junk"
	DefaultProject       = "direct"
	DefaultTypeTransport = "SNAPSHOT_AND_INCREMENT"
	DefaultYavKey        = "mysql_loaddb_user"
	DefaultYavVersionID  = "ver-01e971q19vswaewex55wy29v2f"
	DefaultDatabase      = "ppc"
	DefaultUser          = "loaddb"
	DefaultMysqlPort     = "3306"
	DefaultDebugMode     = false
	DefaultVerbose       = false
	DefaultTokenFile     = ".tm.token"
	DefaultConfigFile    = "/etc/yandex/tmclient.yaml"
	DefaultReplicationID = "42"
	DefaultSkipKeyChecks = false
)

func main() {
	config := flag.String("config", DefaultConfigFile, "default config path")
	debug := flag.Bool("debug", false, "debug mode")
	flag.Parse()

	cnf, err := LoadingConfig(*config)
	if err != nil && *debug {
		log.Printf("cannot load config %s: %s, used defaults values\n", *config, err)
	}

	args := flags.ArgsBlocks{
		flags.NewArgsBlock("global",
			[]flags.Flager{
				flags.NewFlag("debug", "debug mode", cnf.Find("debug", DefaultDebugMode)),
				flags.NewFlag("login", "login for autorization OAuth(staff)", cnf.Find("login", "")),
				flags.NewFlag("token-file", "OAuth token for connect", cnf.Find("token-file", DefaultTokenFile))}),
		flags.NewArgsBlock("help", []flags.Flager{}),
		flags.NewArgsBlock("list-transfers",
			[]flags.Flager{
				flags.NewFlag("verbose", "show operations for transferID", cnf.Find("verbose", DefaultVerbose)),
				flags.NewFlag("folder-id", "folderID for listing", cnf.Find("folder-id", DefaultFolderID))}),
		flags.NewArgsBlock("list-endpoints",
			[]flags.Flager{
				flags.NewFlag("project-name", "имя проекта для добавления в поле description", cnf.Find("project-name", DefaultProject)),
				flags.NewFlag("instance-name", "имя инстанса для добавления в поле description",
					cnf.Find("instance-name", "")),
				flags.NewFlag("folder-id", "идентификатор folderID", cnf.Find("folder-id", DefaultFolderID))}),
		flags.NewArgsBlock("detail-transfer",
			[]flags.Flager{
				flags.NewFlag("transfer-id", "идентификатор transferID", cnf.Find("folder-id", DefaultFolderID))}),
		flags.NewArgsBlock("detail-endpoint",
			[]flags.Flager{
				flags.NewFlag("endpoint-id", "идентификатор endpointID", cnf.Find("endpoint-id", ""))}),
		flags.NewArgsBlock("delete-endpoint",
			[]flags.Flager{
				flags.NewFlag("endpoint-id", "идентификатор endpointID", cnf.Find("endpoint-id", ""))}),
		flags.NewArgsBlock("delete-transfer",
			[]flags.Flager{
				flags.NewFlag("transfer-id", "идентификатор transferID", cnf.Find("transfer-id", DefaultFolderID))}),
		flags.NewArgsBlock("deactivate-transfer",
			[]flags.Flager{
				flags.NewFlag("abort-operations", "остановить операции привязанные к трансферу",
					cnf.Find("abort-operations", false)),
				flags.NewFlag("transfer-id", "идентификатор transferID", cnf.Find("transfer-id", ""))}),
		flags.NewArgsBlock("activate-transfer",
			[]flags.Flager{
				flags.NewFlag("transfer-id", "идентификатор transferID", cnf.Find("transfer-id", ""))}),
		flags.NewArgsBlock("create-mysql-endpoint",
			[]flags.Flager{
				flags.NewFlag("project-name", "имя проекта для добавления в поле description",
					cnf.Find("project-name", DefaultProject)),
				flags.NewFlag("folder-id", "идентификатор folderID",
					cnf.Find("folder-id", DefaultFolderID)),
				flags.NewFlag("instance-name", "имя инстанса для добавления в поле description",
					cnf.Find("instance-name", "")),
				flags.NewFlag("database", "имя базы данных, к которой подключаться",
					cnf.Find("database", DefaultDatabase)),
				flags.NewFlag("slot-id", "идентификатор репликации MySQL",
					cnf.Find("slot-id", DefaultReplicationID)),
				flags.NewFlag("user", "имя пользовтаеля для подключения к БД",
					cnf.Find("user", DefaultUser)),
				flags.NewFlag("port", "порт для подключения к БД",
					cnf.Find("port", DefaultMysqlPort)),
				flags.NewFlag("mysql-cluster", "имя FQDN или ClusterID MDB, куда подключаться",
					cnf.Find("mysql-cluster", "")),
				flags.NewFlag("delivery-type", "источник или потребитель(source/target)",
					cnf.Find("delivery-type", "")),
				flags.NewFlag("yav-key", "имя YAV ключа для подключения",
					cnf.Find("yav-key", DefaultYavKey)),
				flags.NewFlag("yav-version-id", "имя YAV версии ключа",
					cnf.Find("yav-version-id", DefaultYavVersionID)),
				flags.NewFlag("skip-constraint-checks", "разрешить разбивку больших транзакций на чанки и отключение foreign keys",
					cnf.Find("skip-constraint-checks", DefaultSkipKeyChecks))}),
		flags.NewArgsBlock("create-transfer",
			[]flags.Flager{
				flags.NewFlag("project-name", "имя проекта для добавления в поле description",
					cnf.Find("project-name", DefaultProject)),
				flags.NewFlag("folder-id", "идентификатор folderID",
					cnf.Find("folder-id", DefaultFolderID)),
				flags.NewFlag("instance-name", "имя инстанса для добавления в поле description",
					cnf.Find("instance-name", "")),
				flags.NewFlag("source-endpoint-id", "EndpointID источника для скачивания БД",
					cnf.Find("source-endpoint-id", "")),
				flags.NewFlag("destination-endpoint-id", "EndpointID приёмника для скачивания БД",
					cnf.Find("destination-endpoint-id", "")),
				flags.NewFlag("transport-type", "тип доставки TransportType",
					cnf.Find("transport-type", DefaultTypeTransport))}),
		flags.NewArgsBlock("create-mysql-transport",
			[]flags.Flager{
				flags.NewFlag("project-name", "имя проекта для добавления в поле description",
					cnf.Find("project-name", DefaultProject)),
				flags.NewFlag("folder-id", "идентификатор folderID",
					cnf.Find("folder-id", DefaultFolderID)),
				flags.NewFlag("database", "имя базы данных, к которой подключаться",
					cnf.Find("database", DefaultDatabase)),
				flags.NewFlag("slot-id", "идентификатор репликации MySQL",
					cnf.Find("slot-id", DefaultReplicationID)),
				flags.NewFlag("instance-name", "имя инстанса для добавления в поле description",
					cnf.Find("instance-name", "")),
				flags.NewFlag("user", "имя пользовтаеля для подключения к БД",
					cnf.Find("user", DefaultUser)),
				flags.NewFlag("source-mysql-port", "порт для подключения к БД источнику",
					cnf.Find("source-mysql-port", DefaultMysqlPort)),
				flags.NewFlag("destination-mysql-port", "порт для подключения к БД приёмнику",
					cnf.Find("destination-mysql-port", DefaultMysqlPort)),
				flags.NewFlag("transport-type", "тип доставки TransportType",
					cnf.Find("transport-type", DefaultTypeTransport)),
				flags.NewFlag("yav-key", "имя YAV ключа для подключения",
					cnf.Find("yav-key", DefaultYavKey)),
				flags.NewFlag("yav-version-id", "имя YAV версии ключа",
					cnf.Find("yav-version-id", DefaultYavVersionID)),
				flags.NewFlag("source-mysql", "FQDN или ClusterID MDB имя источника",
					cnf.Find("source-mysql", "")),
				flags.NewFlag("destination-mysql", "FQDN или ClusterID MDB имя источника",
					cnf.Find("destination-mysql", "")),
				flags.NewFlag("skip-constraint-checks", "разрешить разбивку больших транзакций на чанки и отключение foreign keys",
					cnf.Find("skip-constraint-checks", DefaultSkipKeyChecks))}),
		flags.NewArgsBlock("delete-mysql-transport",
			[]flags.Flager{
				flags.NewFlag("transfer-id", "идентификатор transferID", cnf.Find("transfer-id", ""))}),
		flags.NewArgsBlock("logs-transfer",
			[]flags.Flager{
				flags.NewFlag("transfer-id", "просмотр логов по указанномуtransferID", cnf.Find("transfer-id", ""))}),
		flags.NewArgsBlock("check-operation",
			[]flags.Flager{
				flags.NewFlag("operation-id", "OperationID полученный при запуске операции",
					cnf.Find("operation-id", ""))}),
		flags.NewArgsBlock("pause-transfer",
			[]flags.Flager{
				flags.NewFlag("transfer-id", "идентификатор transferID", cnf.Find("transfer-id", ""))}),
		flags.NewArgsBlock("unpause-transfer",
			[]flags.Flager{
				flags.NewFlag("transfer-id", "идентификатор transferID", cnf.Find("transfer-id", ""))}),
		flags.NewArgsBlock("mysql-checksum",
			[]flags.Flager{
				flags.NewFlag("transfer-id", "идентификатор transferID", cnf.Find("transfer-id", ""))}),
		flags.NewArgsBlock("update-endpoint",
			[]flags.Flager{
				flags.NewFlag("skip-constraint-checks", "разрешить разбивку больших транзакций на чанки и отключение foreign keys",
					cnf.Find("skip-constraint-checks", DefaultSkipKeyChecks)),
				flags.NewFlag("user", "имя пользовтаеля для подключения к БД",
					cnf.Find("user", DefaultUser)),
				flags.NewFlag("port", "порт для подключения к БД источнику",
					cnf.Find("port", DefaultMysqlPort)),
				flags.NewFlag("database", "имя базы данных, к которой подключаться",
					cnf.Find("database", DefaultDatabase)),
				flags.NewFlag("slot-id", "идентификатор репликации MySQL",
					cnf.Find("slot-id", DefaultReplicationID)),
				flags.NewFlag("endpoint-id", "идентификатор endpointID", cnf.Find("endpoint-id", ""))}),
	}

	params, err := args.Parse()
	if err != nil {
		log.Fatalf("error parse %s", err)
	}
	if ok := cnf.Find("debug", DefaultDebugMode); ok.(bool) {
		for key, val := range params.ProgramFlags {
			log.Printf("%s %+v\n", key, val)
		}
	}

	if params.ProgramFlags["debug"].Value.(bool) {
		logger.WithLogLevel(logger.DebugLvl)
	} else {
		logger.WithLogLevel(logger.InfoLvl)
	}

	var oauthToken string
	myArgs := params.ProgramFlags
	oauthData := oauthlib.NewOAuthToken(
		"",
		"",
		myArgs["token-file"].Value.(string),
		myArgs["login"].Value.(string))
	if oauthToken, err = oauthData.LoadOAuthToken(); err != nil {
		log.Fatalf("error load token %v: %s", oauthData, err)
	}

	conn, err := tm.NewConnection(oauthToken)
	if err != nil {
		log.Fatalf("error new connection: %s", err)
	}

	switch params.Command {
	case "help":
		args.PrintUsage()
	case "list-endpoints":
		err := tm.PrintEndpoints(conn,
			myArgs["folder-id"].Value.(string),
			myArgs["project-name"].Value.(string),
			myArgs["instance-name"].Value.(string))
		if err != nil {
			log.Fatal(err)
		}
	case "create-mysql-endpoint":
		mysqlPassword, err := tm.NewYavPassword(
			myArgs["yav-key"].Value.(string),
			myArgs["yav-version-id"].Value.(string))
		if err != nil {
			log.Fatal(err)
		}
		resp, err := tm.CreateMysqlEndpointTM(conn,
			myArgs["folder-id"].Value.(string),
			myArgs["project-name"].Value.(string),
			myArgs["instance-name"].Value.(string),
			myArgs["database"].Value.(string),
			myArgs["slot-id"].Value.(string),
			myArgs["mysql-cluster"].Value.(string),
			myArgs["user"].Value.(string),
			myArgs["port"].Value.(string),
			mysqlPassword,
			myArgs["delivery-type"].Value.(string),
			myArgs["skip-constraint-checks"].Value.(bool),
		)
		if err != nil {
			log.Fatal(err)
		}
		fmt.Println(resp)

	case "create-mysql-transfer":
		resp, err := tm.CreateMysqlTransferTM(conn,
			myArgs["folder-id"].Value.(string),
			myArgs["project-name"].Value.(string),
			myArgs["instance-name"].Value.(string),
			myArgs["source-mysql"].Value.(string),
			myArgs["destination-mysql"].Value.(string),
			myArgs["source-endpoint-id"].Value.(string),
			myArgs["destination-endpoint-id"].Value.(string),
			myArgs["transport-type"].Value.(string),
		)
		if err != nil {
			log.Fatal(err)
		}
		fmt.Println(resp)
	case "deactivate-transfer":
		transferID := myArgs["transfer-id"].Value.(string)
		rt, err := tm.DeactivateTransferTM(conn,
			transferID)
		if err != nil {
			log.Print(err)
		}
		log.Printf("[DONE] deactivating transfer %s: %+v", transferID, rt)
		if myArgs["abort-operations"].Value.(bool) {
			if ops, err := tm.OperationsByTransferTM(conn, transferID); err == nil {
				for _, op := range ops.All() {
					//прибиваем все операции
					//log.Printf("[DONE] deactivating operation %s\n", op.ID)
					if ort, err := tm.AbortOperationTM(conn, op.ID); err == nil {
						log.Printf("[DONE] deactivating operation %s: %+v\n", op.ID, ort)
					} else if e, ok := err.(tm.HTTPError); ok && e.Code != 404 {
						log.Printf("[ERROR] deactivating operation %s: %s\n", op.ID, err)
					}
				}
			} else {
				fmt.Printf("\t\t error get operations for %s: %s\n", transferID, err)
			}
		}
	case "activate-transfer":
		rt, err := tm.ActivateTransferTM(conn,
			myArgs["transfer-id"].Value.(string))
		if err != nil {
			log.Fatal(err)
		}
		log.Printf("%+v", rt)
	case "logs-transfer":
		rt, err := tm.LogsTransferTM(conn,
			myArgs["transfer-id"].Value.(string))
		if err != nil {
			log.Fatal(err)
		}
		fmt.Printf("%s", rt)
	case "operation-id":
		operationStatus := tm.CheckOperationDoneTM(conn,
			myArgs["operation-id"].Value.(string))
		rt, err := operationStatus.Marshal()
		if err != nil {
			log.Fatal(err)
		} else {
			fmt.Printf("%+v", rt)
		}
	case "create-mysql-transport":
		mysqlPassword, err := tm.NewYavPassword(
			myArgs["yav-key"].Value.(string),
			myArgs["yav-version-id"].Value.(string))
		if err != nil {
			log.Fatal(err)
		}
		rt, err := tm.CreateMysqlTransportTM(conn,
			myArgs["folder-id"].Value.(string),
			myArgs["project-name"].Value.(string),
			myArgs["instance-name"].Value.(string),
			myArgs["database"].Value.(string),
			myArgs["slot-id"].Value.(string),
			myArgs["source-mysql"].Value.(string),
			myArgs["destination-mysql"].Value.(string),
			myArgs["source-mysql-port"].Value.(string),
			myArgs["destination-mysql-port"].Value.(string),
			myArgs["user"].Value.(string),
			myArgs["transport-type"].Value.(string),
			mysqlPassword,
			myArgs["skip-constraint-checks"].Value.(bool),
		)
		if err != nil {
			log.Fatal(err)
		} else {
			fmt.Printf("%+v", rt)
		}
	case "update-endpoint":
		params := make(tm.MapStringInterface)
		for k, v := range myArgs {
			if len(fmt.Sprint(v)) > 0 {
				params[strings.ReplaceAll(k, "-", "_")] = v.Value
			}
		}
		rt, err := tm.UpdateEndpointTM(conn, myArgs["endpoint-id"].Value.(string), params)
		if err != nil {
			log.Fatal(err)
		} else {
			fmt.Printf("%+v\n", rt)
		}
	case "list-transfers":
		transferStatusAll, _ := tm.TransfersByAllStatus(conn,
			myArgs["folder-id"].Value.(string))
		for status, transfers := range transferStatusAll {
			sort.Sort(transfers)
			fmt.Printf("Status %s\n", status)
			for _, transfer := range transfers.Transfers {
				fmt.Printf("\t transferID: %s, %s(%s) ---> %s(%s)\n", transfer.ID, transfer.SourceTM.Name,
					transfer.SourceTM.ID, transfer.TargetTM.Name, transfer.TargetTM.ID)
				if myArgs["verbose"].Value.(bool) {
					if ops, err := tm.OperationsByTransferTM(conn, transfer.ID); err == nil {
						for _, op := range ops.Success() {
							//показываем только незавершенные операции
							fmt.Printf("\t\t operation: %s done %t\n", op.ID, op.Done)
						}
					} else {
						fmt.Printf("\t\t error get operations for %s: %s\n", transfer.ID, err)
					}
				}
			}
		}
	case "detail-endpoint":
		rt, err := tm.DetailEndpointTM(conn,
			myArgs["endpoint-id"].Value.(string))
		if err != nil {
			log.Fatal(err)
		} else {
			fmt.Printf("%+v\n", rt)
		}
	case "detail-transfer":
		rt, err := tm.TransferDetailByID(conn,
			myArgs["transfer-id"].Value.(string))
		if err != nil {
			log.Fatal(err)
		} else {
			fmt.Printf("%+v\n", rt)
		}
	case "pause-transfer":
		rt, err := tm.PauseTransferTM(conn,
			myArgs["transfer-id"].Value.(string))
		if err != nil {
			log.Fatal(err)
		} else {
			fmt.Printf("%+v\n", rt)
		}
	case "unpause-transfer":
		rt, err := tm.UnpauseTransferTM(conn,
			myArgs["transfer-id"].Value.(string))
		if err != nil {
			log.Fatal(err)
		} else {
			fmt.Printf("%+v\n", rt)
		}
	case "run-checksum":
		rt, err := tm.RunMySQLChecksum(conn,
			myArgs["transfer-id"].Value.(string))
		if err != nil {
			log.Fatal(err)
		} else {
			fmt.Printf("%+v\n", rt)
		}
	case "delete-mysql-transport":
		err := tm.DeleteMysqlTransportTM(conn,
			myArgs["transfer-id"].Value.(string))
		if err != nil {
			log.Fatal(err)
		} else {
			fmt.Println("DONE")
		}
	case "delete-transfer":
		rt, err := tm.DeleteTransferTM(conn,
			myArgs["transfer-id"].Value.(string))
		if err != nil {
			log.Fatal(err)
		} else {
			fmt.Printf("%+v\n", rt)
		}
	case "delete-endpoint":
		rt, err := tm.DeleteEndpointTM(conn,
			myArgs["endpoint-id"].Value.(string))
		if err != nil {
			log.Fatal(err)
		} else {
			fmt.Printf("%+v\n", rt)
		}
	}
}
