package main

import (
	"context"
	"flag"
	"fmt"
	"io/ioutil"
	"log"
	"os"
	"strings"
	"sync"
	"time"

	y "a.yandex-team.ru/direct/infra/go-libs/pkg/yttransfer"
	yp "a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yt"
)

var (
	YTSrcCluster, YTDstCluster                                ClusterName
	YTSrcDirectory, YTDstDirectory, ReplicatorLink            Path
	YTClient, YTToken, FilteredNodes                          string
	EnableConvertTables, EnableCopyTables, RemoveSendingFiles bool
	NodesForCopy                                              MyNodes
)

var usage = "Программа копирует данные между одним/разными кластерами, либо перемещает в рамках одного кластера.\n" +
	"1. Копирование между zeno и seneca-sas БД репликатора:\n" +
	"%[1]s -yt-src-cluster zeno -yt-dst-cluster seneca-sas" +
	" -yt-src-directory //home/direct-np/test/mysql-sync/testing/2020-05-10" +
	" -yt-dst-directory //home/direct-np/test/mysql-sync/testing/2020-05-10" +
	" -yt-token /home/dspushkin/token" +
	" -enable-copy-tables -enable-convert-tables\n" +
	"В ходе копирования, если будут обнаружены динамические таблицы, они будут на время freeze и перекопированы " +
	" по соседству в статические с суффиксом '_static'" +
	" Например: bids.index -> bids.index_static, bids -> bids_static." +
	" По окончании копирования через TM '*_static' т '*_pivot' таблицы будут удалены.\n" +
	"2. Перемещение директории в рамках одного кластера zeno БД репликатора:\n" +
	"%[1]s -yt-src-cluster zeno" +
	" -yt-src-directory //home/direct-np/test/mysql-sync/testing/2020-05-10" +
	" -yt-dst-directory //home/direct-np/test/mysql-sync/testing/2020-05-11" +
	" -yt-token /home/dspushkin/token" +
	" -enable-copy-tables -enable-convert-tables\n" +
	" -remove-sending-files\n" +
	"Происходит отмонтирование всех вложенных дирректорий, рекурсивное перемещение родительской дирректории" +
	" и последующие монтирование таблиц.\n" +
	"3. Копирование директории в рамках одного кластера zeno БД репликатора:\n" +
	"%[1]s -yt-src-cluster zeno" +
	" -yt-src-directory //home/direct-np/test/mysql-sync/testing/2020-05-10" +
	" -yt-dst-directory //home/direct-np/test/mysql-sync/testing/2020-05-11" +
	" -yt-token /home/dspushkin/token" +
	" -enable-copy-tables -enable-convert-tables\n" +
	"Происходит freeze всех таблиц, конвертирование динамических таблиц, копирование из родительской дирректории" +
	" и последующие конвертирование и монтирование в новой.\n"

type Path struct {
	value string
	set   bool
}

type NodeDontExists error

func (p *Path) String() string {
	return p.Value()
}

func (p *Path) Set(s string) error {
	if len(s) == 0 {
		return fmt.Errorf("empty set directory")
	}
	if !strings.HasPrefix(s, "//") {
		return fmt.Errorf("error directory path, need prefix '//'")
	}
	*p = Path{strings.TrimSuffix(s, "/"), true}
	return nil
}

func (p *Path) Valid() bool {
	if len(p.Value()) > 0 && p.set {
		return true
	}
	return false
}

func (p *Path) Value() string {
	return fmt.Sprint((*p).value)
}

type ClusterName struct {
	value string
	set   bool
}

func (cn *ClusterName) String() string {
	return (*cn).Value()
}

func (cn *ClusterName) Set(s string) error {
	if len(s) == 0 {
		return fmt.Errorf("don't set cluster")
	}
	*cn = ClusterName{s, true}
	return nil
}

func (cn *ClusterName) Value() string {
	return fmt.Sprint((*cn).value)
}

func (cn *ClusterName) Valid() bool {
	if len(cn.value) > 0 && cn.set {
		return true
	}
	return false
}

func PrintUsage() {
	fmt.Println(usage)
	flag.PrintDefaults()
}

func CopyDirectoryLocal(cluster ClusterName, src, dst Path, client, token string,
	nodesForCopy MyNodes, movingDirectory bool) error {
	if !cluster.Valid() || !src.Valid() || !dst.Valid() {
		return fmt.Errorf("not valid settings: cluster '%v', src '%v', dst '%v'", cluster, src, dst)
	}

	connect, err := y.NewYtConnect(cluster.Value(), client, token)
	if err != nil {
		return err
	}
	defer connect.Client.Stop()

	cntx, cancel := context.WithTimeout(context.Background(), 600*time.Second)
	defer cancel()

	if ok, err := connect.NodeYtExists(src.Value()); ok && err == nil {
		listNodes, err := connect.RecurseListYtNode(src.Value())
		if err != nil {
			return err
		}
		fmt.Printf("umount tables: %+v", listNodes)

		for _, node := range listNodes {
			if ok := nodesForCopy.HasNode(node); len(nodesForCopy) > 0 && !ok {
				continue
			}
			ytpath := yp.NewRich(src.Value() + node)
			err = connect.Client.UnmountTable(cntx, ytpath.Path, nil)
			if err != nil {
				fmt.Printf("error unmount %v: %s\n", ytpath, err)
			}
		}

		if movingDirectory {
			moveOpts := yt.MoveNodeOptions{
				Recursive: true,
			}

			id, err := connect.Client.MoveNode(cntx, yp.NewRich(src.Value()).Path, yp.NewRich(dst.Value()).Path, &moveOpts)
			if err != nil {
				return err
			}
			fmt.Printf("move operation id: %v\n", id)
		} else {
			copyOpts := yt.CopyNodeOptions{
				IgnoreExisting: false,
				Force:          true,
				Recursive:      true,
			}

			id, err := connect.Client.CopyNode(cntx, yp.NewRich(src.Value()).Path, yp.NewRich(dst.Value()).Path, &copyOpts)
			if err != nil {
				return err
			}
			fmt.Printf("copy operation id: %v\n", id)
		}
	} else if !ok && err == nil {
		fmt.Printf("node %s dosnt exists. Skip\n", src.Value())
	} else {
		return err
	}

	if ok, err := connect.NodeYtExists(dst.Value()); ok && err == nil {
		listNodes, err := connect.RecurseListYtNode(dst.Value())
		if err != nil {
			return err
		}
		fmt.Printf("mount tables: %+v", listNodes)

		for _, node := range listNodes {
			ytpath := yp.NewRich(dst.Value() + node)
			err = connect.Client.MountTable(cntx, ytpath.Path, nil)
			if err != nil {
				fmt.Printf("error mount %v: %s\n", ytpath, err)
			}
		}
	} else if !ok && err == nil {
		fmt.Printf("node %s dosnt exists. Skip\n", dst.Value())
	} else {
		return err
	}

	return nil
}

type Transfer struct {
	SrcCluster, DstCluster ClusterName
	SrcPath, DstPath       string
}

func NewTransfer(SrcCluster, DstCluster ClusterName, SrcPath, DstPath string) Transfer {
	return Transfer{SrcCluster, DstCluster, SrcPath, DstPath}
}

func (t Transfer) GetSourceDirectory() string {
	return t.SrcPath
}

func (t Transfer) GetDestinationDirectory() string {
	return t.DstPath
}

func (t Transfer) GetSourceCluster() string {
	return t.SrcCluster.Value()
}

func (t Transfer) GetDestinationCluster() string {
	return t.DstCluster.Value()
}

type MyNodes []string

func (mn MyNodes) HasNode(node string) bool {
	for _, n := range mn {
		//fmt.Println(n, node, strings.Contains(n, node))
		if n == node {
			return true
		}
	}
	return false
}

func CopyDirectoryTM(srcCluster, dstCluster ClusterName, src, dst Path, client, token string,
	nodesForCopy MyNodes, movingDirectory bool) error {
	if !srcCluster.Valid() || !dstCluster.Valid() || !src.Valid() || !dst.Valid() {
		return fmt.Errorf("not valid settings: src cluster '%v', dst cluster '%v', src '%v', dst '%v'",
			srcCluster, dstCluster, src, dst)
	}
	sconnect, err := y.NewYtConnect(srcCluster.Value(), client, token)
	if err != nil {
		return err
	}
	defer sconnect.Client.Stop()

	if token, err := ioutil.ReadFile(token); err == nil {
		_ = os.Setenv("YT_TOKEN", string(token))
	}

	if ok, err := sconnect.NodeYtExists(src.Value()); ok && err == nil {
		listNodes, err := sconnect.RecurseListYtNode(src.Value())
		if err != nil {
			return err
		}

		var tt y.YtTransferTables
		var staticNodes, tmpNodes []string
		//смотрим список нод на источнике и конвертируем динамику в _static
		for _, node := range listNodes {
			fmt.Println(node, nodesForCopy)
			if ok := nodesForCopy.HasNode(node); len(nodesForCopy) > 0 && !ok {
				continue
			}
			srcpath := src.Value() + node
			dstpath := dst.Value() + node
			pivotName := node + "_pivot"
			pivotPath := src.Value() + pivotName
			var suffix string
			if ok, err := sconnect.IsTableOrFile(srcpath); err != nil {
				fmt.Printf("INFO. dont get 'type' attribute node %s: %s. Skip it.\n", srcpath, err)
				continue
			} else if !ok {
				fmt.Printf("INFO. only support 'type' = ['table', 'file']. Skip %s\n", srcpath)
				continue
			}
			if ok, err := sconnect.IsDynamic(srcpath); err != nil {
				fmt.Printf("INFO. dont get 'dynamic' attribute node %s: %s\n", srcpath, err)
			} else if ok {
				fmt.Printf("INFO. start convert 'dynamic' node %s\n", srcpath)
				if pivotKeys, err := sconnect.GetPivotKeys(srcpath); err != nil {
					fmt.Printf("ERROR. failed get pivot_keys, error %s\n", err)
				} else {
					if err = sconnect.CreateTable(pivotPath, pivotKeys); err != nil {
						fmt.Printf("ERROR. failed create pivot_keys, error %s\n", err)
					} else {
						tmpNodes = append(tmpNodes, pivotPath)
					}
					if err = sconnect.WriteTable(pivotPath, []interface{}{pivotKeys}); err != nil {
						fmt.Printf("ERROR. failed insert pivot_keys, error %s\n", err)
					}
					staticNodes = append(staticNodes, pivotName)
				}
				var wg sync.WaitGroup
				suffix = "_static"
				wg.Add(1)
				cstatus := sconnect.ConvertDynamicTable(srcpath, srcpath+suffix, &wg)
				if cstatus.Error != nil {
					fmt.Printf("ERROR. failed convert %s: %s. Skip it\n", srcpath, cstatus.Error)
					continue
				}

				meta := NewTransfer(srcCluster, dstCluster, srcpath+suffix, dstpath+suffix)
				t := y.YtTransferTable{YtMeta: meta}
				tt = append(tt, t)
				tmpNodes = append(tmpNodes, srcpath+suffix)
			} else {
				staticNodes = append(staticNodes, node)
			}
		}

		//удаляем отправленные _static таблицы
		defer func(deletedNodes *[]string) {
			for _, table := range *deletedNodes {
				if strings.HasSuffix(table, "_static") || strings.HasSuffix(table, "_pivot") {
					if err := sconnect.RemoveYtNode(table, true, false); err != nil {
						fmt.Printf("ERROR. failed remove _static %s cluster %v: %s\n", table, srcCluster, err)
					}
				}
			}
		}(&tmpNodes)

		if srcCluster != dstCluster {
			for _, node := range staticNodes {
				srcpath := src.Value() + node
				dstpath := dst.Value() + node
				if _, ok := tt.FindSourceNodePath(srcpath); !ok {
					meta := NewTransfer(srcCluster, dstCluster, srcpath, dstpath)
					t := y.YtTransferTable{YtMeta: meta}
					tt = append(tt, t)
				}
			}

			fmt.Printf("INFO. start transfer tables: %+v\n", tt)

			//отправляем таблицы через TM
			tasks := tt.StartTransferTables()
			//прерываем скрипт если есть незапущенные таски
			for _, task := range tasks {
				err = *task.LastError
				fmt.Printf("INFO. start task %v error %s\n", task, err.Error())
				_, ok := task.GetTaskID()
				if len(err.Error()) != 0 && ok {
					tasks.AbortTasks()
					return err
				}
			}
			if err := tasks.MonitorTransferTasks(); err != nil {
				return err
			}
		}
		fmt.Println("STEP1. success")
		if movingDirectory {
			fmt.Printf("check destination path %[1]s/%[2]s and remove %[3]s/%[4]s manualy\n", YTDstCluster.Value(),
				YTDstDirectory.Value(), YTSrcCluster.Value(), YTSrcDirectory.Value())
		}
	} else {
		fmt.Printf("WARN. source node %s dosnt exists. Skip it\n", src.Value())
	}
	return nil
}

func ConvertStaticToDynamic(cluster ClusterName, path Path, client, tokenFile string) error {
	conn, err := y.NewYtConnect(cluster.Value(), client, tokenFile)
	if err != nil {
		return err
	}
	defer conn.Client.Stop()
	if ok, err := conn.NodeYtExists(path.Value()); ok && err == nil {
		listNodes2, err := conn.RecurseListYtNode(path.Value())
		if err != nil {
			return err
		}

		var convertTables []string
		for _, node2 := range listNodes2 {
			if strings.Contains(node2, "_static") {
				convertTables = append(convertTables, path.Value()+node2)
				fmt.Printf("INFO. start convert static -> dynamic table %s\n", node2)
			}
		}

		convertsStatus := conn.StartConvertStaticTable(convertTables)

		if failed := convertsStatus.ConvertsFailed(); len(failed) > 0 {
			return fmt.Errorf("WARN. found failed converts yt %v", failed)
		}

		if err := conn.RemoveConvertedTables(convertsStatus); err != nil {
			return fmt.Errorf("WARN. error remove converted tables: %s", err)
		}

	} else {
		return fmt.Errorf("WARN. destination node %s dosnt exists, error %s. Skip it", cluster.Value(), err)
	}
	return nil
}

func UpdateReplicatorLink(cluster ClusterName, dst, link Path, client, token string) error {
	connect, err := y.NewYtConnect(cluster.Value(), client, token)
	if err != nil {
		return err
	}
	defer connect.Client.Stop()

	return connect.UpdateYtLink(link.Value(), dst.Value())
}

func AddRootPath(vals []string) []string {
	result := make([]string, len(vals))
	for i, v := range vals {
		if !strings.HasPrefix(v, "/") {
			v = "/" + v
		}
		result[i] = v
	}
	return result
}

func main() {
	flag.Var(&YTSrcCluster, "yt-src-cluster", "имя кластера источника")
	flag.Var(&YTDstCluster, "yt-dst-cluster", "имя кластера приёмника")
	flag.Var(&YTSrcDirectory, "yt-src-directory", "имя директории которую перемещать")
	flag.Var(&YTDstDirectory, "yt-dst-directory", "имя дирректории куда перемещать")
	flag.StringVar(&YTClient, "yt-client", "direct", "имя аккаунта для подключения к YT")
	flag.StringVar(&YTToken, "yt-token", "/tmp/token", "путь до YT токена")
	flag.Var(&ReplicatorLink, "replicator-link", "обновить link для репликатора")
	flag.BoolVar(&EnableConvertTables, "enable-convert-tables", false, "сконвертировать таблицы на приемнике")
	flag.BoolVar(&EnableCopyTables, "enable-copy-tables", false, "копировать таблицы с источника на приёмник")
	flag.StringVar(&FilteredNodes, "nodes-for-copy", "", "список черезе запятую нод для копирование(пустой - копируем все)")
	flag.Usage = PrintUsage
	flag.Parse()

	if !YTSrcCluster.Valid() {
		log.Fatalf("dont set yt-src-cluster: %t", YTSrcCluster.set)
	}
	if !YTDstDirectory.Valid() || !YTSrcDirectory.Valid() {
		log.Fatalf("dont set yt-src-directory/yt-dst-directory: %t/%t", YTSrcDirectory.set, YTDstDirectory.set)
	}

	if len(FilteredNodes) > 0 {
		NodesForCopy = MyNodes(AddRootPath(strings.Split(FilteredNodes, ",")))
	}

	if len(YTDstCluster.Value()) == 0 || strings.EqualFold(YTSrcCluster.String(), YTDstCluster.String()) {
		if EnableCopyTables {
			method := "copy"
			if RemoveSendingFiles {
				method = "move"
			}
			fmt.Printf("STEP1. start %[1]s path: %[2]s/%[3]s --> %[2]s/%[4]s\n", method, YTSrcCluster.Value(),
				YTSrcDirectory.Value(), YTDstDirectory.Value())
			if err := CopyDirectoryLocal(YTSrcCluster, YTSrcDirectory, YTDstDirectory,
				YTClient, YTToken, NodesForCopy, RemoveSendingFiles); err != nil {
				log.Fatalf("error move path %s -> %s: %s", YTSrcDirectory.Value(), YTDstDirectory.Value(), err)
			} else {
				fmt.Println("STEP1. success")
			}
		}
		if EnableConvertTables {
			fmt.Printf("STEP2. start convert tables for directiry: %[1]s/%[2]s\n", YTSrcCluster.Value(),
				YTDstDirectory.Value())
			if err := ConvertStaticToDynamic(YTSrcCluster, YTDstDirectory, YTClient, YTToken); err != nil {
				log.Fatalf("error convert tables: %s", err)
			}
			fmt.Println("STEP2. success")
		}
		if ReplicatorLink.Valid() {
			if err := UpdateReplicatorLink(YTSrcCluster, YTDstDirectory, ReplicatorLink, YTClient, YTToken); err != nil {
				log.Fatalf("error update replicator link %s: %s", YTDstDirectory.Value(), err)
			} else {
				fmt.Printf("INFO. success link %s/%s --> %s\n", YTSrcCluster.Value(), YTDstDirectory.Value(),
					ReplicatorLink.Value())
			}
		}
		if !EnableCopyTables || !EnableConvertTables || !ReplicatorLink.Valid() {
			flag.PrintDefaults()
		}
	} else if len(YTDstCluster.Value()) != 0 && !strings.EqualFold(YTSrcCluster.String(), YTDstCluster.String()) {
		if EnableConvertTables {
			method := "copy"
			if RemoveSendingFiles {
				method = "move"
			}
			fmt.Printf("STEP1. start %[1]s path: %[2]s/%[3]s --> %[5]s/%[4]s\n", method, YTSrcCluster.Value(),
				YTSrcDirectory.Value(), YTDstDirectory.Value(), YTDstCluster.Value())
			if err := CopyDirectoryTM(YTSrcCluster, YTDstCluster, YTSrcDirectory,
				YTDstDirectory, YTClient, YTToken, NodesForCopy, RemoveSendingFiles); err != nil {
				log.Fatalf("error move directory %[1]s/%[2]s -> %[3]s/%[4]s: %[5]s", YTSrcCluster.Value(), YTSrcDirectory.Value(),
					YTDstCluster.Value(), YTDstDirectory.Value(), err)
			} else {
				fmt.Println("STEP1. success")
			}
		}
		if EnableConvertTables {
			fmt.Printf("STEP2. start convert tables for directiry: %[1]s/%[2]s\n", YTDstCluster.Value(),
				YTDstDirectory.Value())
			if err := ConvertStaticToDynamic(YTDstCluster, YTDstDirectory, YTClient, YTToken); err != nil {
				log.Fatalf("error convert tables: %s", err)
			}
			fmt.Println("STEP2. success")
		}
		if ReplicatorLink.Valid() {
			if err := UpdateReplicatorLink(YTDstCluster, YTDstDirectory, ReplicatorLink, YTClient, YTToken); err != nil {
				log.Fatalf("error update replicator link %s: %s", YTDstDirectory.Value(), err)
			} else {
				fmt.Printf("INFO. success link %s/%s --> %s\n", YTDstCluster.Value(), YTDstDirectory.Value(),
					ReplicatorLink.Value())
			}
		}
		if !EnableCopyTables && !EnableConvertTables && !ReplicatorLink.Valid() {
			flag.PrintDefaults()
		}
	}

}
