package dbconfig

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"strings"

	logger "a.yandex-team.ru/direct/infra/go-libs/pkg/logformat"
	"a.yandex-team.ru/direct/infra/go-libs/pkg/mysqls"
	"a.yandex-team.ru/direct/infra/go-libs/pkg/trylock"
)

type Options func()

type DBConfig struct {
	Config  *map[string]interface{}
	Locks   *trylock.RWMutex
	version *int
}

func NewDBConfig() DBConfig {
	conf := make(map[string]interface{})
	var startVersion int
	return DBConfig{
		Config:  &conf,
		Locks:   &trylock.RWMutex{},
		version: &startVersion,
	}
}

func (d *DBConfig) LoadDBConfig(data []byte, options ...Options) error {
	for _, option := range options {
		option()
	}
	return json.Unmarshal(data, d.Config)
}

func (d DBConfig) GetVersion() int {
	return *d.version
}

func (d DBConfig) GetDBConfig() map[string]interface{} {
	if *d.Config == nil {
		return make(map[string]interface{})
	}
	return *d.Config
}

func (d DBConfig) SetVersion(newversion int) Options {
	return func() {
		*d.version = newversion
	}
}

func (d *DBConfig) GetParamForInstance(instance interface{}, parname string) (interface{}, bool) {
	d.Locks.RLock()
	defer d.Locks.RUnlock()
	var value interface{}
	var nameShard NameShard
	switch v := instance.(type) {
	case NameShard:
		nameShard = v
	case string:
		nameShard = NameAndShardByInstance(v)
	default:
		logger.Warn("unsupported type %t for instance name %v", instance, instance)
		return nil, false
	}

	if strings.EqualFold(parname, "db") {
		return nameShard.DatabaseName(), true
	}
	fullConfig, ok := (*d.Config)["db_config"].(map[string]interface{})
	if !ok {
		logger.Warn("wrong struct db_config")
		return value, false
	}
	if comval, ok := fullConfig[parname]; ok {
		value = comval
	}

	childs, ok := fullConfig["CHILDS"].(map[string]interface{})
	if !ok {
		logger.Warn("wrong struct CHILDS")
		return value, false
	}
	if n := nameShard.DatabaseName(); strings.EqualFold(n, "ppc") {
		childs, ok = childs[n].(map[string]interface{})
		if !ok {
			logger.Warn("wrong struct %s", n)
			return value, false
		}
		childs, ok = childs["CHILDS"].(map[string]interface{})
		if !ok {
			logger.Warn("wrong struct ppc/CHILDS")
			return value, false
		}
	}

	for k, v := range childs {
		if strings.EqualFold(nameShard.NameByConfig(), k) {
			v1, ok := v.(map[string]interface{})
			if !ok {
				logger.Warn("wrong struct instance")
				return value, false
			}
			if v, ok := v1[parname]; ok {
				return v, true
			}
		}
	}
	if len(fmt.Sprint(value)) > 0 {
		return value, true
	}
	return value, false
}

func SplitHosts(hosts string) []string {
	if len(hosts) == 0 {
		return make([]string, 0)
	}
	splittedHost := strings.Split(hosts, ",")
	result := make([]string, len(splittedHost))
	for idx, host := range splittedHost {
		result[idx] = strings.TrimSpace(host)
	}
	return result
}
func (d *DBConfig) ClusterHosts(group string) ([]string, error) {
	hosts, ok := d.GetParamForInstance(group, "cluster_hosts")
	if !ok {
		return []string{}, fmt.Errorf("not found cluster_hosts for instance %s", group)
	}
	if hosts == nil {
		return []string{}, fmt.Errorf("empty cluster_hosts for instance %s", group)
	}
	return SplitHosts(fmt.Sprintf("%s", hosts)), nil
}

func (d *DBConfig) Master(group string) (string, error) {
	host, ok := d.GetParamForInstance(group, "host")
	if !ok {
		return "", fmt.Errorf("not found host for instance %s", group)
	}
	if host == nil {
		return "", fmt.Errorf("empty host for instance %s", group)
	}
	return fmt.Sprintf("%s", host), nil
}

func (d *DBConfig) CreateMysqlInstanceStatus(host, group string) (*mysqls.MysqlInstanceStatus, error) {
	port, ok := d.GetParamForInstance(group, "port")
	if !ok {
		return nil, fmt.Errorf("not found port for instance %s", group)
	}
	user, ok := d.GetParamForInstance(group, "user")
	if !ok {
		return nil, fmt.Errorf("not found user for instance %s", group)
	}
	db, ok := d.GetParamForInstance(group, "db")
	if !ok {
		return nil, fmt.Errorf("not found db for instance %s", group)
	}
	var pass string
	p, ok := d.GetParamForInstance(group, "pass")
	if !ok {
		return nil, fmt.Errorf("not found pass for instance %s", group)
	}
	switch v := p.(type) {
	case map[string]interface{}:
		passPath, ok := v["file"]
		if !ok {
			return nil, fmt.Errorf("not found pass/file for instance %s", group)
		}
		raw, err := ioutil.ReadFile(fmt.Sprintf("%s", passPath))
		if err != nil {
			return nil, fmt.Errorf("error read %s, error %s", passPath, err)
		}
		pass = string(bytes.TrimSuffix(raw, []byte("\n")))
	default:
		pass = fmt.Sprintf("%s", v)
	}
	//временный хак для работы со старыми машинами
	myport := int(port.(float64))
	if myport-3300 > 100 {
		myport = myport - 100
	}
	instance := mysqls.NewMySQLInstance(host, myport, user.(string), pass, db.(string), group)
	return mysqls.NewMysqlInstanceStatus(instance), nil
}

func (d *DBConfig) SetParamForInstance(instance interface{}, parname string, value interface{}) bool {
	d.Locks.RWLock()
	defer d.Locks.RWUnlock()
	common, ok := (*d.Config)["db_config"].(map[string]interface{})
	if !ok {
		logger.Warn("wrong struct db_config")
		return false
	}
	var nameShard NameShard
	switch v := instance.(type) {
	case NameShard:
		nameShard = v
	case string:
		nameShard = NameAndShardByInstance(v)
	default:
		logger.Warn("unsupported type %t for instance name %v", instance, instance)
		return false
	}

	childs, ok := common["CHILDS"].(map[string]interface{})
	if !ok {
		logger.Warn("wrong struct CHILDS")
		return false
	}
	if n := nameShard.DatabaseName(); strings.EqualFold(n, "ppc") {
		childs, ok = childs[n].(map[string]interface{})
		if !ok {
			logger.Warn("wrong struct %s", n)
			return false
		}
		childs, ok = childs["CHILDS"].(map[string]interface{})
		if !ok {
			logger.Warn("wrong struct %s/CHILDS", n)
			return false
		}
	}
	for k, v := range childs {
		if strings.EqualFold(nameShard.NameByConfig(), k) {
			v1, ok := v.(map[string]interface{})
			if !ok {
				logger.Warn("wrong struct instance")
				return false
			}
			v1[parname] = value
		}
	}
	return true
}

func (d *DBConfig) AllInstances() (NameShards, error) {
	d.Locks.RLock()
	defer d.Locks.RUnlock()
	var shards NameShards
	common, ok := (*d.Config)["db_config"].(map[string]interface{})
	if !ok {
		return shards, fmt.Errorf("wrong struct db_config: %+v", *d)
	}
	childs, ok := common["CHILDS"].(map[string]interface{})
	if !ok {
		return shards, fmt.Errorf("wrong struct CHILDS")
	}
	for key, val := range childs {
		name := NameAndShardByInstance(key)
		logger.Debug("[DBConfig.AllInstances] %s, %s", key, name)
		if strings.EqualFold(name.Name, "ppc") {
			ppcChilds, ok := val.(map[string]interface{})
			if !ok {
				logger.Warn("error parse ppcdata structure: %s", val)
				continue
			}
			ppcShardsRaw, ok := ppcChilds["CHILDS"]
			if !ok {
				logger.Warn("not found CHILDS structure: %s", ppcShardsRaw)
				continue
			}
			ppcShards, ok := ppcShardsRaw.(map[string]interface{})
			if !ok {
				logger.Warn("error parse shards structure: %s", ppcShards)
				continue
			}
			for num := range ppcShards {
				name.Shard = num
				shards = append(shards, name)
			}
		}
		if strings.EqualFold(name.Name, "ppcdict") ||
			strings.EqualFold(name.Name, "monitor") ||
			strings.EqualFold(name.Name, "chassis") {
			shards = append(shards, name)
		}
	}
	return shards, nil
}

type NameShard struct {
	Name  string
	Shard string
}

type NameShards []NameShard

func (nss NameShards) ByMap() map[NameShard]int {
	shards := make(map[NameShard]int)
	for _, shard := range nss {
		shards[shard] = 0
	}
	return shards
}

func NameAndShardByInstance(instance string) NameShard {
	var dbname, shard string
	if s := "ppcdata"; strings.Contains(instance, s) {
		dbname = "ppc"
		shard = strings.Replace(instance, s, "", 1)
	} else if s := "ppc"; strings.EqualFold(instance, s) {
		dbname = "ppc"
		shard = strings.Replace(instance, s, "", 1)
	} else if s := "ppcdict"; strings.EqualFold(instance, s) {
		dbname = "ppcdict"
		shard = strings.Replace(instance, s, "", 1)
	} else if s := "dict"; strings.EqualFold(instance, s) {
		dbname = "ppcdict"
		shard = strings.Replace(instance, s, "", 1)
	} else if s := "ppcmonitor"; strings.EqualFold(instance, s) {
		dbname = "monitor"
		shard = strings.Replace(instance, s, "", 1)
	} else if s := "monitor"; strings.EqualFold(instance, s) {
		dbname = "monitor"
		shard = strings.Replace(instance, s, "", 1)
	} else if s := "chassis"; strings.EqualFold(instance, s) {
		dbname = "chassis"
		shard = strings.Replace(instance, s, "", 1)
	}
	return NameShard{
		Name:  dbname,
		Shard: shard,
	}
}

func (ns NameShard) NameByConfig() string {
	if len(ns.Shard) == 0 {
		return ns.Name
	}
	return ns.Shard
}

func (ns NameShard) DatabaseName() string {
	return ns.Name
}

func (ns NameShard) Print() string {
	return ns.Name + ns.Shard
}
