package flags

import (
	"fmt"
	"io/ioutil"
	"os"
	"strings"

	"gopkg.in/yaml.v2"
)

type ArgsBlocks []*ArgsBlock

//Вывести список всех поддерживаемых команд
func (abs ArgsBlocks) AllCommands() []string {
	var commands []string
	for _, block := range abs {
		commands = append(commands, block.Command)
	}
	return commands
}

//Найти блок агрументов к указанной команде
func (abs *ArgsBlocks) FindCommand(command string) (*ArgsBlock, error) {
	for _, block := range *abs {
		if block.Command == command {
			return block, nil
		}
	}
	return nil, fmt.Errorf("not found command '%s' in %v", command, abs.AllCommands())
}

type ArgsBlock struct {
	ProgramName  string //имя программы
	Command      string //команда
	ProgramFlags Flags  //список флагов
	//DefaultFlags map[string]Flag //список default флагов
}

type Flager interface {
	GetName() string
	GetFlag() Flag
}

func NewArgsBlock(cmd string, nameFlags []Flager) *ArgsBlock {
	myflags := make(map[string]Flag)
	for _, f := range nameFlags {
		myflags[f.GetName()] = f.GetFlag()
	}
	return &ArgsBlock{
		Command:      cmd,
		ProgramFlags: myflags,
	}
}

//Установить набор флагов, для указанной команды
func (ab *ArgsBlock) SetFlags(args Flags) {
	saved := *ab
	for key, val := range args {
		if len(fmt.Sprintf("%s", val.Value)) > 0 {
			saved.ProgramFlags[key] = val
		}
	}
	saved.ProgramFlags = args
	*ab = saved
}

type Flag struct {
	Name  string    //имя параметра
	Value FlagValue //значение параметра
	Usage string
}

type FlagValue interface{}

/*
func (fv FlagValue) ToStrings() []string {
	var vals []string
	for _, v := range fv {
		if s, ok := fv.(string); ok {
			vals = append(vals, s)
		}
	}
	return vals
}

func (fv FlagValue) ToInts() []int {
	var vals []int
	for _, v := range fv {
		if i, ok := v.(int); ok {
			vals = append(vals, i)
		}
	}
	return vals
}*/

type Flags map[string]Flag

//Находит значение флага и выводит в формате FlagValue. В случае ошибки заполняет error
func (fs *Flags) FindFlag(name string) (Flag, error) {
	for key, val := range *fs {
		if key == name {
			return val, nil
		}
	}
	return Flag{}, fmt.Errorf("not found flag '%s' in %+v", name, fs)
}

//Выводит список всех доступных флагов
func (fs Flags) AllFlagsName() []string {
	var names []string
	for key := range fs {
		names = append(names, key)
	}
	return names
}

//На основе строки с параметрами получить, блок аргументов
func (fs *Flags) ParseArgs(cmdArgs []string) {
	line := strings.Join(cmdArgs, " ")
	var name string
	line = strings.ReplaceAll(line, "=", " ")
	for _, arg := range strings.Split(line, " ") {
		if strings.HasPrefix(arg, "-") {
			if strings.HasPrefix(arg, "--") {
				name = strings.Replace(arg, "--", "", 1)
			} else {
				name = strings.Replace(arg, "-", "", 1)
			}
			if value, ok := (*fs)[name]; ok {
				switch value.Value.(type) {
				case bool:
					value.Value = true
				case int:
					value.Value = 0
				case string:
					value.Value = ""
				}
				(*fs)[name] = value
			}
		} else {
			saved, ok := (*fs)[name]
			if !ok {
				continue
			}
			if val, ok := saved.Value.(string); ok && len(val) > 0 {
				saved.Value = fmt.Sprintf("%s %s", saved.Value, arg)
			} else if ok {
				saved.Value = arg
			}
			(*fs)[name] = saved
		}
	}
}

//На основе указанных блоков аргументов, парсит os.Args и создает структуру ArgsBlock
func (abs ArgsBlocks) Parse() (*ArgsBlock, error) {
	var cmd string
	//args := make(Flags)
	var err error
	var block, settings *ArgsBlock
	if len(os.Args) >= 2 {
		cmd = os.Args[1]
	} else {
		abs.PrintUsage()
		return nil, nil
	}
	settings, err = abs.FindCommand("global")
	if err != nil {
		return settings, err
	}

	block, err = abs.FindCommand(cmd)
	if err != nil {
		return block, err
	} else {
		for key, val := range block.ProgramFlags {
			settings.ProgramFlags[key] = val
		}
	}

	if len(os.Args) == 2 {
		settings.ProgramFlags.ParseArgs([]string{})
	} else if len(os.Args) >= 2 {
		settings.ProgramFlags.ParseArgs(os.Args[2:len(os.Args)])
	}
	settings.ProgramName = os.Args[0]
	settings.Command = cmd

	return settings, nil
}

func (abs ArgsBlocks) PrintUsage() {
	fmt.Printf("%s [comand] [--flag1 val1 --flag2 val2]\n", abs[0].ProgramName)
	for _, block := range abs {
		if block.Command == "help" {
			continue
		}
		fmt.Printf("\tcommand '%s' flags:\n", block.Command)
		for key, val := range block.ProgramFlags {
			fmt.Printf("\t\t--%-25s\t%s\t(%v)\n", key, val.Usage, val.Value)
		}
	}
}

func (f Flag) GetFlag() Flag {
	return f
}

func (f Flag) GetName() string {
	return f.Name
}

func NewFlag(name, usage string, defvalue interface{}) Flag {
	return Flag{
		Name:  name,
		Usage: usage,
		Value: defvalue,
	}
}

func LoadFlagsByConfig(path string) (Flags, error) {
	config := make(map[string]interface{})
	data, err := ioutil.ReadFile(path)
	if err != nil {
		return nil, err
	}

	err = yaml.Unmarshal(data, &config)
	if err != nil {
		return nil, err
	}
	args := make(Flags)
	for key, val := range config {
		args[key] = NewFlag(key, "", val)
	}
	return args, nil
}
