package juggler

import (
	"context"
	"fmt"
	"time"

	"a.yandex-team.ru/infra/rsm/nvgpumanager/pkg/juggler"
)

var (
	jugglerAddress = juggler.DefaultLocalURL
	jugglerTimeout = 60 * time.Second
	environ        string
)

type Option func()

func WithJugglerAddress(address string) Option {
	return func() {
		jugglerAddress = address
	}
}

func WithEnviron(env string) Option {
	return func() {
		environ = env
	}
}

type JugglerClient struct {
	JugglerService string
}

func NewJugglerClient(service string, opts ...Option) JugglerClient {
	for _, option := range opts {
		option()
	}
	return JugglerClient{service}
}

type JugglerMessage struct {
	JugglerClient
	juggler.JugglerRequest
}

func (jc JugglerClient) GetJugglerService() string {
	if len(environ) > 0 {
		return fmt.Sprintf("%s-%s", jc.JugglerService, environ)
	}
	return jc.JugglerService
}

func (jc JugglerClient) GetUserAgent() string {
	return fmt.Sprintf("%s/1.0", jc.JugglerService)
}

func SetJugglerTimeout(timeout time.Duration) {
	jugglerTimeout = timeout
}

func (jc JugglerClient) NewJugglerRequest(code int, description string) JugglerMessage {
	var status = "CRIT"
	if code == 0 {
		status = "OK"
	}
	if code == 1 {
		status = "WARN"
	}

	event := juggler.JugglerEvent{
		Description: description,
		Service:     jc.GetJugglerService(),
		Status:      status,
	}
	return JugglerMessage{
		jc,
		juggler.JugglerRequest{
			Source: jc.GetJugglerService(),
			Events: []juggler.JugglerEvent{event},
		},
	}
}

func (jm JugglerMessage) Send() error {
	if len(jugglerAddress) == 0 {
		jugglerAddress = juggler.DefaultLocalURL
	}
	c := juggler.NewJugglerClientWithURL(jugglerAddress)
	c.SetHeader("User-Agent", jm.GetUserAgent())
	cntx, cancel := context.WithTimeout(context.Background(), jugglerTimeout)
	defer cancel()
	if err := c.PushEvents(cntx, jm.JugglerRequest); err != nil {
		return fmt.Errorf("error juggler send %s: %s", jugglerAddress, err)
	}
	return nil
}

type JugglerMessages chan JugglerMessage
