package logformat

import (
	"fmt"
	"log"
	"log/syslog"
	"os"
	"time"
)

const (
	UnknownLvl int = iota
	DebugLvl
	InfoLvl
	WarningLvl
	CriticalLvl
)

var (
	logLevel int         = InfoLvl
	logger   interface{} = log.New(os.Stdout, "", 0)
	logfmt               = func() string { return "" }
)

type Option func()

func WithLogLevel(newLogLevel int) Option {
	return func() {
		logLevel = newLogLevel
	}
}

func WithLogger(newLogger interface{}) Option {
	return func() {
		logger = newLogger
	}
}

func WithLogFormat(newLogFormat func() string) Option {
	return func() {
		logfmt = newLogFormat
	}
}

func DirectMessageFormat(args ...interface{}) func() string {
	hostname, _ := os.Hostname()
	var service, method string
	if len(args) > 0 {
		service = fmt.Sprint(args[0])
	}
	if len(args) > 1 {
		method = fmt.Sprint(args[1])
	}
	layout := "2006-01-02:15:04:05.000000000"
	return func() string {
		return fmt.Sprintf("%s %s,%s/%s,%d:%d:%d [] %%s null - %%s",
			time.Now().Format(layout), hostname, service, method, 0, 0, 0)
	}
	//return fmt.Sprintln(prefix, "%s")
}

func NewLoggerFormat(opts ...Option) {
	for _, option := range opts {
		option()
	}
}

func GetLevel() int {
	return logLevel
}

func Debug(msg string, args ...interface{}) {
	message := fmt.Sprintf(logfmt(), "DEBUG", msg)
	if logLevel <= DebugLvl {
		if lg, ok := logger.(*log.Logger); ok {
			lg.Printf(message, args...)
		}
		if lg, ok := logger.(*syslog.Writer); ok {
			_ = lg.Debug(fmt.Sprintf(message, args...))
		}
	}
}

func Info(msg string, args ...interface{}) {
	message := fmt.Sprintf(logfmt(), "INFO", msg)
	if logLevel <= InfoLvl {
		if lg, ok := logger.(*log.Logger); ok {
			lg.Printf(message, args...)
		}
		if lg, ok := logger.(*syslog.Writer); ok {
			_ = lg.Info(fmt.Sprintf(message, args...))
		}
	}
}

func Warn(msg string, args ...interface{}) {
	message := fmt.Sprintf(logfmt(), "WARN", msg)
	if logLevel <= WarningLvl {
		if lg, ok := logger.(*log.Logger); ok {
			lg.Printf(message, args...)
			return
		}
		if lg, ok := logger.(*syslog.Writer); ok {
			_ = lg.Warning(fmt.Sprintf(message, args...))
		}
	}
}

func Crit(msg string, args ...interface{}) {
	message := fmt.Sprintf(logfmt(), "ERROR", msg)
	if logLevel <= CriticalLvl {
		if lg, ok := logger.(*log.Logger); ok {
			lg.Printf(message, args...)
			return
		}
		if lg, ok := logger.(*syslog.Writer); ok {
			_ = lg.Crit(fmt.Sprintf(message, args...))
		}
	}
}

func Trace(msg string, args ...interface{}) {
	message := fmt.Sprintf(logfmt(), "TRACE", msg)
	if logLevel <= InfoLvl {
		if lg, ok := logger.(*log.Logger); ok {
			lg.Printf(message, args...)
			return
		}
		if lg, ok := logger.(*syslog.Writer); ok {
			_ = lg.Notice(fmt.Sprintf(message, args...))
		}
	}
}

func Printf(msg string, args ...interface{}) {
	message := fmt.Sprintf(logfmt(), msg)
	if logLevel <= InfoLvl {
		if lg, ok := logger.(*log.Logger); ok {
			lg.Printf(message, args...)
			return
		}
		if lg, ok := logger.(*syslog.Writer); ok {
			_ = lg.Info(fmt.Sprintf(message, args...))
		}
	}
}
