package mdb

import (
	"bytes"
	"fmt"
	"io/ioutil"
	"net/http"
	"net/url"

	logger "a.yandex-team.ru/direct/infra/go-libs/pkg/logformat"
	"a.yandex-team.ru/direct/infra/go-libs/pkg/sslmode"
	"a.yandex-team.ru/direct/infra/go-libs/pkg/trylock"
)

type Connection struct {
	Client   *http.Client
	Lock     *trylock.Mutex
	MDBToken *CloudToken
}

func NewConnection(keypath string) (connect Connection, err error) {
	client, err := sslmode.NewHTTPSClient()
	if err != nil {
		return
	}
	mdbtoken, err := GenerateAIMRequest(keypath)
	if err != nil {
		return
	}
	lock := trylock.Mutex{}
	connect = Connection{
		Client:   client,
		Lock:     &lock,
		MDBToken: mdbtoken,
	}
	return
}

func (c *Connection) Do(reqtype, address string, values interface{}, body []byte) (out []byte, err error) {
	var req *http.Request
	var myapi string

	switch vals := values.(type) {
	case map[string]string:
		if len(vals) > 0 {
			query := url.Values{}
			for k, v := range vals {
				query.Add(k, v)
			}
			myapi = fmt.Sprintf("%s?%s", address, query.Encode())
		} else {
			myapi = address
		}
	case map[string]interface{}:
		if len(vals) > 0 {
			query := url.Values{}
			for k, v := range vals {
				query.Add(k, fmt.Sprintf("%s", v))
			}
			myapi = fmt.Sprintf("%s?%s", address, query.Encode())
		} else {
			myapi = address
		}
	default:
		myapi = address
	}

	logger.Debug("reqtype: %s, address: %s, body: %s", reqtype, myapi, body)
	if body != nil {
		req, err = http.NewRequest(reqtype, myapi, bytes.NewReader(body))
	} else {
		req, err = http.NewRequest(reqtype, myapi, nil)
	}
	if err != nil {
		return nil, err
	}
	aim, err := c.MDBToken.GetAIMToken(c.Client)
	if err != nil {
		return nil, err
	}
	req.Header.Set("Authorization", fmt.Sprintf("Bearer %s", aim))
	resp, err := (*c.Client).Do(req)
	if err != nil {
		return nil, err
	}
	defer func() { _ = resp.Body.Close() }()
	out, _ = ioutil.ReadAll(resp.Body)
	if resp.StatusCode != 200 {
		return nil, fmt.Errorf("http code: %s, req %s, data: %s", resp.Status, myapi, out)
	}
	if len(out) == 0 {
		return nil, fmt.Errorf("empty body response %s", myapi)
	}
	return
}
