package mdb

import (
	"encoding/json"
	"fmt"
	"math"
	"strconv"
	"strings"

	logger "a.yandex-team.ru/direct/infra/go-libs/pkg/logformat"
)

func MergeStructs(oldData, newData interface{}) ([]byte, error) {
	ojs, newjs := make(map[string]interface{}), make(map[string]interface{})
	jsntmp1, err := json.Marshal(oldData)
	if err != nil {
		return nil, err
	}
	err = json.Unmarshal(jsntmp1, &ojs)
	if err != nil {
		logger.Crit("error marshal %s: %s", jsntmp1, err)
		return nil, err
	}
	jsntmp2, err := json.Marshal(newData)
	if err != nil {
		return nil, err
	}
	err = json.Unmarshal(jsntmp2, &newjs)
	if err != nil {
		logger.Crit("error marshal %s: %s", newData, err)
		return nil, err
	}
	logger.Debug("%s %s", ojs, newjs)

	for k, v := range newjs {
		ojs[k] = v
	}
	data, err := json.Marshal(ojs)

	if err != nil {
		logger.Crit("error unmarshal %s: %s", ojs, err)
		return nil, err
	}
	return data, nil
}

type Bytes string

func NewBytes(value interface{}) Bytes {
	switch v := value.(type) {
	case int, int64, int32:
		return Bytes(fmt.Sprintf("%d", v))
	case float64, float32:
		return Bytes(fmt.Sprintf("%f", v))
	case string:
		return Bytes(v)
	}
	return Bytes(fmt.Sprintf("%v", value))
}

func (b Bytes) String() string {
	strSize := strings.ToUpper(string(b))
	if strings.ContainsAny(string(b), "KMGTPE") {
		for i, sfx := range []string{"K", "M", "G", "T", "P", "E"} {
			v := strings.Split(strSize, sfx)
			if len(v) > 1 {
				num, err := strconv.ParseFloat(v[0], 64)
				if err != nil {
					logger.Warn("error convert %s: %s. Use default value %f", v[0], err, num)
				}
				pow := math.Pow(float64(1024), float64(i+1))
				return fmt.Sprintf("%d", int64(num*pow))
			}
		}
	}
	num, err := strconv.ParseInt(strSize, 10, 64)
	if err != nil {
		logger.Crit("error convert bytes to string, %s", err)
	}
	return fmt.Sprint(num)
}

func (b Bytes) Less(value Bytes) (bool, error) {
	v, err := strconv.ParseInt(fmt.Sprint(value), 10, 64)
	if err != nil {
		return false, fmt.Errorf("error argument value convert %v, %s", value, err)
	}
	bv, err := strconv.ParseInt(fmt.Sprint(b), 10, 64)
	if err != nil {
		return false, fmt.Errorf("error base value convert %v, %s", b, err)
	}
	return bv < v, nil
}

func (b Bytes) Humanize() string {
	num, _ := strconv.ParseInt(b.String(), 10, 64)
	const unit = 1024
	if num < unit {
		return fmt.Sprintf("%dB", num)
	}
	div, exp := int64(unit), 0
	for n := num / unit; n >= unit; n /= unit {
		div *= unit
		exp++
	}
	return fmt.Sprintf("%.2f%cB", float64(num)/float64(div), "KMGTPE"[exp])
}
