package mdb

import (
	"encoding/json"
	"fmt"
	"strings"

	logger "a.yandex-team.ru/direct/infra/go-libs/pkg/logformat"
)

var (
	APICloudFmt = "https://gw.db.yandex-team.ru/managed-%s/v1/clusters" //clusterId
)

type Cluster struct {
	ClusterID *string
	FolderID  *string
	KeyPath   string
	Connection
}

func ConnectMysqlCluster(clusterID, folderID, keyPath string) (Cluster, error) {
	conn, err := NewConnection(keyPath)
	if err != nil {
		return Cluster{}, err
	}
	return Cluster{
		ClusterID:  &clusterID,
		FolderID:   &folderID,
		KeyPath:    keyPath,
		Connection: conn,
	}, nil
}

func (c Cluster) SetClusterID(id string) {
	*c.ClusterID = id
}

func (c Cluster) SetFolderID(id string) {
	*c.FolderID = id
}

func (c Cluster) GetClusterID() string {
	return *c.ClusterID
}

func (c *Cluster) URLCluster() string {
	return fmt.Sprintf(APICloudFmt, "mysql")
}

func (c *Cluster) URLOperations() string {
	return APICloudOperation
}

func (c Cluster) MyConnection() Connection {
	return c.Connection
}

func (c *Cluster) Close() {
	c.Connection.Client.CloseIdleConnections()
}

type MysqlHosts struct {
	Hosts []MysqlHost `json:"hosts"`
}

func (mh MysqlHosts) GetMaster() string {
	for _, i := range mh.Hosts {
		if strings.EqualFold(i.Role, "MASTER") {
			return i.Name
		}
	}
	return ""
}

func (mh MysqlHosts) GetHosts() []string {
	var hosts []string
	for _, i := range mh.Hosts {
		hosts = append(hosts, i.Name)
	}
	return hosts
}

type MysqlHost struct {
	Name           string `json:"name"`
	ClusterID      string `json:"clusterId"`
	ZoneID         string `json:"zoneId"`
	Role           string `json:"role"`
	Enviroment     string `json:"environment"`
	MysqlResources `json:"resources"`
}

type MysqlResources struct {
	ResourcePresetID string `json:"resourcePresetId,omitempty" yaml:"resourcePresetId"`
	DiskSize         *Bytes `json:"diskSize,omitempty" yaml:"diskSize"`
	DiskTypeID       string `json:"diskTypeId,omitempty" yaml:"diskTypeId"`
	Zones            string `json:"zones,omitempty" yaml:"zones"`
	Databases        string `json:"databases,omitempty" yaml:"databases"`
}

func NewMysqlResources(presentID, diskSize, diskType string) MysqlResources {
	ds := NewBytes(diskSize)
	return MysqlResources{
		ResourcePresetID: presentID,
		DiskSize:         &ds,
		DiskTypeID:       diskType,
	}
}

func (mr MysqlResources) DatabaseNames() []string {
	return strings.Split(mr.Databases, ",")
}

func (mr MysqlResources) Normalize() MysqlResources {
	if mr.DiskSize != nil {
		*mr.DiskSize = NewBytes(mr.DiskSize.String())
	}
	return mr
}

//Hosts
func (c *Cluster) ListHosts() (MysqlHosts, error) {
	var myhosts MysqlHosts
	myapi := fmt.Sprintf("%s/%s/hosts", c.URLCluster(), c.GetClusterID())
	out, err := c.Do("GET", myapi, nil, nil)
	if err != nil {
		return myhosts, err
	}
	if err := json.Unmarshal(out, &myhosts); err != nil {
		return myhosts, err
	}
	return myhosts, nil
}

//Databases
func (c *Cluster) AllDatabases() (MysqlDatabases, error) {
	dpages, err := c.ListDatabase("")
	return dpages.Databases, err
}

func (c *Cluster) ListDatabase(pageToken string) (MysqlDatabasePage, error) {
	var page MysqlDatabasePage
	var out []byte
	var err error
	var vals map[string]string
	myapi := fmt.Sprintf("%s/%s/databases", c.URLCluster(), c.GetClusterID())
	if len(pageToken) != 0 {
		vals = map[string]string{
			"pageToken": pageToken,
		}
	}

	out, err = c.Do("GET", myapi, vals, nil)
	if err != nil {
		return page, fmt.Errorf("error request %s, %s", myapi, err)
	}

	var i interface{}
	if err := json.Unmarshal(out, &i); err == nil {
		fmt.Printf("%+v\n", i)
	}

	if err := json.Unmarshal(out, &page); err != nil {
		return page, fmt.Errorf("error unmarshal %s: %s", out, err)
	}
	if len(page.NextPage) > 0 {
		next, err := c.ListDatabase(page.NextPage)
		if err == nil {
			page.Databases = append(page.Databases, next.Databases...)
		} else {
			return page, err
		}
	}
	return page, nil
}

type MysqlDatabase struct {
	Name      string `json:"name"`
	ClusterID string `json:"clusterID"`
}

type MysqlDatabases []MysqlDatabase

func (mds MysqlDatabases) Names() []string {
	var names []string
	for _, db := range mds {
		names = append(names, db.Name)
	}
	return names
}

type MysqlDatabasePage struct {
	Databases MysqlDatabases `json:"databases"`
	NextPage  string         `json:"nextPageToken"`
}

type MysqlConfig map[string]interface{}

type MysqlSettings struct {
	ID          string `json:"id"`
	FolderID    string `json:"folderId"`
	CreateAt    string `json:"createdAt"`
	Name        string `json:"name"`
	Labels      string `json:"labels"`
	Enviroment  string `json:"environment"`
	NetworkID   string `json:"networkId"`
	MysqlConfig `json:"config"`
}

func (c *Cluster) DatabaseSettings() (MysqlSettings, error) {
	var s MysqlSettings
	myapi := fmt.Sprintf("%s/%s", c.URLCluster(), c.GetClusterID())
	out, err := c.Do("GET", myapi, nil, nil)
	if err != nil {
		return s, fmt.Errorf("error request %s: %s", myapi, err)
	}
	if err := json.Unmarshal(out, &s); err != nil {
		return s, fmt.Errorf("error unmarshal %s: %s", out, err)
	}
	return s, nil
}

//Backups
type MysqlBackups struct {
	Backups []MysqlBackup `json:"backups"`
}

func (mb MysqlBackups) ToList() []string {
	var backups []string
	for _, b := range mb.Backups {
		backups = append(backups, b.ID)
	}
	return backups
}

func (mb MysqlBackups) FindBackupID(id string) (MysqlBackup, bool) {
	for _, backup := range mb.Backups {
		if backup.ID == id {
			return backup, true
		}
	}
	return MysqlBackup{}, false
}

type MysqlBackup struct {
	ID              string `json:"id"`
	FolderID        string `json:"folderId"`
	CreatedAt       string `json:"createdAt"`
	SourceClusterID string `json:"sourceClusterId"`
}

func (c *Cluster) ListBackups() (MysqlBackups, error) {
	var mybackups MysqlBackups
	if len(c.GetClusterID()) == 0 {
		return mybackups, fmt.Errorf("empty ClusterID")
	}
	myapi := fmt.Sprintf("%s/%s/backups", c.URLCluster(), c.GetClusterID())
	out, err := c.Do("GET", myapi, nil, nil)
	if err != nil {
		return mybackups, fmt.Errorf("error request %s: %s", myapi, err)
	}
	if err := json.Unmarshal(out, &mybackups); err != nil {
		return mybackups, err
	}
	return mybackups, nil
}

//Restore backup
type MysqlRestote struct {
	BackupID   string      `json:"backupId"`
	Time       string      `json:"time"`
	Name       string      `json:"name"`
	Enviroment string      `json:"environment"`
	FolderID   string      `json:"folderId"`
	NetworkID  string      `json:"networkId"`
	ConfigSpec MysqlConfig `json:"configSpec"`
	HostSpecs  `json:"hostSpecs"`
}

//type ConfigSpec struct {
//	MysqlResources `json:"resources"`
//}

func (c *Cluster) RestoreBackup(newClusterName string, backup MysqlBackup,
	zoneSpec []string, publicIP bool) (OperationID, error) {
	var opID OperationID
	db, err := c.DatabaseSettings()
	if err != nil {
		return opID, err
	}
	restoreRequest := MysqlRestote{
		BackupID:   backup.ID,
		Time:       backup.CreatedAt,
		Name:       newClusterName,
		ConfigSpec: db.MysqlConfig,
		Enviroment: db.Enviroment,
		NetworkID:  db.NetworkID,
		FolderID:   db.FolderID,
		HostSpecs:  NewMysqlHostSpecs(zoneSpec, publicIP),
	}
	body, err := json.Marshal(restoreRequest)
	if err != nil {
		return opID, fmt.Errorf("error marshal %+v: %s", restoreRequest, err)
	}
	myapi := fmt.Sprintf("%s:restore", c.URLCluster())
	out, err := c.Do("POST", myapi, nil, body)
	if err != nil {
		return opID, fmt.Errorf("error request %s: %+v, %s", myapi, restoreRequest, err)
	}
	if err := json.Unmarshal(out, &opID); err != nil {
		return opID, fmt.Errorf("error unmarshal %s: %s", out, err)
	}
	return opID, nil
}

//Clusters
func (c *Cluster) AllClusters() (ClusterConfigs, error) {
	upages, err := c.ListClusters("")
	return upages.Clusters, err
}

type ClusterConfig struct {
	ID            string           `json:"id,omitempty"`
	FolderID      string           `json:"folderId,omitempty"`
	Name          string           `json:"name,omitempty"`
	Enviroment    string           `json:"environment,omitempty"`
	ConfigSpec    MysqlConfig      `json:"configSpec,omitempty"`
	DatabaseSpecs MysqlDatabases   `json:"databaseSpecs,omitempty"`
	UserSpecs     UsersPermissions `json:"userSpecs,omitempty"`
	HostSpecs     `json:"hostSpecs,omitempty"`
}

/*type ClusterConfigSpec struct {
	Version string `json:"version"` //
	Resources MysqlResources `json:"resources"` //
}*/

type ClusterConfigs []ClusterConfig

func (cc ClusterConfigs) ClusterNameByID(id string) (ClusterConfig, bool) {
	for _, c := range cc {
		if c.ID == id {
			return c, true
		}
	}
	return ClusterConfig{}, false
}

type MysqlClusterPages struct {
	Clusters ClusterConfigs `json:"clusters"`
	NextPage string         `json:"nextPageToken"`
}

func (c *Cluster) CreateCluster(newClusterName string, zoneSpec, dbnames []string, publicIP bool,
	env string, usersSpec UsersPermissions, configSpec MysqlConfig) (OperationID, error) {
	var opID OperationID
	var hostSpecs HostSpecs
	for _, zone := range zoneSpec {
		var i HostSpec
		i.ZoneID = zone
		i.AssignPublicIP = publicIP
		hostSpecs = append(hostSpecs, i)
	}

	var dbSpecs MysqlDatabases
	for _, db := range dbnames {
		var i MysqlDatabase
		i.Name = db
		dbSpecs = append(dbSpecs, i)
	}

	createReq := ClusterConfig{
		FolderID:      *c.FolderID,
		Name:          newClusterName,
		Enviroment:    strings.ToUpper(env),
		ConfigSpec:    configSpec,
		DatabaseSpecs: dbSpecs,
		UserSpecs:     usersSpec,
		HostSpecs:     hostSpecs,
	}

	body, err := json.Marshal(createReq)
	if err != nil {
		return opID, fmt.Errorf("error marshal %+v: %s", createReq, err)
	}
	myapi := c.URLCluster()
	out, err := c.Do("POST", myapi, nil, body)
	if err != nil {
		return opID, fmt.Errorf("error request %s: %+v, %s", myapi, createReq, err)
	}
	if err := json.Unmarshal(out, &opID); err != nil {
		return opID, fmt.Errorf("error unmarshal %s: %s", out, err)
	}
	return opID, nil
}

func (c *Cluster) UpdateClusterSettings(clusterName string, configSpec MysqlConfig) (OperationID, error) {
	var opID OperationID
	var cc ClusterConfig
	cc.ConfigSpec = configSpec
	cc.Name = clusterName
	body, err := json.Marshal(cc)
	if err != nil {
		return opID, fmt.Errorf("error marshal %+v: %s", configSpec, err)
	}
	myapi := fmt.Sprintf("%s/%s", c.URLCluster(), c.GetClusterID())
	out, err := c.Do("PATCH", myapi, nil, body)
	if err != nil {
		return opID, fmt.Errorf("error request %s: %+v, %s", myapi, configSpec, err)
	}
	if err := json.Unmarshal(out, &opID); err != nil {
		return opID, fmt.Errorf("error unmarshal %s: %s", out, err)
	}
	return opID, nil
}

type DatabaseConfig struct {
	DatabaseSpec MysqlDatabase `json:"databaseSpec,omitempty"`
}

func (c *Cluster) CreateDatabase(dbname string) (OperationID, error) {
	var opID OperationID
	createReq := DatabaseConfig{
		DatabaseSpec: MysqlDatabase{
			Name: dbname,
		},
	}
	body, err := json.Marshal(createReq)
	if err != nil {
		return opID, fmt.Errorf("error marshal %+v: %s", dbname, err)
	}
	logger.Debug("%s", body)
	myapi := fmt.Sprintf("%s/%s/databases", c.URLCluster(), c.GetClusterID())
	out, err := c.Do("POST", myapi, nil, body)
	if err != nil {
		return opID, fmt.Errorf("error request %s: %+v, %s", myapi, dbname, err)
	}
	if err := json.Unmarshal(out, &opID); err != nil {
		return opID, fmt.Errorf("error unmarshal %s: %s", out, err)
	}
	return opID, nil
}

func (c *Cluster) RemoveDatabase(db string) (OperationID, error) {
	var opID OperationID
	myapi := fmt.Sprintf("%s/%s/databases/%s", c.URLCluster(), c.GetClusterID(), db)
	out, err := c.Do("DELETE", myapi, nil, nil)
	if err != nil {
		return opID, fmt.Errorf("error request %s: %+v, %s", myapi, db, err)
	}
	if err := json.Unmarshal(out, &opID); err != nil {
		return opID, fmt.Errorf("error unmarshal %s: %s", out, err)
	}
	return opID, nil
}

func (c *Cluster) ListClusters(pageToken string) (MysqlClusterPages, error) {
	var curPage MysqlClusterPages
	var out []byte
	var err error
	vals := map[string]string{
		"folderId": *c.FolderID,
	}
	myapi := c.URLCluster()
	if len(pageToken) != 0 {
		vals["pageToken"] = pageToken
	}
	out, err = c.Do("GET", myapi, vals, nil)
	if err != nil {
		return curPage, fmt.Errorf("error request %s, %s", myapi, err)
	}

	if err := json.Unmarshal(out, &curPage); err != nil {
		return curPage, fmt.Errorf("error unmarshal %s: %s", out, err)
	}

	if len(curPage.NextPage) > 0 {
		next, err := c.ListClusters(curPage.NextPage)
		if err == nil {
			curPage.Clusters = append(curPage.Clusters, next.Clusters...)
		} else {
			return curPage, err
		}
	}
	return curPage, nil
}

func (c *Cluster) AllDatabaseUsers() (UsersPermissions, error) {
	upages, err := c.ListUsers("")
	return upages.Users, err
}

func (c *Cluster) ListUsers(pageToken string) (MysqlUsersPages, error) {
	var curPage MysqlUsersPages
	var out []byte
	var err error
	var vals map[string]string
	myapi := fmt.Sprintf("%s/%s/users", c.URLCluster(), c.GetClusterID())
	if len(pageToken) != 0 {
		vals = map[string]string{
			"pageToken": pageToken,
		}
	}
	out, err = c.Do("GET", myapi, vals, nil)
	if err != nil {
		return curPage, fmt.Errorf("error request %s, %s", myapi, err)
	}
	var i interface{}
	if err := json.Unmarshal(out, &i); err == nil {
		fmt.Printf("%+v\n", i)
	}

	if err := json.Unmarshal(out, &curPage); err != nil {
		return curPage, fmt.Errorf("error unmarshal %s: %s", out, err)
	}
	if len(curPage.NextPage) > 0 {
		next, err := c.ListUsers(curPage.NextPage)
		if err == nil {
			curPage.Users = append(curPage.Users, next.Users...)
		} else {
			return curPage, err
		}
	}
	return curPage, nil
}

type MysqlUsersPages struct {
	Users    UsersPermissions `json:"users"`
	NextPage string           `json:"nextPageToken"`
}

func (c *Cluster) CreateMysqlUser(user UserPermission) (OperationID, error) {
	var opID OperationID
	userSpec := MysqlUserSpec{
		UserSpec: user,
	}
	userSpec.UserSpec.ClusterID = *c.ClusterID
	body, err := json.Marshal(userSpec)
	if err != nil {
		return opID, fmt.Errorf("error marshal %+v: %s", user, err)
	}
	logger.Debug("%s", body)
	myapi := fmt.Sprintf("%s/%s/users", c.URLCluster(), c.GetClusterID())
	out, err := c.Do("POST", myapi, nil, body)
	if err != nil {
		return opID, fmt.Errorf("error request %s: %+v, %s", myapi, user, err)
	}
	if err := json.Unmarshal(out, &opID); err != nil {
		return opID, fmt.Errorf("error unmarshal %s: %s", out, err)
	}
	return opID, nil
}

func (c *Cluster) RemoveMysqlUser(user UserPermission) (OperationID, error) {
	var opID OperationID
	myapi := fmt.Sprintf("%s/%s/users/%s", c.URLCluster(), c.GetClusterID(), user.Username)
	out, err := c.Do("DELETE", myapi, nil, nil)
	if err != nil {
		return opID, fmt.Errorf("error request %s: %+v, %s", myapi, user, err)
	}
	if err := json.Unmarshal(out, &opID); err != nil {
		return opID, fmt.Errorf("error unmarshal %s: %s", out, err)
	}
	return opID, nil
}

type MysqlUserSpec struct {
	UserSpec UserPermission `json:"userSpec"`
}

func (c *Cluster) UpdateMysqlUser(user UserPermission) (OperationID, error) {
	var opID OperationID
	body, err := json.Marshal(user)
	if err != nil {
		return opID, fmt.Errorf("error marshal %+v: %s", user, err)
	}
	myapi := fmt.Sprintf("%s/%s/users/%s", c.URLCluster(), c.GetClusterID(), user.Username)
	out, err := c.Do("PATCH", myapi, nil, body)
	if err != nil {
		return opID, fmt.Errorf("error request %s: %+v, %s", myapi, user, err)
	}
	if err := json.Unmarshal(out, &opID); err != nil {
		return opID, fmt.Errorf("error unmarshal %s: %s", out, err)
	}
	return opID, nil
}

type UserPermission struct {
	Username            string            `json:"name"`
	ClusterID           string            `json:"clusterID"`
	DatabasePermissions DatabasesRoles    `json:"permissions"`
	GlobalPermissions   Roles             `json:"globalPermissions"`
	Settings            map[string]string `json:"settings"`
	Password            string            `json:"password"`
}

type UsersPermissions []UserPermission

func (ups UsersPermissions) FindUser(value interface{}) (UserPermission, bool) {
	var username string
	switch v := value.(type) {
	case string:
		username = v
	case UserPermission:
		username = v.Username
	default:
		username = fmt.Sprintf("%s", v)
	}
	for _, user := range ups {
		logger.Debug("%s %s", user, username)
		if user.Username == username {
			return user, true
		}
	}
	return UserPermission{}, false
}

func (ups UsersPermissions) UsersNames() []string {
	var result []string
	for _, user := range ups {
		result = append(result, user.Username)
	}
	return result
}

type Roles []string

func (rs Roles) HasRole(value interface{}) bool {
	for _, role := range rs {
		if strings.EqualFold(strings.ReplaceAll(role, " ", "_"),
			strings.ReplaceAll(fmt.Sprint(value), " ", "_")) {
			return true
		}
	}
	return false
}

type DatabasesRoles []DatabaseRole

type DatabaseRole struct {
	Database string `json:"databaseName"`
	Roles    Roles  `json:"roles"`
}

func (dr DatabaseRole) HasRole(value interface{}) bool {
	var userArg UserPermission
	switch v := value.(type) {
	case UserPermission:
		userArg = v
	case DatabaseRole:
		userArg = UserPermission{
			DatabasePermissions: DatabasesRoles{v},
		}
	default:
		logger.Warn("unsupported type %T for value %v", v, v)
	}
	for _, perm := range userArg.DatabasePermissions {
		if len(perm.Database) == 0 && dr.Database == perm.Database {
			for _, role := range perm.Roles {
				if !dr.Roles.HasRole(role) {
					return false
				}
			}
			return true
		}
	}
	return false
}

type MysqlClusters struct {
	Clusters []MysqlSettings `json:"clusters"`
}

func (c *Cluster) FindClusterIDByName(clusterName string) (string, error) {
	var clusters MysqlClusters
	vals := map[string]string{
		"filter":   fmt.Sprintf("name='%s'", clusterName),
		"folderId": *c.FolderID,
	}
	myapi := c.URLCluster()
	out, err := c.Do("GET", myapi, vals, nil)
	if err != nil {
		return "", fmt.Errorf("error request %s: %s", myapi, err)
	}
	logger.Debug("FindClusterByName json: %s", out)

	if err := json.Unmarshal(out, &clusters); err != nil {
		return "", fmt.Errorf("error unmarshal %s: %s", out, err)
	}
	logger.Debug("FindClusterByName struct: %+v", clusters)

	for _, cluster := range clusters.Clusters {
		return cluster.ID, nil
	}
	return "", fmt.Errorf("not found cluster id for %s", clusterName)
}

//ZoneSpecifications. Example sas, vla, man.
type HostSpec struct {
	ZoneID         string `json:"zoneId"`
	AssignPublicIP bool   `json:"assignPublicIp"`
}

type HostSpecs []HostSpec

func NewMysqlHostSpecs(zones []string, publicIP bool) HostSpecs {
	var specs HostSpecs
	for _, zone := range zones {
		hs := HostSpec{
			ZoneID:         zone,
			AssignPublicIP: publicIP,
		}
		specs = append(specs, hs)
	}
	return specs
}

func IsGlobalPermition(value interface{}) bool {
	AvailibleGlobalPermissions := []string{
		"REPLICATION_SLAVE",
		"REPLICATION_CLIENT",
		"PROCESS",
	}
	for _, awail := range AvailibleGlobalPermissions {
		if fmt.Sprint(value) == awail {
			return true
		}
	}
	return false
}

func CreateCluster() {}
