package mdb

import (
	"context"
	"encoding/json"
	"fmt"
	"strings"
	"time"

	logger "a.yandex-team.ru/direct/infra/go-libs/pkg/logformat"
)

var (
	APICloudOperation = "https://gw.db.yandex-team.ru/operations" //operationId
	timeInterval      = 120 * time.Second
)

type OperationID struct {
	ID          string         `json:"id"`
	Description string         `json:"description"`
	CreatedAt   string         `json:"createdAt"`
	CreatedBy   string         `json:"createdBy"`
	ModifiedAt  string         `json:"modifiedAt"`
	Done        bool           `json:"done"`
	Error       OperationError `json:"error"`
}

type OperationError struct {
	Code    int      `json:"code"`
	Message string   `json:"message"`
	Details []string `json:"details"`
}

type ClientInterface interface {
	MyConnection() Connection
}

func OperationStatus(id string, ci ClientInterface) (OperationID, error) {
	var opID OperationID
	myapi := fmt.Sprintf("%[1]s/%[2]s", APICloudOperation, id)
	client := ci.MyConnection()
	resp, err := client.Do("GET", myapi, nil, nil)
	if err != nil {
		return opID, fmt.Errorf("error %s: data %s, err %s", myapi, resp, err)
	}
	if err := json.Unmarshal(resp, &opID); err != nil {
		return opID, fmt.Errorf("error unmarshal %s: %s", resp, err)
	}
	logger.Debug("method %s ID %s\n", myapi, resp)

	return opID, nil
}

func WaitingOperationDone(cntx context.Context, id string, ci ClientInterface, params ...interface{}) bool {
	if len(params) > 0 {
		if v, ok := params[0].(time.Duration); ok {
			timeInterval = v
		}
	}
	ticker := time.NewTicker(timeInterval)
	defer ticker.Stop()
L:
	for range ticker.C {
		select {
		case <-cntx.Done():
			logger.Crit("error operation waiting: context timeout\n")
			break L
		default:
			status, err := OperationStatus(id, ci)
			if err != nil {
				logger.Crit("error request OperationStatus: %s", err)
				continue
			}
			if status.Done {
				if status.Error.Code == 0 {
					logger.Crit("task completed successfully(taskid: %s, done: %t, code: %d)\n",
						status.ID, status.Done, status.Error.Code)
					return true
				} else {
					logger.Crit("task completed with errors(taskid: %s, done: %t, code: %d)\n",
						status.ID, status.Done, status.Error.Code)
					logger.Crit(status.Error.Message, status.Error.Details)
					return false
				}
			} else {
				logger.Info("the task is still in progress(taskid: %s, done: %t)\n", status.ID, status.Done)
			}
		}
	}
	return false
}

func (op OperationID) Status() string {
	if op.Done {
		if op.Error.Code == 0 {
			return "SUCCESS"
		} else {
			return "FAILED"
		}
	}
	return "PROCESS"
}

func (op OperationID) Message() string {
	msg := op.Error.Message
	if len(op.Error.Details) > 0 {
		details := strings.Join(op.Error.Details, ",")
		msg = fmt.Sprintf("%s/%s", msg, details)
	}
	return msg
}

func (op OperationID) HashID() string {
	return op.ID
}

type State interface {
	Status() string
	Message() string
	HashID() string
}

type Monitoring struct {
	Status  string `json:"status"`
	Message string `json:"message"`
	HashID  string `json:"hashid"`
}

func JSONStatus(s State) ([]byte, error) {
	out := Monitoring{
		Status:  s.Status(),
		Message: s.Message(),
		HashID:  s.HashID(),
	}
	return json.Marshal(out)
}
