package mdb

import (
	"crypto/rsa"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"path/filepath"
	"strings"
	"time"

	"github.com/golang-jwt/jwt/v4"
)

var (
	debug bool

	ps256WithSaltLengthEqualsHash = &jwt.SigningMethodRSAPSS{
		SigningMethodRSA: jwt.SigningMethodPS256.SigningMethodRSA,
		Options: &rsa.PSSOptions{
			SaltLength: rsa.PSSSaltLengthEqualsHash,
		},
	}
)

const (
	AIMURL = "https://gw.db.yandex-team.ru:443/iam/v1/tokens"
)

func SetDebug(value bool) {
	debug = value
}

type CloudKey struct {
	ID               string          `json:"id"`
	AccountID        string          `json:"service_account_id"`
	Created          string          `json:"created_at"`
	Algorithm        string          `json:"key_algorithm"`
	PublicKeyString  string          `json:"public_key"`
	PrivateKeyString string          `json:"private_key"`
	RSAPublicKey     *rsa.PublicKey  `json:"-"`
	RSAPrivateKey    *rsa.PrivateKey `json:"-"`
}

func loadCloudKey(key string) (ck CloudKey, err error) {
	path, err := filepath.Abs(key)
	if err != nil {
		return
	}
	data, err := ioutil.ReadFile(path)
	if err != nil {
		return ck, fmt.Errorf("error open %s: %s", path, err)
	}
	err = json.Unmarshal(data, &ck)
	if err != nil {
		err = fmt.Errorf("error unmarshal %s: %s", data, err)
		return
	}

	ck.RSAPrivateKey, err = jwt.ParseRSAPrivateKeyFromPEM([]byte(ck.PrivateKeyString))
	if err != nil {
		return
	}
	ck.RSAPublicKey, err = jwt.ParseRSAPublicKeyFromPEM([]byte(ck.PublicKeyString))

	if debug {
		fmt.Printf("[DEBUG] cloud key: %+v\n", ck)
	}
	return
}

type CloudToken struct {
	CloudKey
	Token    *jwt.Token
	IAMToken *string
	KeyPath  string
}

func GenerateAIMRequest(keypath string) (ct *CloudToken, err error) { //signed string, err error) {
	key, err := loadCloudKey(keypath)
	if debug {
		fmt.Printf("[DEBUG] loadCloudKey %s: %+v\n", keypath, key)
	}
	if err != nil {
		return
	}

	issuedAt := time.Now()
	token := jwt.NewWithClaims(ps256WithSaltLengthEqualsHash, jwt.StandardClaims{
		Issuer:    key.AccountID,
		IssuedAt:  issuedAt.Unix(),
		ExpiresAt: issuedAt.Add(time.Hour).Unix(),
		Audience:  "https://iam.api.cloud.yandex.net/iam/v1/tokens",
	})
	token.Header["kid"] = key.ID
	iamtoken := ""

	return &CloudToken{key, token, &iamtoken, keypath}, nil
}

//получение iam токена
func (ct *CloudToken) GetAIMToken(client *http.Client) (aim string, err error) {
	//если токен iam протух, то выводим пустую строку и ошибку
	if err := ct.Token.Claims.Valid(); err != nil {
		ct, err = GenerateAIMRequest(ct.KeyPath)
		if err != nil {
			return aim, fmt.Errorf("error generate new iamtoken (old iamtoken expired): %s", err)
		}
	}
	//если сохраненый токен не пустой, то возвращаем его
	if len(*ct.IAMToken) != 0 {
		if debug {
			fmt.Printf("use current aimtoken: %s", *ct.IAMToken)
		}
		return *ct.IAMToken, nil
	}

	//получаем новый токен, если сохраненный пустой
	jot, err := ct.Token.SignedString(ct.RSAPrivateKey)
	if err != nil {
		return
	}
	if debug {
		fmt.Printf("request new iam token: %s\n", jot)
	}

	resp, err := client.Post(
		AIMURL,
		"application/json",
		strings.NewReader(fmt.Sprintf(`{"jwt":"%s"}`, jot)),
	)
	if err != nil {
		return
	}
	defer func() { _ = resp.Body.Close() }()
	if resp.StatusCode != http.StatusOK {
		body, _ := ioutil.ReadAll(resp.Body)
		err = fmt.Errorf("%s: %s", resp.Status, body)
		return
	}
	var data struct {
		IAMToken string `json:"iamToken"`
	}
	err = json.NewDecoder(resp.Body).Decode(&data)
	if err != nil {
		return
	}
	*ct.IAMToken = data.IAMToken
	if debug {
		fmt.Printf("saved iam token: %s\n", *ct.IAMToken)
	}
	return *ct.IAMToken, nil
}
