package mdbgrpc

import (
	"crypto/rsa"
	"encoding/json"
	"fmt"
	"github.com/golang-jwt/jwt/v4"
	"io/ioutil"
	"path/filepath"
	"time"

	logger "a.yandex-team.ru/direct/infra/go-libs/pkg/logformat"
)

var jwtSigningMethodPS256WithSaltLengthEqualsHash = &jwt.SigningMethodRSAPSS{
	SigningMethodRSA: jwt.SigningMethodPS256.SigningMethodRSA,
	Options: &rsa.PSSOptions{
		SaltLength: rsa.PSSSaltLengthEqualsHash,
	},
}

type CloudKey struct {
	KeyID                  string `json:"id,omitempty"`
	ServiceAccountID       string `json:"service_account_id,omitempty"`
	PrivateKey             string `json:"private_key,omitempty"`
	PublicKey              string `json:"public_key,omitempty"`
	PrivateKeyPasswordFile string `json:"private_key_password_file,omitempty"`
}

//Вывести приватный ключ в формате rsa.PrivateKey
func (ck CloudKey) GetPrivateKey() (*rsa.PrivateKey, error) {
	return jwt.ParseRSAPrivateKeyFromPEM([]byte(ck.PrivateKey))
}

//Вывести публичный ключ в формате rsa.PublicKey
func (ck CloudKey) GetPublickKey() (*rsa.PublicKey, error) {
	return jwt.ParseRSAPublicKeyFromPEM([]byte(ck.PublicKey))
}

//Вывести идентификатор аккаунта
func (ck CloudKey) GetServiceAccountID() string {
	return ck.ServiceAccountID
}

func (ck *CloudKey) SignedJwtToken() (string, error) {
	pk, err := ck.GetPrivateKey()
	if err != nil {
		return "", err
	}
	return ck.IssueJwtToken().SignedString(pk)
}

func (ck *CloudKey) IssueJwtToken() *jwt.Token {
	issuedAt := time.Now()

	token := jwt.NewWithClaims(jwtSigningMethodPS256WithSaltLengthEqualsHash, jwt.StandardClaims{
		Issuer:    ck.GetServiceAccountID(),
		IssuedAt:  issuedAt.Unix(),
		ExpiresAt: issuedAt.Add(time.Hour).Unix(),
		Audience:  "https://iam.api.cloud.yandex.net/iam/v1/tokens",
	})
	token.Header["kid"] = ck.KeyID
	logger.Debug("JWT token %+v", token) //DEBUG
	return token
}

//Загружает ключ для работы с API MDB
func LoadCloudKey(path string) (CloudKey, error) {
	var cloudKey CloudKey
	mypath, err := filepath.Abs(path)
	if err != nil {
		return cloudKey, err
	}
	data, err := ioutil.ReadFile(mypath)
	if err != nil {
		return cloudKey, fmt.Errorf("error open %s: %s", path, err)
	}
	err = json.Unmarshal(data, &cloudKey)
	return cloudKey, err
}

type IamToken struct {
	CloudKey
	//CreateIamTokenRequest
	//token
}
