package mdbgrpc

import (
	"context"
	"encoding/json"
	"fmt"
	"google.golang.org/grpc"
	"strings"
	"time"

	msql "a.yandex-team.ru/cloud/bitbucket/public-api/yandex/cloud/mdb/mysql/v1"
	mconf "a.yandex-team.ru/cloud/bitbucket/public-api/yandex/cloud/mdb/mysql/v1/config"
	mop "a.yandex-team.ru/cloud/bitbucket/public-api/yandex/cloud/operation"
	logger "a.yandex-team.ru/direct/infra/go-libs/pkg/logformat"
)

const (
	DefaultFolderID = "fooa07bcrr7souccreru"
)

type CloudMysqlClusters []*msql.Cluster

func (cmc CloudMysqlClusters) FindByClusterName(name string) (*msql.Cluster, error) {
	for _, cluster := range cmc {
		if strings.EqualFold(cluster.Name, name) {
			return cluster, nil
		}
	}
	return nil, fmt.Errorf("not found cluster with name %s", name)
}

func GetMysqlConfig(cm *msql.Cluster) (interface{}, error) {
	config := cm.Config
	if config == nil {
		return nil, fmt.Errorf("empty config")
	}

	switch config.GetVersion() {
	case "5.7":
		cnf1 := config.GetMysqlConfig_5_7()
		defaultConfig := cnf1.GetDefaultConfig()
		effectiveConfig := cnf1.GetEffectiveConfig()
		userConfig := cnf1.GetUserConfig()
		var temp map[string]interface{}
		if tmp, err := json.Marshal(defaultConfig); err == nil {
			if err = json.Unmarshal(tmp, &temp); err != nil {
				return nil, fmt.Errorf("erro unmarshal defaultConfig: %s", err)
			}
		} else {
			return nil, fmt.Errorf("error marshal defaultConfig: %s", err)
		}
		if tmp, err := json.Marshal(effectiveConfig); err == nil {
			if err = json.Unmarshal(tmp, &temp); err != nil {
				return nil, fmt.Errorf("erro unmarshal effectiveConfig: %s", err)
			}
		} else {
			return nil, fmt.Errorf("error marshal effectiveConfig: %s", err)
		}
		if tmp, err := json.Marshal(userConfig); err == nil {
			if err = json.Unmarshal(tmp, &temp); err != nil {
				return nil, fmt.Errorf("erro unmarshal userConfig: %s", err)
			}
		} else {
			return nil, fmt.Errorf("error marshal userConfig: %s", err)
		}
		var result *mconf.MysqlConfig5_7
		if tmp, err := json.Marshal(temp); err == nil {
			if err = json.Unmarshal(tmp, &result); err != nil {
				return nil, fmt.Errorf("erro unmarshal userConfig: %s", err)
			}
		} else {
			return nil, fmt.Errorf("error marshal userConfig: %s", err)
		}

		return result, nil

	case "8.0":
		cnf1 := config.GetMysqlConfig_8_0()
		defaultConfig := cnf1.GetDefaultConfig()
		effectiveConfig := cnf1.GetEffectiveConfig()
		userConfig := cnf1.GetUserConfig()
		var temp map[string]interface{}
		if tmp, err := json.Marshal(defaultConfig); err == nil {
			if err = json.Unmarshal(tmp, &temp); err != nil {
				return nil, fmt.Errorf("erro unmarshal defaultConfig: %s", err)
			}
		} else {
			return nil, fmt.Errorf("error marshal defaultConfig: %s", err)
		}
		if tmp, err := json.Marshal(effectiveConfig); err == nil {
			if err = json.Unmarshal(tmp, &temp); err != nil {
				return nil, fmt.Errorf("erro unmarshal effectiveConfig: %s", err)
			}
		} else {
			return nil, fmt.Errorf("error marshal effectiveConfig: %s", err)
		}
		if tmp, err := json.Marshal(userConfig); err == nil {
			if err = json.Unmarshal(tmp, &temp); err != nil {
				return nil, fmt.Errorf("erro unmarshal userConfig: %s", err)
			}
		} else {
			return nil, fmt.Errorf("error marshal userConfig: %s", err)
		}
		var result *mconf.MysqlConfig8_0
		if tmp, err := json.Marshal(temp); err == nil {
			if err = json.Unmarshal(tmp, &result); err != nil {
				return nil, fmt.Errorf("erro unmarshal userConfig: %s", err)
			}
		} else {
			return nil, fmt.Errorf("error marshal userConfig: %s", err)
		}

		return result, nil
	}
	return nil, fmt.Errorf("not found version %s config", config.GetVersion())
}

func ListClusters(conn *grpc.ClientConn, folderID string) (clusters CloudMysqlClusters, err error) {
	client := msql.NewClusterServiceClient(conn)
	if len(folderID) == 0 {
		folderID = DefaultFolderID
	}
	req := msql.ListClustersRequest{
		FolderId: folderID,
		PageSize: 10,
	}
	cntx, cancel := context.WithTimeout(context.Background(), 30*time.Second)
	defer cancel()
	for {
		logger.Debug("current page token(list) %s", req.GetPageToken())

		resp, err := client.List(cntx, &req)
		if err != nil {
			logger.Crit("error client.List: %s", err)
			return clusters, err
		}
		cl := resp.GetClusters()
		if len(cl) > 0 {
			clusters = append(clusters, cl...)
		}
		if nextToken := resp.GetNextPageToken(); len(nextToken) == 0 {
			break
		} else {
			req.PageToken = nextToken
		}
	}
	return
	/*b := clusters[0]
	mysqlConf := b.GetConfig().GetMysqlConfig()
	m := mysqlConf.(*msql.ClusterConfig_MysqlConfig_5_7)
	jsoner, err := json.Marshal(m.MysqlConfig_5_7.DefaultConfig)
	if err != nil {
		fmt.Println(err)
	}
	fmt.Printf("resp: %s type %t, error: %s\n", jsoner, mysqlConf, err)
	var g config.MysqlConfig5_7

	//h := []byte("{\"innodb_buffer_pool_size\":{\"value\":6442450944}}")
	if err := json.Unmarshal(jsoner, &g); err != nil {
		fmt.Printf("error unmarshal %s", err)
	}
	fmt.Printf("%+v %d\n", g, g.InnodbBufferPoolSize.Value)
	*/
}

func UpdateMysql57Config(conn *grpc.ClientConn, clusterID, clusterName string, config *mconf.MysqlConfig5_7) (resp *mop.Operation, err error) {
	client := msql.NewClusterServiceClient(conn)
	if len(clusterID) == 0 {
		return nil, fmt.Errorf("empty cluster name")
	}

	req := msql.UpdateClusterRequest{
		ClusterId: clusterID,
		Name:      clusterName,
		ConfigSpec: &msql.ConfigSpec{
			MysqlConfig: &msql.ConfigSpec_MysqlConfig_5_7{MysqlConfig_5_7: config},
		},
	}

	cntx, cancel := context.WithTimeout(context.Background(), 30*time.Second)
	defer cancel()
	for range [3]int{} {
		resp, err = client.Update(cntx, &req)
		if err != nil {
			logger.Crit("error UpdateMysql57Config: %s", err)
			time.Sleep(5 * time.Second)
			continue
		}
		return resp, err
	}
	return nil, err
}

func ListHosts(conn *grpc.ClientConn, clusterID string) (hosts *msql.ListClusterHostsResponse, err error) {
	client := msql.NewClusterServiceClient(conn)
	if len(clusterID) == 0 {
		return nil, fmt.Errorf("empty cluster id")
	}
	req := msql.ListClusterHostsRequest{
		ClusterId: clusterID,
	}
	cntx, cancel := context.WithTimeout(context.Background(), 30*time.Second)
	defer cancel()
	for range [3]int{} {
		hosts, err = client.ListHosts(cntx, &req)
		if err != nil {
			logger.Crit("error SwitchMaster: %s", err)
			time.Sleep(5 * time.Second)
			continue
		}
		return hosts, err
	}
	return nil, err
}
